import std.string;
import std.algorithm;
import std.random;
import std.conv;
import main;
import std.math;

struct QUESTION {
	string message;
	string rightAnswer;
	private bool delegate(const(char)[]) answerChecker;
}

//Persistent state across questions
int questionCount;
string currentNoAnswer;
int numNoAnswersLeft;
int questionStartTime;
int questionDeadline;

//Question stack for push and pop
QUESTION questionStack[];

//Current question
QUESTION question;

void resetQuestions() {
	questionCount = 0;
	numNoAnswersLeft = 0;
}

enum newQuestionTypeEvery = 3;

float probabilityOverTime(int start, int end, float max) {
	float p = (questionCount - start) * max / (end - start);
	return p < 0 ? 0 : p > max ? max : p;
}

@property float computerSaysNoProb() {
	return probabilityOverTime(10, 50, 0.2f);
}
@property float pushPopProb() {
	return probabilityOverTime(30, 70, 0.2f);
}
@property int maxStack() {
	return (questionCount - 20) / 30;
}

int chooseArithmeticRadix(int phase) {	//greater phase = strange radixes come later (e.g. for "*")
	int max = (questionCount - 30 - 10*phase) / 15;
	if (max < 1) max = 1;
	int radix = [10, 8, 16, 2, 0][uniform(0, max>$?$:max)];
	if (radix == 0) radix = uniform!"[]"(2, 16);
	return radix;
}
int chooseBitwiseRadix() {
	int max = (questionCount - 30) / 10;
	if (max < 1) max = 1;
	return [2, 4, 8, 16, 10][uniform(0, max>$?$:max)];
}

int arithmeticMax(int radix) {
	if (radix == 10) {
		enum initial = 10;
		enum doubleEvery = 30;
		float mul = pow(2, cast(float)questionCount/doubleEvery);
		if (mul > 1000000) mul = 1000000;
		return cast(int)(initial * mul);
	} else {
		enum initial = 10;
		enum doubleEvery = 30;
		enum afterFirst = 40;
		if (questionCount < afterFirst) return initial;
		float mul = pow(2, cast(float)(questionCount - afterFirst) / doubleEvery);
		if (mul > 1000000) mul = 1000000;
		return cast(int)(initial * mul);
	}
}

int multiplyMax(int radix) {
	if (radix == 10) {
		enum initial = 10;
		enum doubleAfter = 50;
		return cast(int)(initial * (1 + cast(float)questionCount/doubleAfter));
	} else {
		enum initial = 10;
		enum doubleAfter = 50;
		enum afterFirst = 40;
		if (questionCount < afterFirst) return initial;
		return cast(int)(initial * (1 + cast(float)(questionCount - afterFirst) / doubleAfter));
	}
}

@property int baseConvertMax() {
	enum initial = 20;
	enum doubleAfter = 20;
	return cast(int)(initial * (1 + cast(float)questionCount/doubleAfter));
}

@property int bitwiseBits() {
	int bits = 4 + (questionCount - 40) / 20;
	return bits < 4 ? 4 : bits > 16 ? 16 : bits;
}

void generateQuestion() {
	questionCount++;
	//Deadline starts nice and long, but goes down the more questions you answer.
	//A hyperbola seems to be quite a nice way to squeeze it over time.
	questionStartTime = ticks;
	questionDeadline = ticks + 100*60*10 / questionCount;

	if (numNoAnswersLeft > 0) numNoAnswersLeft--;
	if (numNoAnswersLeft > 0) {
		questionGenerators[uniform(0, questionRange())]();	//Don't allow another 'Computer says' at the same time - it's not clear what the player should do
		if (uniform01() < 0.5f)
			question.message = "sudo " ~ question.message;
		else
			question.rightAnswer = currentNoAnswer;
	} else if (questionStack.length > 0 && uniform01() < pushPopProb) {
		question = questionStack[$-1];
		questionStack.length--;
		question.rightAnswer = question.message ~ ": " ~ question.rightAnswer;
		question.message = "POP";
	} else if (uniform01() < computerSaysNoProb) {
		/*
		question = "Computer says no";
		rightAnswer = "no";
		answerChecker = a => a.icmp("no") == 0;
		*/
		char n = ['n', 'N'][uniform(0, $)];
		char o = ['o', 'O', '0'][uniform(0, $)];
		question.rightAnswer = currentNoAnswer = [n, o];
		numNoAnswersLeft = uniform!"[]"(1, 5);
		question.message = "Computer says " ~ currentNoAnswer;
		if (numNoAnswersLeft > 1) question.message ~= " for " ~ to!string(numNoAnswersLeft) ~ " instructions";
	} else {
		questionGenerators[uniform(0, questionRange())]();
		string prefix;
		while (numNoAnswersLeft == 0 && questionStack.length < maxStack && uniform01() < pushPopProb) {
			prefix ~= "PUSH " ~ question.message ~ "; ";
			questionStack ~= question;
			questionGenerators[uniform(0, questionRange())]();
		}
		question.message = prefix ~ question.message;
	}
}

bool answerIsRight(const(char)[] answer) {
	if (numNoAnswersLeft > 0 && !question.message.startsWith("sudo "))
		return answer == currentNoAnswer;
	else
		return question.answerChecker(answer);
}

int questionRange() {
	int r = 1 + questionCount / newQuestionTypeEvery;
	if (r > questionGenerators.length) r = questionGenerators.length;
	return r;
}

auto questionGenerators = [
	{
		//Arithmetic
		int radix = chooseArithmeticRadix(0);
		int x = uniform!"[]"(0, arithmeticMax(radix));
		int y = uniform!"[]"(0, arithmeticMax(radix));
		question.message = toString(x, radix) ~ " + " ~ toString(y, radix);
		question.rightAnswer = toString(x + y, radix);
		question.answerChecker = (a) { try { return parse(a, radix) == x + y; } catch { return false; } };
	}, {
		int radix = chooseArithmeticRadix(0);
		int max = arithmeticMax(radix);
		//If max is 2, we want to choose from 0,1,1,2,2,2 which means starting with uniform(0,6), where 6 is the third triangular number.
		//Greater numbers need to be more likely because of how many possible numbers we can reasonably ask the user to subtract from it.
		int triangular = uniform(0, (max+1)*(max+2)/2);
		//Then we need to collapse it from 0,1,2,3,4,5 down to 0-0, 1-0, 1-1, 2-0, 2-1, 2-2.
		int x = 0;
		while (triangular > x) {x++; triangular -= x;}
		int y = triangular;
		question.message = toString(x, radix) ~ " - " ~ toString(y, radix);
		question.rightAnswer = toString(x - y, radix);
		question.answerChecker = (a) { try { return parse(a, radix) == x - y; } catch { return false; } };
	}, {
		int radix = chooseArithmeticRadix(3);
		int x = uniform!"[]"(0, multiplyMax(radix));
		int y = uniform!"[]"(0, multiplyMax(radix));
		question.message = toString(x, radix) ~ " * " ~ toString(y, radix);
		question.rightAnswer = toString(x * y, radix);
		question.answerChecker = (a) { try { return parse(a, radix) == x * y; } catch { return false; } };
	}, {
		int radix = chooseArithmeticRadix(2);
		int y = uniform!"[]"(1, multiplyMax(radix));
		int x = uniform!"[]"(0, multiplyMax(radix)) * y;
		question.message = toString(x, radix) ~ " / " ~ toString(y, radix);
		question.rightAnswer = toString(x / y, radix);
		question.answerChecker = (a) { try { return parse(a, radix) == x / y; } catch { return false; } };
	}, {
		int radix = chooseArithmeticRadix(1);
		int x = uniform!"[]"(0, multiplyMax(radix) * 2);
		int y = uniform!"[]"(1, multiplyMax(radix));
		question.message = toString(x, radix) ~ " MOD " ~ toString(y, radix);
		question.rightAnswer = toString(x % y, radix);
		question.answerChecker = (a) { try { return parse(a, radix) == x % y; } catch { return false; } };
	}, {
		//Base conversions
		int x = uniform!"[]"(0, baseConvertMax);
		question.message = to!string(x) ~ " in HEX";
		question.rightAnswer = to!string(x, 16);
		question.answerChecker = (a) { try { return parse(a, 16) == x; } catch { return false; } };
	}, {
		int x = uniform!"[]"(0, baseConvertMax);
		question.message = toString(x, 16) ~ " in DEC";
		question.rightAnswer = to!string(x);
		question.answerChecker = (a) { try { return to!int(a) == x; } catch { return false; } };
	}, {
		int x = uniform!"[]"(0, baseConvertMax);
		question.message = to!string(x) ~ " in BIN";
		question.rightAnswer = to!string(x, 2);
		question.answerChecker = (a) { try { return parse(a, 2) == x; } catch { return false; } };
	}, {
		int x = uniform!"[]"(0, baseConvertMax);
		question.message = toString(x, 2) ~ " in DEC";
		question.rightAnswer = to!string(x);
		question.answerChecker = (a) { try { return to!int(a) == x; } catch { return false; } };
	}, {
		int x = uniform!"[]"(0, baseConvertMax);
		question.message = toString(x, 2) ~ " in HEX";
		question.rightAnswer = to!string(x, 16);
		question.answerChecker = (a) { try { return parse(a, 16) == x; } catch { return false; } };
	}, {
		int x = uniform!"[]"(0, baseConvertMax);
		question.message = toString(x, 16) ~ " in BIN";
		question.rightAnswer = to!string(x, 2);
		question.answerChecker = (a) { try { return parse(a, 2) == x; } catch { return false; } };
	}, {
		int radix = [2, 10, 16][uniform(0, $)];
		int x = uniform!"[]"(0, baseConvertMax);
		question.message = toString(x, 8) ~ " in " ~ radixString(radix);
		question.rightAnswer = to!string(x, radix);
		question.answerChecker = (a) { try { return parse(a, radix) == x; } catch { return false; } };
	}, {
		int radix = [2, 10, 16][uniform(0, $)];
		int x = uniform!"[]"(0, baseConvertMax);
		question.message = toString(x, radix) ~ " in OCT";
		question.rightAnswer = to!string(x, 8);
		question.answerChecker = (a) { try { return parse(a, 8) == x; } catch { return false; } };
	}, {
		//ASCII conversions
		char c = uniform!"[]"(' ', '~');
		question.message = (c == '"' ? "ASC('\"')" : "ASC(\"" ~ c ~ "\")");
		question.rightAnswer = to!string(cast(int)c);
		question.answerChecker = (a) { try { return to!int(a) == cast(int)c; } catch { return false; } };
	}, {
		char c = uniform!"[]"(' ', '~');
		question.message = "CHR$(" ~ to!string(cast(int)c) ~ ")";
		question.rightAnswer = (c == ' ' ? "Space" : to!string(c));
		question.answerChecker = a => a.length == 1 && a[0] == c;
	}, {
		//Bitwise
		int radix = chooseBitwiseRadix();
		int bitwiseMask = (1 << bitwiseBits) - 1;
		int x = uniform!"[]"(0, bitwiseMask);
		int y = uniform!"[]"(0, bitwiseMask);
		question.message = toString(x, radix) ~ " AND " ~ toString(y, radix);
		question.rightAnswer = toString(x & y, radix);
		question.answerChecker = (a) { try { return parse(a, radix) == (x & y); } catch { return false; } };
	}, {
		int radix = chooseBitwiseRadix();
		int bitwiseMask = (1 << bitwiseBits) - 1;
		int x = uniform!"[]"(0, bitwiseMask);
		int y = uniform!"[]"(0, bitwiseMask);
		question.message = toString(x, radix) ~ " OR " ~ toString(y, radix);
		question.rightAnswer = toString(x | y, radix);
		question.answerChecker = (a) { try { return parse(a, radix) == (x | y); } catch { return false; } };
	}, {
		int radix = chooseBitwiseRadix();
		int bitwiseMask = (1 << bitwiseBits) - 1;
		int x = uniform!"[]"(0, bitwiseMask);
		int y = uniform!"[]"(0, bitwiseMask);
		question.message = toString(x, radix) ~ " XOR " ~ toString(y, radix);
		question.rightAnswer = toString(x ^ y, radix);
		question.answerChecker = (a) { try { return parse(a, radix) == (x ^ y); } catch { return false; } };
		/*No NAND and NOR - they were a nice idea but they're too fiddly in terms of when the 1's on the left should stop.
	}, {
		int radix = chooseBitwiseRadix();
		int bitwiseMask = (1 << bitwiseBits) - 1;
		int x = uniform!"[]"(0, bitwiseMask);
		int y = uniform!"[]"(0, bitwiseMask);
		question.message = toString(x, radix) ~ " NAND " ~ toString(y, radix) ~ " (MOD " ~ toString(1 << bitwiseBits, radix) ~ ")";
		question.rightAnswer = toString((x & y) ^ bitwiseMask, radix);
		question.answerChecker = (a) { try { return parse(a, radix) == ((x & y) ^ bitwiseMask); } catch { return false; } };
	}, {
		int radix = chooseBitwiseRadix();
		int bitwiseMask = (1 << bitwiseBits) - 1;
		int x = uniform!"[]"(0, bitwiseMask);
		int y = uniform!"[]"(0, bitwiseMask);
		question.message = toString(x, radix) ~ " NOR " ~ toString(y, radix) ~ " (MOD " ~ toString(1 << bitwiseBits, radix) ~ ")";
		question.rightAnswer = toString((x | y) ^ bitwiseMask, radix);
		question.answerChecker = (a) { try { return parse(a, radix) == ((x | y) ^ bitwiseMask); } catch { return false; } };
		*/
	}
];

string radixString(int radix) {
	switch (radix) {
		case 2: return "BIN";
		case 8: return "OCT";
		case 10: return "DEC";
		case 16: return "HEX";
		default: return "Base " ~ to!string(radix);
	}
}

string toString(int v, int radix) {
	string s = to!string(v, radix);
	if (radix == 10) return s;
	if (radix == 16) return s ~ "[HEX]";
	if (radix == 2) return s ~ "[BIN]";
	if (radix == 8) return s ~ "[OCT]";
	return s ~ "[Base" ~ to!string(radix) ~ "]";
}

int parse(const(char)[] a, int radix) {
	string suffix = "[" ~ to!string(radix) ~ "]";
	if (a.endsWith(suffix)) return to!int(a[0..$-suffix.length], radix);
	suffix = "[Base" ~ to!string(radix) ~ "]";
	if (a.endsWith!"a.toLower()==b.toLower()"(suffix)) return to!int(a[0..$-suffix.length], radix);
	if (radix == 16) {
		if (a.startsWith("0x") || a.startsWith("0X") || a.startsWith("&h") || a.startsWith("&H")) a = a[2..$];
		else if (a.startsWith("&")) a = a[1..$];
		else if (a.endsWith("h") || a.endsWith("H")) a = a[0..$-1];
		else if (a.endsWith!"a.toLower()==b.toLower()"("[HEX]")) a = a[0..$-5];
	}
	if (radix == 2) {
		if (a.startsWith("0b") || a.startsWith("0B") || a.startsWith("&b") || a.startsWith("&B")) a = a[2..$];
		else if (a.endsWith("b") || a.endsWith("B")) a = a[0..$-1];
		else if (a.endsWith!"a.toLower()==b.toLower()"("[BIN]")) a = a[0..$-5];
	}
	if (radix == 8) {
		if (a.startsWith("0o") || a.startsWith("0O") || a.startsWith("&o") || a.startsWith("&O")) a = a[2..$];
		else if (a.endsWith("o") || a.endsWith("O")) a = a[0..$-1];
		else if (a.endsWith!"a.toLower()==b.toLower()"("[OCT]")) a = a[0..$-5];
	}
	return to!int(a, radix);
}
