import std.stdio;
import allegro5.allegro;
import std.exception;
static import textbuffer;
static import game;
static import highscores;
import hyperparticles;
import allegro5.allegro_image;
import std.conv;
import shaders;
import sound;

int main(string[] argv) {
	return al_run_allegro({
		//In debug mode, make sure the console stays open in the case of an error.
		debug {
			try {
				return main2();
			}
			catch (Throwable e) {
				writeln(e.toString());
				writeln("Press Enter to close");
				stdin.readln();
				return 1;
			}
		}
		else {
			return main2();
		}
	});
}

ALLEGRO_DISPLAY* display;

ALLEGRO_BITMAP* font;
ALLEGRO_BITMAP*[96] fontChars;

bool shouldExit = false;
int lastDraw, ticks;

enum DEFAULT_SCREEN_W = 800;
enum DEFAULT_SCREEN_H = 600;
int windowX, windowY, windowWidth, windowHeight;

int main2() {
	enforce(al_init(), "al_init");
	enforce(al_install_keyboard(), "al_install_keyboard");
	enforce(al_init_image_addon(), "al_init_image_addon");

	auto eventQueue = enforce(al_create_event_queue(), "al_create_event_queue");
	scope (exit) al_destroy_event_queue(eventQueue);

	al_register_event_source(eventQueue, al_get_keyboard_event_source());

	auto timer = enforce(al_create_timer(0.01), "al_create_timer");
	scope (exit) al_destroy_timer(timer);
	al_register_event_source(eventQueue, al_get_timer_event_source(timer));

	bool fullscreen;
	highscores.init(fullscreen, windowX, windowY, windowWidth, windowHeight);

	if (windowX != int.min) al_set_new_window_position(windowX, windowY);

	enum flags = ALLEGRO_OPENGL | ALLEGRO_PROGRAMMABLE_PIPELINE | ALLEGRO_RESIZABLE;
	if (fullscreen) {
		al_set_new_display_flags(flags | ALLEGRO_FULLSCREEN_WINDOW);
		display = al_create_display(windowWidth, windowHeight);	//might fail
	}
	if (!display) {
		al_set_new_display_flags(flags);	//no fullscreen
		display = al_create_display(windowWidth, windowHeight);	//also might fail
		if (!display) {
			al_set_new_window_position(int.max, int.max);	//default behaviour
			windowWidth = DEFAULT_SCREEN_W;
			windowHeight = DEFAULT_SCREEN_H;
			display = enforce(al_create_display(windowWidth, windowHeight), "al_create_display");
		}
	}
	scope (exit) al_destroy_display(display);
	al_register_event_source(eventQueue, al_get_display_event_source(display));

	al_set_window_constraints(display, 8*textbuffer.w, 20*textbuffer.h, 0, 0);
	al_set_window_title(display, "The Human CPU Experiment");

	al_set_new_bitmap_flags(ALLEGRO_VIDEO_BITMAP | ALLEGRO_MAG_LINEAR | ALLEGRO_MIN_LINEAR);
	font = enforce(al_load_bitmap("font.png"), "font.png");
	scope (exit) al_destroy_bitmap(font);
	scope (exit) foreach (int i; 0..96) al_destroy_bitmap(fontChars[i]);
	foreach (int i; 0..96) {
		fontChars[i] = enforce(al_create_sub_bitmap(font, 1+(i%16)*9, 1+(i/16)*9, 8, 8), "font.png sub " ~ to!string(i));
	}

	al_set_display_icon(display, fontChars['X' - 32]);

	scope (exit) destroyShaders();
	initShaders();

	scope (exit) stopSound();
	initSound();

	init();

	al_start_timer(timer);

	ALLEGRO_EVENT event;
	while (!shouldExit) {
		if (al_is_event_queue_empty(eventQueue)) {
			draw();
			lastDraw = ticks;
		}
		al_wait_for_event(eventQueue, &event);
		switch (event.type) {
			case ALLEGRO_EVENT_TIMER:
				if (ticks - lastDraw < 10) {
					updateSound();
					tick();
					ticks++;
				}
				break;
			case ALLEGRO_EVENT_KEY_CHAR:
				handleKeyChar(event);
				break;
			case ALLEGRO_EVENT_DISPLAY_CLOSE:
				shouldExit = true;
				break;
			case ALLEGRO_EVENT_DISPLAY_RESIZE:
				al_acknowledge_resize(display);
				if (!(al_get_display_flags(display) & ALLEGRO_FULLSCREEN_WINDOW)) {
					al_get_window_position(display, &windowX, &windowY);
					windowWidth = al_get_display_width(display);
					windowHeight = al_get_display_height(display);

					//Call this as a workaround for Allegro not remembering the new size when going into and out of fullscreen.
					al_resize_display(display, windowWidth, windowHeight);
				}
				break;
			default:
				break;
		}
	}

	al_get_window_position(display, &windowX, &windowY);
	highscores.save();

	return 0;
}

bool inGame;
bool particlesOn;
typeof(textbuffer.scr) particleScr;
int particleCx, particleCy;	//cursor
bool nonParticleScreenOn = true;

void init() {
	textbuffer.clear();
	highscores.activate(0);

	particlesOn = false;
}

void tick() {
	if (inGame) game.tick();
	else highscores.tick();
	if (particlesOn) updateParticles();
}

void handleKeyChar(ref ALLEGRO_EVENT event) {
	//Alt+Enter = switch in and out of full screen!
	version (OSX) {
		bool fsPressed = (event.keyboard.keycode == ALLEGRO_KEY_ENTER && (event.keyboard.modifiers & ALLEGRO_KEYMOD_COMMAND));
	} else {
		bool fsPressed = (event.keyboard.keycode == ALLEGRO_KEY_ENTER && (event.keyboard.modifiers & ALLEGRO_KEYMOD_ALT));
	}
	if (fsPressed) {
		if (al_get_display_flags(display) & ALLEGRO_FULLSCREEN_WINDOW) {
			al_set_display_flag(display, ALLEGRO_FULLSCREEN_WINDOW, false);
			al_set_window_position(display, windowX, windowY);
		} else {
			al_get_window_position(display, &windowX, &windowY);
			al_set_display_flag(display, ALLEGRO_FULLSCREEN_WINDOW, true);
		}
		return;
	}

	if (inGame) game.handleKeyChar(event);
	else highscores.handleKeyChar(event);
}

void draw() {
	al_use_shader(fontShader);

	al_clear_to_color(al_map_rgb(0, 0, 0));

	if (particlesOn) drawScreen!true(particleScr, particleCx, particleCy);
	if (nonParticleScreenOn) drawScreen!false(textbuffer.scr, textbuffer.cx, textbuffer.cy);

	al_flip_display();
}

void drawScreen(bool useParticles)(ref typeof(textbuffer.scr) scr, int cx, int cy) {
	al_hold_bitmap_drawing(true);

	foreach (int y; 0..textbuffer.h) {
		foreach (int x; 0..textbuffer.w) {
			char c = scr[y][x];
			auto p = &particles[y][x];
			drawCharacter!useParticles(x, y, c, *p);
		}
	}

	//Cursor
	if (ticks % 100 < 50)
		drawCharacter!useParticles(textbuffer.cx, textbuffer.cy, '_', cursorParticle);

	al_hold_bitmap_drawing(false);
}

void drawCharacter(bool useParticles)(int x, int y, char c, ref HParticles!3.PARTICLE p) {
	//Character size on screen
	float cw = cast(float)al_get_display_width(display) / textbuffer.w;
	float ch = cast(float)al_get_display_height(display) / textbuffer.h;

	if (c >= 32 && c < 128) {
		float x2 = x, y2 = y, zoomScale = 1;
		if (useParticles && !projectParticle(p, x2, y2, zoomScale)) return;
		al_draw_scaled_bitmap(fontChars[c - 32], 0, 0, 8, 8, x2*cw, y2*ch, zoomScale*cw, zoomScale*ch, 0);
	}
}
