import textbuffer;
import std.math;
import std.random;
import main;

//A particle engine using hyperbolic space! :D

//A beautiful "intuitionist" description of hyperbolic geometry: http://www.cabinetmagazine.org/issues/16/crocheting.php
//We can summarise it like this:
//- On a normal plane, you can tessellate hexagons.
//- On a sphere's surface, you get less space than you think as you radiate out from a point, and you have to insert pentagons to make the tessellation work.
//- In 2D hyperbolic space, you get more space than you think as you radiate out from a point, and you have to insert heptagons to make the tessellation work.
//Is your mind blown yet?

//Here's a discussion of how we represent points in hyperbolic space: https://www.physicsforums.com/threads/coordinates-in-hyperbolic-geometry.311320/

version = usePoincaréBall;

//Wow. It is AMAZING how fast precision is lost, even with 'real' here!!! I guess that's because sinh and cosh increase so rapidly.
//Hyperbolic space is hyperbolic.
alias flt = real;
version (usePoincaréBall) {
	//The smaller the Poincaré ball (used when projecting into and out of hyperbolic space), the more pronounced the hyperbolic space distortion becomes.
	enum ballRadius = 2;
} else {
	//The large the scale (used when projecting into and out of hyperbolic space), the more pronounced the hyperbolic space distortion becomes.
	enum scale = 5;
}

//N is the number of dimensions that the hyperbolic space has.
//Hey, if we're going to implement this at all, we might as well do it generalised for all dimensionalities! :D
template HParticles(int N) {
	//We use the hyperboloid model.
	//Vectors are points in (N+1)D Euclidean space.
	//Position vectors are always normalised such that u^2 - sum(v[i]^2 : i=0..N) = 1 (and u > 0).
	//Velocity vectors are specified as a second vector that can be 'added' to the first using Lorentz boost composition.
	struct HVEC {
		flt u;
		flt v[N];
	}

	flt distance(ref HVEC x, ref HVEC y) {
		flt b = x.u*y.u;
		foreach (int i; 0..N) b -= x.v[i]*y.v[i];
		return acosh(b);
	}

	void setRelativeToOrigin(ref HVEC x, flt v[N]) {
		//Wish: I'm not sure this will give a distance() from origin that matches the magnitude of x, but it's good enough for this game.
		flt uSq = 0;
		foreach (int i; 0..N) {
			flt vi = sinh(v[i]);
			x.v[i] = vi;
			uSq += vi*vi;
		}
		x.u = sqrt(1 + uSq);
	}

	struct PARTICLE {
		HVEC pos;
		HVEC vel;
		flt size;	//gets cleared to 0 if the particle overflows to inf, so we can skip it.
		//Wish: orientation!
	}

	//Does a Lorentz boost that effectively sets 'pos' to 'pos + vel'.
	//This is only half the story, as 'vel' effectively needs to rotate, but the caller takes care of that by calling us twice.
	void composeBoost(ref HVEC pos, ref HVEC vel) {
		//Applying the velocity to the particles is a bit tricky.

		//First we need a hyperbolic translation that would take a position from (1,0,0,0) to this particle's position.
		//It turns out Lorentz boosts (named for the idea that if you want to translate your observations into a moving object's observations in relativity,
		//you 'boost' your velocity to match, which is done using a transformation matrix affecting both space and time relative to you)
		//are also appropriate here and will preserve the overall shape of the hyperboloid, but instead of velocity, we have a vector from (0,0,0,0) to the current position.
		//See https://en.wikipedia.org/wiki/Lorentz_transformation#boost
		flt beta[N] = pos.v[] / -pos.u;
		flt betaSq = 0; foreach (int i; 0..N) betaSq += beta[i] * beta[i];
		flt gamma = 1 / sqrt(1 - betaSq);

		//Now we need to translate our new position back using the translation we prepared earlier.
		//See https://en.wikipedia.org/wiki/Lorentz_transformation#Matrix_forms
		//If N were 2, we'd want:
		//pos.u = gamma * vel.u - gamma * beta[0] * vel.v[0] - gamma * beta[1] * vel.v[1];
		pos.u = gamma * vel.u;
		foreach (int j; 0..N) {
			flt gammaBetaJ = gamma * beta[j];
			pos.u -= gammaBetaJ * vel.v[j];
			//If N were 2, we'd want:
			//pos.v[0] = -gamma * beta[0] * vel.u + (1 + (gamma - 1) * beta[0] * beta[0] / betaSq) * vel.v[0] + ((gamma - 1) * beta[0] * beta[1] / betaSq) * vel.v[1];
			//pos.v[1] = -gamma * beta[1] * vel.u + ((gamma - 1) * beta[1] * beta[0] / betaSq) * vel.v[0] + (1 + (gamma - 1) * beta[1] * beta[1] / betaSq) * vel.v[1];
			pos.v[j] = vel.v[j] - gammaBetaJ * vel.u;
			//If N were 2, we'd now need to do:
			//pos.v[0] += ((gamma - 1) * beta[0] * beta[0] / betaSq) * vel.v[0] + ((gamma - 1) * beta[0] * beta[1] / betaSq) * vel.v[1];
			//pos.v[1] += ((gamma - 1) * beta[1] * beta[0] / betaSq) * vel.v[0] + ((gamma - 1) * beta[1] * beta[1] / betaSq) * vel.v[1];
			flt gammaM1BetaJOnBetaSq = (gamma - 1) * beta[j] / betaSq;
			foreach (int i; 0..N) pos.v[j] += gammaM1BetaJOnBetaSq * beta[i] * vel.v[i];
		}

	}

	void updateParticle(ref PARTICLE particle) {
		if (particle.size == 0) return;

		//Set the position using boost composition: (1,0,0,0) to pos followed by (1,0,0,0) to vel.
		HVEC oldPos = particle.pos;
		particle.pos.composeBoost(particle.vel);

		//It doesn't quite stop there. Composing two boosts actually leads to a boost plus a rotation.
		//We therefore need to apply a rotation to the velocity ready for next time.
		//The easiest way for us is to use the asymmetry: what if we had started at the new position and now want the old position?
		particle.vel = particle.pos;
		oldPos.v[] = -oldPos.v[];
		particle.vel.composeBoost(oldPos);

		if (!particle.pos.u.isFinite()) particle.size = 0;
	}
}

//Let's use a 3D hyperbolic space. Because SPAAAAAACE.
HParticles!3.PARTICLE particles[h][w];
HParticles!3.PARTICLE cursorParticle;

//Needs to be called at the time of explosion (not earlier) so that we get the correct cursor position!
void initParticles() {
	foreach (int y; 0..h) {
		foreach (int x; 0..w) {
			initParticle(particles[y][x], x, y);
		}
	}
	initParticle(cursorParticle, cx, cy);
}

void initParticle(ref HParticles!3.PARTICLE p, int x, int y) {
	version (usePoincaréBall) {
		//Set a position in the Poincaré ball based on the letter's screen position.
		flt v[3];
		v[0] = (x + 0.5f) / w - 0.5f;
		v[1] = (y + 0.5f) / h - 0.5f;
		v[2] = 1f;
		v[] /= ballRadius;
		//Project this on to the hyperboloid.
		//See https://en.wikipedia.org/wiki/Poincar%C3%A9_disk_model#Relation_to_the_hyperboloid_model
		p.pos.u = 1;
		flt d = 1;
		foreach (int i; 0..3) {
			flt vSq = v[i]*v[i];
			p.pos.u += vSq;
			d -= vSq;
		}
		p.pos.u /= d;
		p.pos.v[] = v[]*(2/d);
	} else {
		//Set the hyperbolic position based on the letter's screen position.
		flt xf = (x + 0.5f) / w - 0.5f;
		flt yf = (y + 0.5f) / h - 0.5f;
		HParticles!3.setRelativeToOrigin(p.pos, [xf*scale, yf*scale, scale]);
	}

	//Choose a random velocity.
	flt vel[3];
	enum xr = 0.01f, yr = 0.01f, zr = 0.03f;
	vel[0] = uniform!"[]"(-xr, +xr);
	vel[1] = uniform!"[]"(-yr, +yr);
	vel[2] = uniform!"[]"(-zr, +zr);

	//We need to do some cosh and sinh stuff so that if we were to call distance((1,0,0,0), vel), we'd get our speed.
	//See graphs of sinh and cosh, e.g. http://home.scarlet.be/math/hypf.gif
	HParticles!3.setRelativeToOrigin(p.vel, vel);

	//For making characters smaller in the distance (hyperbolic equivalent thereof),
	//I'll just use 'u+1' (distance from plane containing projection point for Poincaré ball) as a rough measure of how far it is from the origin.
	//Store the initial value of this so we can ensure the characters start at their normal on-screen size.
	//This probably isn't correct but I've spent enough time on this and it looks nice.
	p.size = p.pos.u + 1;
}

bool projectParticle(ref HParticles!3.PARTICLE p, out float x, out float y, out float zoomScale) {
	if (p.size == 0) return false;

	version (usePoincaréBall) {
		//Project the particle into the unit Poincaré ball.
		//See https://en.wikipedia.org/wiki/Poincar%C3%A9_disk_model#Relation_to_the_hyperboloid_model
		flt v[3] = p.pos.v[] / (p.pos.u + 1);

		/*Test code: move/rotate the ball so we can see it all
		flt a=v[0], b=v[2], s=sin(ticks*0.01f), c=cos(ticks*0.01f);
		v[0] = a*c - b*s; v[2] = b*c + a*s;
		v[2] += 2;
		//*/

		//Now use standard 3D projection.
		zoomScale = p.size / (p.pos.u + 1);
		x = (v[0]/v[2] + 0.5f) * w - 0.5f*zoomScale;
		y = (v[1]/v[2] + 0.5f) * h - 0.5f*zoomScale;
		return v[2] > 0.01f;
	} else {
		//Project the particle back to a 3D camera space position...
		float x3 = asinh(p.pos.v[0]);
		float y3 = asinh(p.pos.v[1]);
		float z3 = asinh(p.pos.v[2]);
		//...and then to a 2D screen position using normal 3D-to-2D projection.
		zoomScale = scale/z3;
		x = (x3/z3 + 0.5f) * w - 0.5f*zoomScale;
		y = (y3/z3 + 0.5f) * h - 0.5f*zoomScale;
		return z3 > 0.01f;
	}
}

void updateParticles() {
	foreach (int y; 0..h) {
		foreach (int x; 0..w) {
			HParticles!3.updateParticle(particles[y][x]);
		}
	}
	HParticles!3.updateParticle(cursorParticle);
}
