/*

INVINCIBLE COUNTERMEASURE
Copyright 2014 Linley Henzell
Licensed under the GPL v3 (or later)

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.


How to compile

I use Code::Blocks and don't really understand makefiles, so there isn't one.
It shouldn't be too hard to compile the game, though. Basically, compile
everything in the source directory and link it against Allegro 5.0
(I have 5.0.10. 5.1 may also work, but I haven't tested this).

The game should run on any platform supported by Allegro 5. It shouldn't have
any other dependencies.


Structure

The source code is organised into categories indicated by the first letter
of each file. Each .c file has a header file, and there are some other special
header files as well.

The categories are:


Main/miscellaneous

m_input.c - contains Allegro calls that read user input (see also i_input.c)
m_main.c - this file. Contains the main function and some initialisation stuff
m_maths.c - special maths functions

m_config.h - header file containing some configuration options


Game

g_client.c - deals with client programs
g_cloud.c - handles clouds (particle effects and similar)
g_code.c - executes system, client and observer code
g_game.c - contains the main game loop and some initialisation stuff
g_group.c - handles physics for groups (multiple processes joined together)
g_method.c - general method-related code. Contains some detailed method code, although much of it is in the following files
g_method_clob.c - client/observer method code. Note: console method code is in i_console.c
g_method_ex.c - not used (I should remove this)
g_method_misc.c - miscellaneous method code
g_method_pr.c - process method code
g_method_sy.c - system method code
g_misc.c - some miscellaneous game stuff
g_motion.c - handles physics
g_packet.c - runs packets
g_proc.c - deals with certain aspects of process creation (e.g. deriving process features from bcode interface definitions)
g_proc_run.c - runs processes
g_shape.c - initialises the shape data structures
g_world.c - general game-running functions

g_header.h - contains a vast amount of game data declarations


Interface (runs the display)

i_buttons.c - contains code for interface buttons
i_console.c - runs consoles and the process data box. Contains console method code
i_disp_in.c - contains some display initialisation code
i_display.c - runs the display
i_error.c - writes some errors to consoles
i_input.c - takes the raw input data generated by m_input.c and prepares it to be accessed by the observer/operator input method
i_programs.c - displays the Pr program execution panel
i_sysmenu.c - displays the Sy system menu panel
i_view.c - contains some view initialisation stuff

i_header.h - display-related declarations


Sound effects

x_init.c - initialises sound. Creates the separate sound thread.
x_music.c - unfinished music code
x_sound.c - sound effect code
x_synth.c - an unfinished synthesiser


Start (runs the start menu and some other things that actually belong in the g_ files)

s_menu.c - runs the start menu
s_mission.c - loads and starts missions
s_pregame.c - handles the pregame
s_setup.c - game setup code
s_turn.c - handles turns and inter-turn time


Templates

t_files.c - opens and reads files for templates
t_template.c - handles templates and the Te template menu


Virtual machine

v_interp.c - contains the bcode interpreter


Compiler

c_asm.c - contains the assembler (called by the compiler)
c_comp.c - a sprawling mess that contains most of the compiler
c_ctoken.c - the text parser and bits of the compiler
c_data.c - decodes the data() built-in function
c_init.c - initialises the compiler
c_inter.c - code generator
c_prepr.c - preprocessor

c_header.h - contains a lot of compiler-related stuff


Code editor

e_build.c - interface between the editor and compiler
e_clip.c - clipboard (cut/paste, undo/redo etc)
e_complete.c - code completion
e_editor.c - main editor functions
e_files.c - file functions
e_help.c - contextual help
e_inter.c - editor interface display
e_log.c - runs the message window for the editor/template panels
e_slider.c - runs scrollbars and similar things
e_tools.c - a few general editor functions

e_header.h - general editor declarations


Files

f_game.c - gamefiles
f_load.c - loads and verifies saved games
f_save.c - saves games
f_turn.c - turnfiles


*/

#include "m_config.h"

#include <allegro5/allegro.h>
#include <stdio.h>
#include <allegro5/allegro_primitives.h>
#include <allegro5/allegro_image.h>
#include <string.h>

#include "g_header.h"

#include "i_header.h"
#include "i_disp_in.h"
#include "i_view.h"
#include "i_sysmenu.h"
#include "i_programs.h"
//#include "g_init.h"
#include "g_game.h"
#include "g_misc.h"
#include "g_shape.h"

#include "c_header.h"
#include "c_asm.h"
#include "c_comp.h"
#include "e_slider.h"
#include "e_header.h"
#include "e_editor.h"
#include "t_template.h"
#include "m_input.h"
#include "s_menu.h"
#include "s_mission.h"
#include "x_init.h"
#include "x_sound.h"

#include "m_maths.h"

// timer interrupt functions and variables:
void framecount(void);

volatile int framecounter;
volatile int frames_per_second;

// startup function declarations:
void init_at_startup(void);
int finish_initfile_line(char* buffer, int buffer_length, int bpos);
int read_initfile_line(char* buffer, int buffer_length, int bpos);
int read_initfile_number(int* read_number, char* buffer, int buffer_length, int bpos);
int read_initfile_word(char* ifword, char* buffer, int buffer_length, int bpos);
char check_initfile_char(char* buffer, int buffer_length, int bpos);
void load_font(int f, const char* font_file_name, int height);
void read_initfile(void);

extern ALLEGRO_DISPLAY* display;
extern ALLEGRO_BITMAP* display_window; // in i_display.c
extern ALLEGRO_EVENT_QUEUE* event_queue;
extern ALLEGRO_EVENT_QUEUE* fps_queue;
struct fontstruct font [FONTS];
ALLEGRO_TIMER* timer;
ALLEGRO_TIMER* timer_1_second;

// the following are all externed in m_globvars.h:
struct settingsstruct settings;
struct worldstruct w;
struct ex_controlstruct ex_control;

extern ALLEGRO_BITMAP* title_bitmap; // in s_menu.c

int main(void)
{

   display = NULL;
   timer = NULL;


   if (!al_init())
   {
      fprintf(stdout, "\nError: failed to initialize Allegro.");
      return -1;
   }


   timer = al_create_timer(0.03);
   if (!timer)
   {
      fprintf(stdout, "\nError: failed to create timer.");
      return -1;
   }

   timer_1_second = al_create_timer(1);
   if (!timer_1_second)
   {
      fprintf(stdout, "Error: failed to create second timer.");
      return -1;
   }


 al_start_timer(timer);
 al_start_timer(timer_1_second);

 init_random_numbers(0); // this initialises the rand seed (using srand) and also prepares the irand buffer (see g_misc.c)

 init_at_startup();

 initialise_display();

 init_shapes(); // for debugging purposes, needs to be after initialise_display()

 init_editor(); // in e_editor.c. Must come after initialise_display

 init_templates(); // in t_template.c. This call must be after init_editor() and init_at_startup()

 init_sysmenu(); // in i_sysmenu.c. This call must be after init_editor()
 init_programs_menu(); // in i_programs.c. This call must be after init_editor()

 start_menus(); // game loop is called from here

 return 0;

}



void init_at_startup(void)
{

   int i;

   al_install_keyboard();
   al_install_mouse();

   al_init_primitives_addon();

   al_init_image_addon();

   fprintf(stdout, "Invincible Countermeasure");
   fprintf(stdout, "\nCopyright 2014-2015 Linley Henzell");
   fprintf(stdout, "\nVersion: beta 2");

   fprintf(stdout, "\n\nThis is free software and comes with no warranty (see licence.txt).");

//   fprintf(stdout, "\n\nThis program comes with absolutely no warranty!");
/*

       <program>  Copyright (C) <year>  <name of author>
    This program comes with ABSOLUTELY NO WARRANTY; for details type `show w'.
    This is free software, and you are welcome to redistribute it
    under certain conditions; type `show c' for details.
  */

   fprintf(stdout, "\nFor instructions, read manual.htm.");
   fprintf(stdout, "\nTo configure screen resolution etc, edit init.txt.");
   fprintf(stdout, "\n\nHave fun!\n\n");

//  al_set_new_display_option(ALLEGRO_SAMPLE_BUFFERS, 1, ALLEGRO_SUGGEST);
//  al_set_new_display_option(ALLEGRO_SAMPLES, 2, ALLEGRO_SUGGEST);
//  These might be nice, but cause strange behaviour and segfaults on my development computer.

//   al_set_new_display_flags(ALLEGRO_FULLSCREEN);

   settings.option [OPTION_WINDOW_W] = 1024;
   settings.option [OPTION_WINDOW_H] = 768;
   settings.option [OPTION_FULLSCREEN] = 0;
   settings.option [OPTION_VOL_MUSIC] = 80;
   settings.option [OPTION_VOL_EFFECT] = 80;
   settings.option [OPTION_SPECIAL_CURSOR] = 0;

// any settings values that could be set in the initfile need to be initialised to default values before read_initfile() is called.
   read_initfile();

// Set up multisampled anti-aliasing
    al_set_new_display_option(ALLEGRO_SAMPLE_BUFFERS, 1, ALLEGRO_SUGGEST);
    al_set_new_display_option(ALLEGRO_SAMPLES, 2, ALLEGRO_SUGGEST);

//settings.option [OPTION_FULLSCREEN] = 1;

   if (settings.option [OPTION_FULLSCREEN] == 1)
   {
// We use ALLEGRO_FULLSCREEN_WINDOW rather than ALLEGRO_FULLSCREEN here because true fullscreen has problems with native file menus
    al_set_new_display_flags(ALLEGRO_FULLSCREEN_WINDOW);

// OPTION_WINDOW_W/H are not used (although I think they are used if for some reason the game swaps out of fullscreen? Not sure)
    display = al_create_display(settings.option [OPTION_WINDOW_W], settings.option [OPTION_WINDOW_H]);

    if (!display)
    {
       fprintf(stdout, "\nError: failed to create fullscreen display.");
       safe_exit(-1);
    }

    settings.option [OPTION_WINDOW_W] = al_get_display_width(display);
    settings.option [OPTION_WINDOW_H] = al_get_display_height(display);

    if (settings.option [OPTION_WINDOW_W] < 1024
     || settings.option [OPTION_WINDOW_H] < 768)
    {
       fprintf(stdout, "\nError: display too small (should be at least 1024x768, but is %ix%i).", settings.option [OPTION_WINDOW_W], settings.option [OPTION_WINDOW_H]);
       safe_exit(-1);
    }

   }
    else
    {
     display = al_create_display(settings.option [OPTION_WINDOW_W], settings.option [OPTION_WINDOW_H]);

     if (!display)
     {
       fprintf(stdout, "\nError: failed to create display window.");
       safe_exit(-1);
     }

    }

// init_sound must come after read_initfile() (as read_initfile may set volume levels)
   init_sound(); // calls allegro sound init functions and loads samples. If it fails, it will disable sound (through settings.sound_on)

   settings.edit_window_columns = 80;
   settings.editor_x_split = settings.option [OPTION_WINDOW_W] - (settings.edit_window_columns * 6) - 30;

   init_view_at_startup(settings.option [OPTION_WINDOW_W], settings.option [OPTION_WINDOW_H]); // in i_view.c

   al_set_target_bitmap(al_get_backbuffer(display));

   if (settings.option [OPTION_SPECIAL_CURSOR] == 0)
    al_show_mouse_cursor(display);
     else
      al_hide_mouse_cursor(display);

   event_queue = al_create_event_queue();
   if (!event_queue)
   {
      fprintf(stdout, "\nError: failed to create event_queue.");
      al_destroy_display(display);
      al_destroy_timer(timer);
      safe_exit(-1);
   }

   fps_queue = al_create_event_queue();
   if (!fps_queue)
   {
      fprintf(stdout, "\nError: failed to create fps_queue.");
      al_destroy_display(display);
      al_destroy_timer(timer);
      safe_exit(-1);
   }

// abcdefghijklmnopqrstuvwxyz

   al_register_event_source(event_queue, al_get_timer_event_source(timer));
   al_register_event_source(fps_queue, al_get_timer_event_source(timer_1_second));

   al_init_font_addon(); // initialize the font addon

   load_font(FONT_BASIC, "data/images/fwss_font.bmp", 12);
   load_font(FONT_BASIC_BOLD, "data/images/fwb_font.bmp", 12);
   load_font(FONT_SQUARE, "data/images/fwt_font.bmp", 16);
   load_font(FONT_SQUARE_BOLD, "data/images/fwt_font.bmp", 16); // currently the same as FONT_SQUARE
   load_font(FONT_SQUARE_LARGE, "data/images/fwtb_font.bmp", 20);

   init_trig();

   init_nearby_distance(); // in maths.c

   init_key_type(); // in m_input.c

   init_ex_control(); // in m_input.c

   w.allocated = 0; // indicates world doesn't need to be deallocated before use

// The basic (non-process) programs all use static arrays in the worldstruct as their bcode:
   w.system_program.bcode.op = w.system_bcode_op;
   w.system_program.bcode.bcode_size = SYSTEM_BCODE_SIZE;
   w.observer_program.bcode.op = w.observer_bcode_op;
   w.observer_program.bcode.bcode_size = CLIENT_BCODE_SIZE;

   for (i = 0; i < PLAYERS; i ++)
   {
    w.player[i].client_program.bcode.op = w.player_bcode_op[i];
    w.player[i].client_program.bcode.bcode_size = CLIENT_BCODE_SIZE;
   }

// Read in the mission progress file:
  load_mission_status_file();

// load the title bitmap
  title_bitmap = al_load_bitmap("data/images/title.bmp");

  if (!title_bitmap)
		{
			fprintf(stdout, "\nError: failed to load data/images/title.bmp.");
			error_call();
		}

  al_convert_mask_to_alpha(title_bitmap, al_get_pixel(title_bitmap, 0, 0));


}


void load_font(int f, const char* font_file_name, int height)
{

   ALLEGRO_BITMAP *font_bmp = al_load_bitmap(font_file_name);

   if(!font_bmp)
   {
      fprintf(stdout, "\nError:failed to load font file (%s)", font_file_name);
      safe_exit(-1);
   }

   al_convert_mask_to_alpha(font_bmp, al_get_pixel(font_bmp, 1, 1));

   int ranges[] = {0x0020, 0x007F};
   font[f].fnt = al_grab_font_from_bitmap(font_bmp, 1, ranges);

   if (!font[f].fnt)
   {
      fprintf(stdout, "\nError: failed to grab font from file (%s)", font_file_name);
      error_call();
   }

   al_destroy_bitmap(font_bmp);

// should only use fix-width fonts:
   font[f].width = al_get_text_width(font[f].fnt, "a");

   font[f].height = height;


}



// Tries to open file init.txt and read game settings from it.
// If it fails for some reason, settings are not updated (they should have already been set to defaults).
void read_initfile(void)
{

#define INITFILE_SIZE 4096
#define INITFILE_WORD_LENGTH 20

 FILE *initfile;
 char buffer [INITFILE_SIZE];

 initfile = fopen("init.txt", "rt");

 if (!initfile)
 {
  fprintf(stdout, "\nFailed to open init.txt. Starting with default settings.");
  return;
 }

 int read_in = fread(buffer, 1, INITFILE_SIZE, initfile);

 if (ferror(initfile)
  || read_in == 0)
 {
  fprintf(stdout, "\nFailed to read settings from init.txt. Starting with default settings.");
  fclose(initfile);
  return;
 }

 int bpos = 0;

 while(TRUE)
 {
  bpos = read_initfile_line(buffer, read_in, bpos);
  if (bpos == -1)
   break;
 };

 fclose(initfile);

}

// returns -1 if end of file reached, new position in buffer otherwise
int read_initfile_line(char* buffer, int buffer_length, int bpos)
{

 char initfile_word [INITFILE_WORD_LENGTH];

// first check for a # at the start of the line, indicating a comment:
 if (check_initfile_char(buffer, buffer_length, bpos) == '#')
 {
  bpos = finish_initfile_line(buffer, buffer_length, bpos);
  return bpos;
 }

 bpos = read_initfile_word(initfile_word, buffer, buffer_length, bpos);

 if (bpos == -1)
  return -1;

 int read_number = 0;
 int invalid_value_fixed = 0;

 if (strcmp(initfile_word, "display_w") == 0)
 {
  bpos = read_initfile_number(&read_number, buffer, buffer_length, bpos);
  if (bpos == -1)
   return -1;
  settings.option [OPTION_WINDOW_W] = read_number;
  if (settings.option [OPTION_WINDOW_W] < 1024)
		{
   settings.option [OPTION_WINDOW_W] = 1024;
   invalid_value_fixed = 1;
		}
  if (settings.option [OPTION_WINDOW_W] > 1920)
		{
   settings.option [OPTION_WINDOW_W] = 1920;
   invalid_value_fixed = 1;
		}

		if (invalid_value_fixed != 0)
   fprintf(stdout, "\nDisplay width (%i) fixed to %i.", read_number, settings.option [OPTION_WINDOW_W]);

  return bpos;
 }

 invalid_value_fixed = 0;

 if (strcmp(initfile_word, "display_h") == 0)
 {
  bpos = read_initfile_number(&read_number, buffer, buffer_length, bpos);
  if (bpos == -1)
   return -1;
  settings.option [OPTION_WINDOW_H] = read_number;
  if (settings.option [OPTION_WINDOW_H] < 768)
		{
   settings.option [OPTION_WINDOW_H] = 768;
   invalid_value_fixed = 1;
		}
  if (settings.option [OPTION_WINDOW_H] > 1200)
		{
   settings.option [OPTION_WINDOW_H] = 1200;
   invalid_value_fixed = 1;
		}
		if (invalid_value_fixed != 0)
   fprintf(stdout, "\nDisplay height (%i) fixed to %i.", read_number, settings.option [OPTION_WINDOW_H]);
  return bpos;
 }

 if (strcmp(initfile_word, "debug") == 0)
 {
  bpos = read_initfile_number(&read_number, buffer, buffer_length, bpos);
  if (bpos == -1)
   return -1;
  settings.option [OPTION_DEBUG] = read_number;
  return bpos;
 }



 if (strcmp(initfile_word, "fullscreen") == 0)
 {
  settings.option [OPTION_FULLSCREEN] = 1;
  return bpos;
 }

 if (strcmp(initfile_word, "special_cursor") == 0)
 {
  settings.option [OPTION_SPECIAL_CURSOR] = 1;
  return bpos;
 }

 invalid_value_fixed = 0;

 if (strcmp(initfile_word, "vol_music") == 0)
 {
  bpos = read_initfile_number(&read_number, buffer, buffer_length, bpos);
  if (bpos == -1)
   return -1;
  settings.option [OPTION_VOL_MUSIC] = read_number;
  if (settings.option [OPTION_VOL_MUSIC] < 0)
		{
   settings.option [OPTION_VOL_MUSIC] = 0;
   invalid_value_fixed = 1;
		}
  if (settings.option [OPTION_VOL_MUSIC] > 100)
		{
   settings.option [OPTION_VOL_MUSIC] = 100;
   invalid_value_fixed = 1;
		}
		if (invalid_value_fixed != 0)
   fprintf(stdout, "\nMusic volume (%i) fixed to %i.", read_number, settings.option [OPTION_VOL_MUSIC]);
  return bpos;
 }

 invalid_value_fixed = 0;

 if (strcmp(initfile_word, "vol_effect") == 0)
 {
  bpos = read_initfile_number(&read_number, buffer, buffer_length, bpos);
  if (bpos == -1)
   return -1;
  settings.option [OPTION_VOL_EFFECT] = read_number;
  if (settings.option [OPTION_VOL_EFFECT] < 0)
		{
   settings.option [OPTION_VOL_EFFECT] = 0;
   invalid_value_fixed = 1;
		}
  if (settings.option [OPTION_VOL_EFFECT] > 100)
		{
   settings.option [OPTION_VOL_EFFECT] = 100;
   invalid_value_fixed = 1;
		}
		if (invalid_value_fixed != 0)
   fprintf(stdout, "\nEffects volume (%i) fixed to %i.", read_number, settings.option [OPTION_VOL_EFFECT]);
  return bpos;
 }

 if (settings.option [OPTION_VOL_MUSIC] == 0
  && settings.option [OPTION_VOL_EFFECT] == 0)
   settings.sound_on = 0;

// unrecognised word, so just read to the end of the line:

  bpos = read_initfile_line(buffer, buffer_length, bpos);

  return bpos;
}


char check_initfile_char(char* buffer, int buffer_length, int bpos)
{

 if (bpos >= buffer_length)
  return 0;

 return buffer [bpos];

}

// returns new value of bpos on success, or -1 if EOL or EOF found
int read_initfile_word(char* ifword, char* buffer, int buffer_length, int bpos)
{

 int word_pos = 0;

 while(TRUE)
 {
  if (bpos >= buffer_length)
   return -1;
  if (buffer [bpos] == '\0'
   || buffer [bpos] == '\n'
   || buffer [bpos] == '\r'
   || buffer [bpos] == ' '
   || word_pos >= INITFILE_WORD_LENGTH - 1)
  {
   ifword [word_pos] = '\0';
   bpos++;
   return bpos;
  }
  ifword [word_pos] = buffer [bpos];
  word_pos++;
  bpos++;
 };

 return bpos;

}



// returns new value of bpos on success, or -1 if EOL or EOF found
int read_initfile_number(int* read_number, char* buffer, int buffer_length, int bpos)
{
#define INITFILE_NUMBER_LENGTH 6
 int number_pos = 0;
 int number_string [INITFILE_NUMBER_LENGTH];
 *read_number = 0;

 while(TRUE)
 {
  if (bpos >= buffer_length)
   return -1;
  if (buffer [bpos] < '0'
   || buffer [bpos] > '9'
   || number_pos >= INITFILE_NUMBER_LENGTH - 1)
  {
   bpos++;
   break;
  }
  number_string [number_pos] = buffer [bpos] - '0';
  number_pos++;
  bpos++;
 };

 int multiplier = 1;

 while(number_pos > 0)
 {
  number_pos --;
  *read_number += number_string [number_pos] * multiplier;
  multiplier *= 10;
 };

 return bpos;

}

int finish_initfile_line(char* buffer, int buffer_length, int bpos)
{

 while(TRUE)
 {
  if (bpos >= buffer_length + 1)
   return -1;
  if (buffer [bpos] == '\n'
   || buffer [bpos] == '\r')
   return bpos + 1;
  bpos ++;
 };

 return bpos;

}

