
/*
 *
 *     _______       ___       ____      __       _______
 *    /\  ____\    /|   \     /  __\    /\ \     /\  ____\
 *    \ \ \___/_   ||  _ \   |  /__/____\ \ \    \ \ \___/_
 *     \ \  ____\  || |_\ \  |\ \ /\_  _\\ \ \    \ \  ____\
 *      \ \ \___/_ ||  ___ \ \ \ \\//\ \/ \ \ \____\ \ \___/_
 *       \ \______\||_|__/\_\ \ \ \_\/ |   \ \_____\\ \______\
 *        \/______/|/_/  \/_/  \_\_____/    \/_____/ \/______/
 *
 *
 *    EAGLE
 *    Edgar's Agile Gui Library and Extensions
 *
 *    Copyright 2009-2013+ by Edgar Reynaldo
 *
 *    See EagleLicense.txt for allowed uses of this library.
 *
 */



#ifndef EagleGuiWidgetBase_HPP
#define EagleGuiWidgetBase_HPP


#include "Eagle/Area.hpp"
#include "Eagle/Events.hpp"
#include "Eagle/Object.hpp"
#include "Eagle/GraphicsContext.hpp"
#include "Eagle/Image.hpp"

#include "Eagle/Gui/WidgetArea.hpp"
#include "Eagle/Gui/WidgetColorset.hpp"
#include "Eagle/Gui/WidgetMessage.hpp"



extern const unsigned int TOPIC_DIALOG;

/** Bitfield values for messages returned to a dialog from the Update() and CheckInput() functions. */
enum DIALOG_MSGS {
   DIALOG_OKAY       = 0x00,// Also used to signify that there are no messages for the WidgetHandler::TakeNextMessage function.
   DIALOG_CLOSE      = 0x01,
   DIALOG_REDRAW_ALL = 0x02,/// For when the background needs to be cleared / redrawn
   DIALOG_INPUT_USED = 0x04,/// Play nice, a WidgetHandler will stop checking other widgets input after this is received.
   DIALOG_TAKE_FOCUS = 0x08,
   DIALOG_DISABLED   = 0x10,// Returned when you try to update or check the inputs of a disabled dialog.
   DIALOG_REMOVE_ME  = 0x20
};

enum WIDGET_FLAGS {
   FLAGS_NONE      = 0x000,
   ENABLED         = 0x001,   // Currently active or not
   VISIBLE         = 0x002,   // Whether anything is drawn
   HOVER           = 0x004,   // If it has mouse hover - don't steal the focus for hover!
   HASFOCUS        = 0x008,   // If it has the focus
   MOVEABLE        = 0x010,   // Whether it can be moved
   RESIZEABLE      = 0x020,   // Whether it can be resized
   NEEDS_REDRAW    = 0x040,   // Whether it should be redrawn before display?
   NEEDS_BG_REDRAW = 0x080,   // Whether it needs its background redrawn before being redrawn
   ALLOW_CLOSE     = 0x100,   // If it is allowed to close the dialog based on input
   ALLOW_OVERLAP   = 0x200,   // TODO : DEPRECATED - REMOVE - If it is allowed to overlap other widgets
   USER_FLAGS_START= 0x400
};

#define NUM_WIDGET_FLAGS 10 // ENABLED thru ALLOW_OVERLAP

typedef unsigned int UINT;

std::string PrintFlags(UINT flags);

extern const UINT BASIC_FLAGS;
extern const UINT DEFAULT_FLAGS;
extern const UINT DEFAULT_OVERLAP_FLAGS;


/// Widgets with higher display priority will display on top of widgets with lower display priority
/// When widgets have equal priority, the widget that last had the focus will display on top
enum WIDGET_DISPLAY_PRIORITY {
   LOWEST_DISPLAY_PRIORITY = 0,
   LOW_DISPLAY_PRIORITY = 64,
   MID_DISPLAY_PRIORITY = 128,
   HIGH_DISPLAY_PRIORITY = 192,
   HIGHEST_DISPLAY_PRIORITY = 255
};


class Layout;
class WidgetHandler;



class WidgetBase : public EagleObject , public EagleEventSource {

protected :
   
   WidgetBase* wparent;
   Layout* layout;
   
   WidgetColorset wcols;
   WidgetColorset privwcols;
   bool use_private_colorset;

   WidgetArea area;
   int minw;
   int minh;
   
   UINT flags;
   
   int display_priority;

   virtual void PrivateRaiseEvent(WidgetMsg msg) {(void)msg;}

   void RaiseEvent(WidgetMsg msg);
   
   EagleEvent MakeEagleEvent(WidgetMsg msg);

public :
   
   WidgetBase();
   WidgetBase(std::string name);
   virtual ~WidgetBase() {}

/*   
   virtual void EagleEventSource::ReadEvents()=0;
   virtual void EagleEventListener::ProcessEvent(EagleEvent e)=0;
*/
	/// EventSource
   void ReadEvents() {}
   
   /// EventListener
//   void RespondToEvent(EagleEvent e);


   /// Widgetbase virtual functions. Overload these to give your widgets the proper behaviour
//   virtual int Update(double tsec);
//private :
//   virtual int PrivateHandleEvent(EagleEvent e);// Only handle events not handled in CheckInputs here
//public :
//   virtual int CheckInputs();// Called automatically by HandleEvent and by system during gui update.
//   virtual void Display(EagleGraphicsContext* win , int xpos , int ypos)=0;
   
/**
protected :
   virtual int PrivateHandleEvent(EagleEvent e);
   virtual int CheckInputs();
public :
   virtual void Display(EagleGraphicsContext* win , int xpos , int ypos)=0;
   virtual int Update(double tsec);
//*/
protected :
   /// Handle events here
   virtual int PrivateHandleEvent(EagleEvent e);// Only handle events not handled in CheckInputs here

   virtual void PrivateDisplay(EagleGraphicsContext* win , int xpos , int ypos);

   /// Only handle events not handled in PrivateHandleEvent here in CheckInputs, like state checks.
   virtual int CheckInputs();// Called automatically by HandleEvent
                             // ?and by system during gui update.
                             // Overload to check state machine, will be called once per system event if not
                             // handled directly by PrivateHandleEvent already as signaled by using
                             // the DIALOG_INPUT_USED flag
public :

   int HandleEvent(EagleEvent e);

   void Display(EagleGraphicsContext* win , int xpos , int ypos);// Drawing target will already be set for you
                                                                           // But you can push and pop other drawing targets as well

   virtual int Update(double tsec);



   virtual void QueueUserMessage(const WidgetMsg& wmsg);// For the top level WidgetHandlers to store messages to the user from 
                                                        // the widgets it is handling. Messages are passed up the chain using
                                                        // the parent pointer.
                                                        /// Warning - if the widget has no parent, these messages just get lost,
                                                        /// and only in the case that the top level parent widget is a 
                                                        /// WidgetHandler do they get spooled for the user.
   
   void QueueUserMessage(WidgetBase* widget_address , UINT widget_topic , int messages);// Shortcut to virtual Q.U.M.

	/// Setters
	virtual void SetParent(WidgetBase* parent);
	void         SetOwnerLayout(Layout* l);
   
   /// These only set the colorset, they don't determine which you are using, and public colors are
   /// set by default - call UseColorset if you wanna change the setting
   virtual void SetColorset(const WidgetColorset& colors , bool set_descendants_colors = false);
   virtual void SetPrivateColorset(const WidgetColorset& colors);

   virtual void UseColorset(bool use_public_colorset);// false means use private colorset

   /// Virtual Setters, override them if you need to
   virtual void SetFlagStates(UINT FLAGS , bool state);/// (Which flags set to which state)
   
   /// Default behaviour for state setters          Behaviour regarding redraw state
   virtual void SetEnabledState      (bool state);// Sets bg redraw flag
   virtual void SetVisibilityState   (bool state);// Sets bg redraw flag
   virtual void SetHoverState        (bool state);// /* Does not set any redraw flag */ - OLD:Sets redraw flag and (false) sets bg redraw flag
   virtual void SetFocusState        (bool state);// Sets redraw flag and (false) sets bg redraw flag
   virtual void SetMoveableState     (bool state);// Does not set redraw flag
   virtual void SetResizeableState   (bool state);// Does not set redraw flag
   virtual void SetNeedsRedrawState  (bool state);
   virtual void SetNeedsBgRedrawState(bool state);// (true) sets redraw flag
   virtual void SetAllowCloseState   (bool state);// Does not set redraw flag
   virtual void SetAllowOverlapState (bool state);// Does not set redraw flag

   virtual void SetRedrawFlag();// Shortcut to SetNeedsRedrawState(true)
   virtual void SetBgRedrawFlag();// Shortcut to SetNeedsBgRedrawState(true)
   virtual void ClearRedrawFlag();// Shortcut to SetNeedsRedrawState(false) and SetNeedsBgRedrawState(false)

   virtual void SetRedrawAllFlag();// To tell parent widget handlers to redraw all widgets

   virtual void ShowWidget();// Makes the widget enabled and visible
   virtual void HideWidget();// Makes the widget disabled and invisible
   virtual void ToggleWidgetVisibility();// Toggles the enabled and visible state of the widget

   virtual bool AcceptsFocus() {return true;}
   virtual bool IsMouseOver(int msx , int msy) const;
   
   /// For widgets to ask their parent widget handlers to give them the focus
///   virtual bool GiveWidgetFocus(WidgetBase* widget);
   virtual bool GiveWidgetFocus(WidgetBase* widget , bool notify_parent = true);

//   virtual void SetFont(FONT* textfont) {}

   virtual void SetDrawPos(int xpos , int ypos);
   virtual void SetDrawDimensions(int width , int height);
   virtual void SetArea(int xpos , int ypos , int width , int height);
   void         SetArea(const Rectangle& r);

   void SetMinInnerWidth(int w);
   void SetMinInnerHeight(int h);
   void SetMinInnerDimensions(int w , int h);

   void SetDisplayPriority(int priority);

   virtual int AbsMinWidth() {return 1;}
   virtual int AbsMinHeight() {return 1;}
   
   virtual bool HasGui() {return false;}
   virtual WidgetHandler* GetGui() {return 0;}
   
   WidgetHandler* NearestParentGui();
   WidgetBase*    Root();
   WidgetHandler* RootGui();

   /// Getters , some references for direct modification
   int                   AbsParentX() const ;
   int                   AbsParentY() const ;
   Pos2d                 GetParentOffset() const ;
   
   WidgetBase*           Parent()          const {return wparent;}
   WidgetColorset&       WCols();// SetRedrawFlag if you change the colors!
   virtual Rectangle     Area()            const {return area.OuterArea();}// kept for compatibility reasons
   virtual Rectangle     OuterArea()       const {return area.OuterArea();}
   virtual Rectangle     InnerArea()       const {return area.InnerArea();}
   int                   MinWidth()        const {return minw + area.MLeft() + area.MRight();}
   int                   MinHeight()       const {return minh + area.MTop() + area.MBot();}
   int                   MinInnerWidth()   const {return minw;}
   int                   MinInnerHeight()  const {return minh;}
   UINT                  Flags()           const {return flags;}
   int                   DisplayPriority() const {return display_priority;}
   
   virtual ostream& DescribeTo(ostream& os , Indenter indent = Indenter()) const;
};




bool DrawPriorityIsLess(WidgetBase* lhs , WidgetBase* rhs);




#endif // EagleGuiWidgetBase_HPP

