
/*
 *
 *     _______       ___       ____      __       _______
 *    /\  ____\    /|   \     /  __\    /\ \     /\  ____\
 *    \ \ \___/_   ||  _ \   |  /__/____\ \ \    \ \ \___/_
 *     \ \  ____\  || |_\ \  |\ \ /\_  _\\ \ \    \ \  ____\
 *      \ \ \___/_ ||  ___ \ \ \ \\//\ \/ \ \ \____\ \ \___/_
 *       \ \______\||_|__/\_\ \ \ \_\/ |   \ \_____\\ \______\
 *        \/______/|/_/  \/_/  \_\_____/    \/_____/ \/______/
 *
 *
 *    EAGLE
 *    Edgar's Agile Gui Library and Extensions
 *
 *    Copyright 2009-2013+ by Edgar Reynaldo
 *
 *    See EagleLicense.txt for allowed uses of this library.
 *
 */



#include "Eagle/Gui/Layout/Layout.hpp"
#include "Eagle/Gui/WidgetHandler.hpp"



int Layout::WidgetIndex(WidgetBase* widget) {
   if (!widget) {return -1;}
   for (unsigned int i = 0 ; i < wchildren.size() ; ++i) {
      if (widget == wchildren[i]) {
         return (int)i;
      }
   }
   return -1;// use -1 to signify not in children
}



int Layout::NextFreeSlot() {
   for (int i = 0 ; i < (int)wchildren.size() ; ++i) {
      if (wchildren[i] == 0) {
         return i;
      }
   }
   return -1;
}



void Layout::ReserveSlots(int nslots) {
   if (nslots == (int)wchildren.size()) {return;}
   
   if (nslots < 1) {
      nslots = 0;// don't crash resize below
      ClearLayout();
   }
   if (nslots < (int)wchildren.size()) {
      for (unsigned int i = nslots ; i < wchildren.size() ; ++i) {
         EmptySlot(i);
      }
   }
   wchildren.resize(nslots , 0);
}



void Layout::RemoveWidgetFromLayout(WidgetBase* widget) {
   if (!widget) {return;}
   
   std::map<WidgetBase* , bool>::iterator it = delete_map.find(widget);
//   EAGLE_ASSERT(!(it == delete_map.end()));
   EAGLE_ASSERT(it != delete_map.end());

   if (whandler) {
      whandler->StopTrackingWidget(widget);
      widget->SetOwnerLayout(0);
   }

   if (it->second) {
      delete it->first;
   }
   delete_map.erase(it);
   
}



void Layout::ReplaceWidget(WidgetBase* widget , int slot) {
   RemoveWidgetFromLayout(wchildren[slot]);
   wchildren[slot] = widget;
   RepositionChild(slot);

   if (whandler && widget) {
      whandler->TrackWidget(widget);
      widget->SetOwnerLayout(this);
   }
}



Layout::Layout() :
      WidgetBase(),
      attributes(LAYOUT_ALLOWS_RESIZE_AND_REPOSITION),
      halign(HALIGN_LEFT),
      valign(VALIGN_TOP),
      size_fixed(false),
      wchildren(),
      delete_map(),
      whandler(0)
{
   SetName(StringPrintF("Layout object at %p" , this));
   SetDisplayPriority(LOW_DISPLAY_PRIORITY);
}



void Layout::SetDrawPos(int xpos , int ypos) {
   WidgetBase::SetDrawPos(xpos,ypos);
   RepositionAllChildren();
}



void Layout::SetDrawDimensions(int width , int height) {
   WidgetBase::SetDrawDimensions(width,height);
   RepositionAllChildren();
}



void Layout::SetArea(int xpos , int ypos , int width , int height) {
   WidgetBase::SetArea(xpos,ypos,width,height);
   RepositionAllChildren();
}




/**
void Layout::TakeOverLayoutFrom(Layout* l) {
   ClearLayout();
	if (l) {
		wchildren = l->wchildren;
		l->ClearLayout();
		for (unsigned int i = 0 ; i < wchildren.size() ; ++i) {
			WidgetBase* w = wchildren[i];
			if (w) {
            w->SetLayout(this);
            if (whandler) {
               whandler->TrackWidget(w);
            }
			}
		}
		RepositionAllChildren();
	}
}
//*/


Rectangle Layout::RequestPosition(WidgetBase* widget , int newx , int newy) {
   EAGLE_ASSERT(widget);
   EAGLE_ASSERT(WidgetIndex(widget) != -1);
   
   
   Rectangle warea = widget->OuterArea();
   
   unsigned int wflags = widget->Flags();
   
   if (attributes & LAYOUT_ALLOWS_REPOSITION) {
      if (wflags && MOVEABLE) {
         warea.SetPos(newx , newy);
      }
   }
   
   return warea;
}



Rectangle Layout::RequestSize(WidgetBase* widget , int newwidth , int newheight) {
   EAGLE_ASSERT(widget);
   EAGLE_ASSERT(WidgetIndex(widget) != -1);
   EAGLE_ASSERT(newwidth > 0);
   EAGLE_ASSERT(newheight > 0);
   
   Rectangle warea = widget->OuterArea();
   
   int minwidth = widget->MinWidth();
   int minheight = widget->MinHeight();
   
   if (newwidth < minwidth) {
      newwidth = minwidth;
   }
   if (newheight < minheight) {
      newheight = minheight;
   }
   
   if (attributes & LAYOUT_ALLOWS_RESIZE) {
      warea.SetDimensions(newwidth , newheight);
   }
   
   return warea;
}



Rectangle Layout::RequestArea(WidgetBase* widget , int newx , int newy , int newwidth , int newheight) {
   EAGLE_ASSERT(widget);
   EAGLE_ASSERT(WidgetIndex(widget) != -1);
   EAGLE_ASSERT(newwidth > 0);
   EAGLE_ASSERT(newheight > 0);
   
   Rectangle warea = widget->OuterArea();
   
   int minwidth = widget->MinWidth();
   int minheight = widget->MinHeight();
   unsigned int wflags = widget->Flags();

   if (newwidth < minwidth) {
      newwidth = minwidth;
   }
   if (newheight < minheight) {
      newheight = minheight;
   }
   
   if (attributes & LAYOUT_ALLOWS_RESIZE_AND_REPOSITION) {
      if ((wflags & MOVEABLE) && (wflags & RESIZEABLE)) {
         warea.SetArea(newx , newy , newwidth , newheight);
      }
      else if (wflags & MOVEABLE) {
         warea.SetPos(newx , newy);
      }
      else if (wflags & RESIZEABLE) {
         warea.SetDimensions(newwidth , newheight);
      }
   }
   else if (attributes & LAYOUT_ALLOWS_RESIZE) {
      if (wflags & RESIZEABLE) {
         warea.SetDimensions(newwidth , newheight);
      }
   }
   else if (attributes & LAYOUT_ALLOWS_REPOSITION) {
      if (wflags & MOVEABLE) {
         warea.SetPos(newx , newy);
      }
   }
   return warea;
}



Rectangle Layout::RequestArea(WidgetBase* widget , Rectangle newarea) {
   return RequestArea(widget , newarea.X() , newarea.Y() , newarea.W() , newarea.H());
}



bool Layout::PlaceWidget(WidgetBase* widget , int slot , bool delete_when_removed) {
   if ((slot < 0) || (slot >= (int)wchildren.size())) {
      return false;
   }

   if (widget) {
      delete_map[widget] = delete_when_removed;
   }

   ReplaceWidget(widget , slot);
   
   return true;
}



bool Layout::AddWidget(WidgetBase* widget , bool delete_when_removed) {
   int slot = NextFreeSlot();
   if (slot == -1) {
      if (!size_fixed) {
         ReserveSlots((int)wchildren.size() + 1);
         slot = (int)wchildren.size() - 1;
      }
      else {
         return false;
      }
   }
   if (widget) {
      delete_map[widget] = delete_when_removed;
   }

   ReplaceWidget(widget , slot);
   
   return true;
}



void Layout::EmptySlot(int slot) {
   PlaceWidget(0 , slot , false);
}



void Layout::RemoveWidget(WidgetBase* widget) {
   EAGLE_ASSERT(widget);
   PlaceWidget(0 , WidgetIndex(widget) , false);
}



void Layout::ClearLayout() {
   std::vector<WidgetBase*> children = WChildren();
   for (unsigned int i = 0 ; i < children.size() ; ++i) {
      WidgetBase* widget = children[i];
      
      if (whandler) {
         whandler->StopTrackingWidget(children[i]);
      }
      
      widget->SetOwnerLayout(0);
      
      std::map<WidgetBase* , bool>::iterator it = delete_map.find(widget);
      EAGLE_ASSERT(it != delete_map.end());// these should all be on the map
      if (it->second) {
         delete widget;
      }
   }
   wchildren.clear();
   delete_map.clear();
}



void Layout::SetAlignment(LAYOUT_HALIGN h_align , LAYOUT_VALIGN v_align) {
	halign = h_align;
	valign = v_align;
	RepositionAllChildren();
}



void Layout::SetGuiHandler(WidgetHandler* handler) {
   whandler = handler;
}



std::vector<WidgetBase*> Layout::WChildren() {
   
   std::vector<WidgetBase*> children;
   children.reserve(wchildren.size());
   for (unsigned int i = 0 ; i < wchildren.size() ; ++i) {
      if (wchildren[i]) {
         children.push_back(wchildren[i]);
      }
   }
   return children;
}



std::vector<WidgetBase*> Layout::Descendants() {
	std::vector<WidgetBase*> descendants;
	for (unsigned int i = 0 ; i < wchildren.size() ; ++i) {
		WidgetBase* widget = wchildren[i];
		if (!widget) {continue;}
		descendants.push_back(widget);
		Layout* l = dynamic_cast<Layout*>(widget);
		if (l) {
			std::vector<WidgetBase*> grandchildren = l->Descendants();
			for (unsigned int j = 0 ; j < grandchildren.size() ; ++j) {
				descendants.push_back(grandchildren[j]);
			}
			
		}
	}
	return descendants;
}



Layout* Layout::RootLayout() {
   if (layout) {
      return layout->RootLayout();
   }
   return this;
}



bool Layout::IsRootLayout() {
   return layout == 0;
}



WidgetHandler* Layout::WHandler() {
   return whandler;
}




