package it.fmc.santahack2012.gui;

import it.fmc.santahack2012.events.FogOfWar;
import it.fmc.santahack2012.events.Interactable;
import it.fmc.santahack2012.events.Selectable;
import it.fmc.santahack2012.events.SimpleChar;
import it.fmc.santahack2012.gfx.GraphicsHandler;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Image;
import java.util.ArrayList;
import java.util.List;

import map.GameMap;
import map.Gfx;
import map.Tile;

public class Camera {
	private static final int panAmount = 10;
	private int screenW, screenH;
	private int showX, showY;
	private final int maxX, maxY;
	private final GameMap gmap;
	private final GraphicsHandler gfxH;
	private final int maxC;
	private final int maxR;

	public Camera(GameMap gmap, GraphicsHandler gfxH) {
		this.gfxH = gfxH;
		this.gmap = gmap;
		Dimension d = gmap.getDimensions();
		maxX = d.width;
		maxY = d.height;
		maxC = gmap.getTilesX();
		maxR = gmap.getTilesY();
		screenW = 0;
		screenH = 0;
		showX = showY = 0;
	}

	public void render(Graphics2D g, List<Interactable> inScreenActors, FogOfWar fogOfWar) {
		g.clearRect(0, 0, screenW, screenH);
		final int startX = getStartingColumn();
		final int startY = getStartingRow();
		final int endX = getEndColumn();
		final int endY = getEndRow();
		final int tSize = Gfx.TILE_SIZE;
		Tile[][] tilemap = gmap.getTileMap();
		for (int y = startY; y < endY; y++)
			for (int x = startX; x < endX; x++) {
				Image im = gfxH.getTile(tilemap[y][x].getTileID());
				g.drawImage(im, (x * tSize) - showX, (y * tSize) - showY, null);
			}
		// actors
		for (Interactable a : inScreenActors) {
			int x = a.getX() - showX;
			int y = a.getY() - showY;
			int w = a.getW();
			g.drawImage(a.getImage(), x, y, null);
			if (a.isSelectable() && ((Selectable) a).isSelected()) {
				g.setColor(Color.CYAN);
				g.drawRect(x, y, w, a.getH());
			}
			if (a.isSimpleChar() && ((SimpleChar) a).isAlive()) {
				SimpleChar sc = (SimpleChar) a;
				int maxLife = sc.getMaxLife();
				int curLife = sc.getCurrLife();
				g.setColor(new Color(157, 0, 0));
				g.fillRect(x, y, w, 8);
				g.setColor(new Color(255, 0, 0));
				w = (int) ((w - 2) * (float) curLife / maxLife);
				g.fillRect(x + 1, y + 1, w, 6);
			}
			// draw lifebar
		}

		// fogOfWar
		boolean grid[][] = fogOfWar.getFogGrid();
		g.setColor(Color.black);
		for (int y = startY; y < endY; y++)
			for (int x = startX; x < endX; x++) {
				if (grid[y][x]) g.fillRect((x * tSize) - showX, (y * tSize) - showY, tSize, tSize);
			}
	}

	private int getEndColumn() {
		return Math.min((int) Math.ceil((showX + screenW) / (float) Gfx.TILE_SIZE) + 1, maxC);
	}

	private int getEndRow() {
		return Math.min((int) Math.ceil((showY + screenH) / (float) Gfx.TILE_SIZE) + 1, maxR);
	}

	private int getStartingColumn() {
		int c = 0;
		c = showX / Gfx.TILE_SIZE;
		return c;
	}

	private int getStartingRow() {
		int r = 0;
		r = showY / Gfx.TILE_SIZE;
		return r;
	}

	public void setScreenBounds(int w, int h) {
		screenW = w;
		screenH = h;
		checkBounds();
	}

	private void checkBounds() {
		showX = Math.min(showX, maxX - screenW);
		showX = Math.max(showX, 0);
		showY = Math.min(showY, maxY - screenH);
		showY = Math.max(showY, 0);
	}

	public void panRight() {
		showX += panAmount;
		checkBounds();
	}

	public void panLeft() {
		showX -= panAmount;
		checkBounds();
	}

	public void panUp() {
		showY -= panAmount;
		checkBounds();
	}

	public void panDown() {
		showY += panAmount;
		checkBounds();
	}

	public List<Interactable> getActorsInScreen(List<Interactable> actors) {
		List<Interactable> inScreen = new ArrayList<Interactable>();
		for (Interactable a : actors) {
			int sX = a.getX();
			int sY = a.getY();
			int eX = sX + a.getW();
			int eY = sY + a.getH();
			if (eX > showX && sX < (showX + screenW)) if (eY > showY && sY < (showY + screenH)) inScreen.add(a);
		}
		return inScreen;
	}

	public int getShowX() {
		return showX;
	}

	public int getShowY() {
		return showY;
	}

}
