package it.fmc.santahack2012.astar;

import it.fmc.santahack2012.events.DIR;
import it.fmc.santahack2012.events.Interactable;
import it.fmc.santahack2012.events.SimpleAction;
import it.fmc.santahack2012.events.SimpleDirection;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;

import map.GameMap;
import map.Gfx;
import map.Tile;

public class PathFinder {
	private final int grid[][];
	private final int w;
	private final int h;
	private final int tileSize;

	@SuppressWarnings("boxing")
	public PathFinder(GameMap gmap, List<Integer> solidTiles) {
		int pFact = 2; // precision factor
		w = gmap.getTilesX() * pFact;
		h = gmap.getTilesY() * pFact;
		Tile[][] tiles = gmap.getTileMap();
		grid = new int[h][w];
		tileSize = Gfx.TILE_SIZE / pFact;
		for (int y = 0; y < h; y++)
			for (int x = 0; x < w; x++) {
				int val = 0;
				int tileId = tiles[Math.round(y / pFact)][Math.round(x / pFact)].getTileID();
				if (solidTiles.contains(tileId)) val = 1;
				grid[y][x] = val;
			}
	}

	public Queue<SimpleAction> getDirectionsToPoint(int sx, int sy, int ex, int ey) {
		int stx = sx / tileSize;
		int sty = sy / tileSize;
		int etx = ex / tileSize;
		int ety = ey / tileSize;
		Position sol;
		if (grid[ety][etx] == 1)
			sol = null;
		else
			sol = astar(stx, sty, etx, ety);
		return positionsToDirection(sol);
	}

	private Queue<SimpleAction> positionsToDirection(Position sol) {
		Queue<SimpleAction> solution = new LinkedList<SimpleAction>();
		if (sol == null) return solution;
		Position prevPos = sol.getParent();
		Position curPos = sol;
		while (prevPos != null) {
			SimpleDirection dir = null;
			int px = prevPos.getX();
			int py = prevPos.getY();
			int cx = curPos.getX();
			int cy = curPos.getY();
			if (px == cx) {
				if (py < cy)
					dir = new SimpleDirection(DIR.S, tileSize);
				else
					dir = new SimpleDirection(DIR.N, tileSize);
			} else if (px < cx) {
				if (py < cy)
					dir = new SimpleDirection(DIR.SE, tileSize);
				else if (py > cy)
					dir = new SimpleDirection(DIR.NE, tileSize);
				else
					dir = new SimpleDirection(DIR.E, tileSize);
			} else if (px > cx) {
				if (py < cy)
					dir = new SimpleDirection(DIR.SW, tileSize);
				else if (py > cy)
					dir = new SimpleDirection(DIR.NW, tileSize);
				else
					dir = new SimpleDirection(DIR.W, tileSize);
			}
			if (dir == null) throw new RuntimeException("Simple direction shouldn't be null");
			solution.add(dir);
			curPos = prevPos;
			prevPos = curPos.getParent();
		}
		Collections.reverse((LinkedList<SimpleAction>) solution);
		return solution;
	}

	public Position astar(int sx, int sy, int ex, int ey) {
		Position start = new Position(sx, sy, 0, 0, null);
		List<Position> openList = new ArrayList<Position>();
		List<Position> closedList = new ArrayList<Position>();
		openList.add(start);// start point

		Position solution = null;
		while (!openList.isEmpty()) {
			Position current = getLowestF(openList);
			if (current.getX() == ex && current.getY() == ey) {
				solution = current;
				break;
			}
			openList.remove(current);
			closedList.add(current);
			addNeighbours(current, grid, ex, ey, closedList, openList);
		}
		return solution;
	}

	private Position getLowestF(List<Position> openList) {
		if (openList.isEmpty()) return null;
		Position l = openList.get(0);
		for (Position p : openList) {
			if (l.getF() > p.getF()) l = p;
		}
		return l;
	}

	private void addNeighbours(Position parent, int[][] mapGrid, int eX, int eY, List<Position> closedList,
			List<Position> openList) {
		int Y = parent.getY();
		int X = parent.getX();
		for (int y = Y - 1; y <= Y + 1; y++)
			for (int x = X - 1; x <= X + 1; x++) {
				if (x >= 0 && x < w && y >= 0 && y < h) {
					if (x == X && y == Y) continue; // parent square
					if (mapGrid[y][x] == 1) continue; // blocking
					// set G
					int G = 10;
					if (y != Y && x != X) G = 14;
					G = G + parent.getG();
					// set H
					int H = (Math.abs(eX - x) + Math.abs(eY - y)) * 10;
					Position p = new Position(x, y, H, G, parent);
					if (!closedList.contains(p)) {
						int index = openList.indexOf(p);
						if (index == -1)
							openList.add(p);
						else {
							Position old = openList.get(index);
							if (old.getG() > p.getG()) {// this parent is better
								old.setParent(parent);
								old.recalculate();
							}
						}
					}
				}
			}
	}

	public void addBuilding(List<Interactable> buildings) {
		for (Interactable a : buildings) {
			final int x1 = Math.round(a.getX() / (float) tileSize);
			final int y1 = Math.round(a.getY() / (float) tileSize);
			final int x2 = (int) (x1 + Math.floor(a.getW() / (float) tileSize));
			final int y2 = (int) (y1 + Math.floor(a.getH() / (float) tileSize));
			for (int x = x1; x <= x2; x++)
				for (int y = y1; y <= y2; y++)
					grid[y - 1][x] = 1;

		}
	}
}
