/*
 *
 *     _______       ___       ____      __       _______ 
 *    /\  ____\    /|   \     /  __\    /\ \     /\  ____\ 
 *    \ \ \___/_   ||  _ \   |  /__/____\ \ \    \ \ \___/_ 
 *     \ \  ____\  || |_\ \  |\ \ /\_  _\\ \ \    \ \  ____\ 
 *      \ \ \___/_ ||  ___ \ \ \ \\//\ \/ \ \ \____\ \ \___/_ 
 *       \ \______\||_|__/\_\ \ \ \_\/ |   \ \_____\\ \______\
 *        \/______/|/_/  \/_/  \_\_____/    \/_____/ \/______/
 *
 *
 *    EAGLE
 *    Edgar's Allegro Gui Library and Extensions
 *
 *    Copyright 2009-2011 by Edgar Reynaldo
 *
 *    See EagleLicense.txt for allowed uses of this library.
 *
 */



#include <allegro.h>



enum ANIMATION_TYPE {
   FORWARD_ONCE            = 0,
   BACKWARD_ONCE           = 1,
   FORWARD_AND_BACK_ONCE   = 2,
   FORWARD_REPEAT          = 3,
   BACKWARD_REPEAT         = 4,
   FORWARD_AND_BACK_REPEAT = 5
};



class AnimationBase {
protected :
   BITMAP** frames;
   bool* free_frame;
   int* xposition;
   int* yposition;
   double* rotation;// degrees clockwise from zero pointing right
   double* scale;
   int* color1;
   int* color2;
   
   int base_x;
   int base_y;
   
   ANIMATION_TYPE anim_type;
   double duration;
   double frames_per_sec;
   double frametime;
   double time_factor;
   int num_frames;
   int frame_num;
   int loop_num;
   int loop_stop_num;
   
   int num_trails;
   int frames_per_trail;
   bool paused;
   
   void SetColorHSV(int* color_array , double hue , double huespeed , double sat , double satspeed,
                     double val , double valspeed , int frame_start , int frame_stop);
   void SetColorRGB(int* color_array , int start_color , int finish_color , int frame_start , int frame_stop);
   
   AnimationBase(const AnimationBase& ab) {}
   
public :
   
   AnimationBase(ANIMATION_TYPE atype);
   virtual ~AnimationBase() {Free();}
   
   virtual void Reallocate(int number_of_frames , double total_play_time) {}
   virtual void Free();
   
   virtual void CalcSize()=0;// must be in the derived class

   virtual void DrawOn(BITMAP* bmp , int x , int y)=0;// must be in the derived class

   /// AdvanceFrameTime, SetFrameTime, and SetFrameNumber all return the now current loop number, -1 if no longer playing
   // If frame_time is negative , the animation will stop playing
   int AdvanceFrameTime(double delta_time);
   virtual int SetFrameTime(double frame_time);
   virtual int SetFrameNumber(int frame_number);

   /// CurrentFrame and CurrentLoop may return -1 to indicate the animation is not currently playing
   int CurrentFrame() {return frame_num;}
   int CurrentLoop() {return loop_num;}
   
   double CurrentFrameTime() {return frametime;}
   double FramesPerSecond() {return frames_per_sec;}
   double SecondsPerFrame() {ASSERT(frames_per_sec != 0.0);return 1.0/frames_per_sec;}
   double TimeBetweenFrames(int frame_start , int frame_stop);// this frame_stop is inclusive

   void SetTimeFactor(double factor);// multiplier for delta time passed to AdvanceFrameTime
   void SetLoopStop(int loop_stop);// -1 infinite repeat, loop_stop is loop_num to complete before stopping (zero based)

   void Pause() {paused = true;}
   void Resume() {paused = false;}
   void TogglePause() {paused = !paused;}

   /// These functions retrieve the currently stored data
   BITMAP* GetBitmap  (int frame_number);
   int     GetXpos    (int frame_number);
   int     GetYpos    (int frame_number);
   float   GetRotation(int frame_number);
   float   GetScale   (int frame_number);
   int     GetColor1  (int frame_number);
   int     GetColor2  (int frame_number);
   int     GetBaseX() {return base_x;}
   int     GetBaseY() {return base_y;}
   int     GetNumTrails() {return num_trails;}
   int     GetFramesPerTrail() {return frames_per_trail;}


   virtual void SetNumTrails(int numtrails , int num_frames_per_trail);

   /// These functions set all the data
   void SetBitmap  (BITMAP* bmp , bool free_bmp);
   void SetXpos    (int x);
   void SetYpos    (int y);
   void SetPos     (int x , int y);
   void SetRotation(double Rotation);
   void SetScale   (double Scale);
   void SetColor1  (int Color);
   void SetColor2  (int Color);

   /// These functions set the data for a given frame
   // frame_number is zero based - x frames gives frame numbers in the range [0,x-1]
   void SetBitmap(BITMAP* bmp , bool free_bmp , int frame_number);
   void SetXpos(int x , int frame_number);
   void SetYpos(int y , int frame_number);
   void SetPos(int x , int y , int frame_number);
   void SetRotation(double Rotation , int frame_number);
   void SetScale(double Scale , int frame_number);
   void SetColor1(int Color , int frame_number);
   void SetColor2(int Color , int frame_number);
   
   /// These functions set the data for a range of frames
   // frame_stop is exclusive so data[frame_stop] won't be set
   // frame_number is zero based - x frames gives frame numbers in the range [0,x-1]
   void SetBitmap(BITMAP* bmp , bool free_bmp , int frame_start , int frame_stop);
   void SetXpos(int x , int frame_start , int frame_stop);
   void SetYpos(int y , int frame_start , int frame_stop);
   void SetPos(int x , int y , int frame_start , int frame_stop);
   void SetRotation(double Rotation , int frame_start , int frame_stop);
   void SetScale(double Scale , int frame_start , int frame_stop);
   void SetColor1(int Color , int frame_start , int frame_stop);
   void SetColor2(int Color , int frame_start , int frame_stop);

   /// Data setting functions
   virtual void SetBasePos(int xpos , int ypos);
   
   void SetXPos(double xstart , double xspeed , double xaccel , int frame_start , int frame_stop);
   void SetYPos(double ystart , double yspeed , double yaccel , int frame_start , int frame_stop);
   // positive degrees rotate clockwise from zero pointing right
   void SetRotation(double rdeg , double rspeed , double raccel , int frame_start , int frame_stop);
   void SetColor1HSV(double hue , double huespeed , double sat , double satspeed ,
                     double val , double valspeed , int frame_start , int frame_stop);
   void SetColor2HSV(double hue , double huespeed , double sat , double satspeed ,
                     double val , double valspeed , int frame_start , int frame_stop);
   void SetColor1RGB(int start_color , int finish_color , int frame_start , int frame_stop);
   void SetColor2RGB(int start_color , int finish_color , int frame_start , int frame_stop);

};

enum BMP_DRAW_TYPE {
   BMP_DRAW_SOLID = 0,
   BMP_DRAW_MASKED = 1,
   BMP_DRAW_TRANSLUCENT = 2
};




class SimpleAnimation : public AnimationBase {
private :
   BMP_DRAW_TYPE drawtype;
public :
   
   SimpleAnimation(BMP_DRAW_TYPE dtype , ANIMATION_TYPE atype) :
         AnimationBase(atype),
         drawtype(dtype)
   {}
   ~SimpleAnimation() {Free();}
   
   void Reallocate(int number_of_frames , double total_play_time);

   void CalcSize() {}
   void DrawOn(BITMAP* bmp , int x , int y);

};


