/*
 *
 *     _______       ___       ____      __       _______ 
 *    /\  ____\    /|   \     /  __\    /\ \     /\  ____\ 
 *    \ \ \___/_   ||  _ \   |  /__/____\ \ \    \ \ \___/_ 
 *     \ \  ____\  || |_\ \  |\ \ /\_  _\\ \ \    \ \  ____\ 
 *      \ \ \___/_ ||  ___ \ \ \ \\//\ \/ \ \ \____\ \ \___/_ 
 *       \ \______\||_|__/\_\ \ \ \_\/ |   \ \_____\\ \______\
 *        \/______/|/_/  \/_/  \_\_____/    \/_____/ \/______/
 *
 *
 *    EAGLE
 *    Edgar's Allegro Gui Library and Extensions
 *
 *    Copyright 2009-2011 by Edgar Reynaldo
 *
 *    See EagleLicense.txt for allowed uses of this library.
 *
 */


#include <vector>
using std::vector;

#include "Animation.hpp"



void AnimationBase::SetColorHSV(int* color_array , double hue , double huespeed , double sat , double satspeed,
                  double val , double valspeed , int frame_start , int frame_stop) {
   ASSERT((frame_start >= 0) && (frame_start < num_frames));
   ASSERT(frame_stop > frame_start);
   ASSERT((frame_stop >= 1) && (frame_stop <= num_frames));
   ASSERT(frames_per_sec != 0.0);
   double t = 0.0;
   double spf = 1.0/frames_per_sec;// seconds per frame
   double h,s,v;
   int r,g,b;
   int col;
   for (int i = frame_start ; i < frame_stop ; ++i) {
      h = hue + huespeed*t;
      while(h >= 360.0) {h -= 360.0;}
      while(h < 0.0) {h += 360.0;}
      s = sat + satspeed*t;
      if (s > 1.0) {s = 1.0;}
      if (s < 0.0) {s = 0.0;}
      v = val + valspeed*t;
      if (v > 1.0) {v = 1.0;}
      if (v < 0.0) {v = 0.0;}
      hsv_to_rgb(h,s,v,&r,&g,&b);
      if ((r == 255) && (g == 0) && (b == 255)) {b = 254;}// fudge to prevent use of the mask color
      col = makecol(r,g,b);
      color_array[i] = col;
      t += spf;
   }
}



void AnimationBase::SetColorRGB(int* color_array , int start_color , int finish_color , int frame_start , int frame_stop) {
   ASSERT((frame_start >= 0) && (frame_start < num_frames));
   ASSERT(frame_stop > frame_start);
   ASSERT((frame_stop >= 1) && (frame_stop <= num_frames));
   int sr = getr(start_color);
   int sg = getg(start_color);
   int sb = getb(start_color);
   int fr = getr(finish_color);
   int fg = getg(finish_color);
   int fb = getb(finish_color);
   // Next two lines are a fudge to prevent use of the mask color
   if ((sr == 255) && (sg == 0) && (sb == 255)) {sb = 254;}
   if ((fr == 255) && (fg == 0) && (fb == 255)) {fb = 254;}
   double nframes = frame_stop - frame_start - 1;
   if (nframes == 0.0) {nframes = 1.0;}
   double dr = ((double)(fr - sr))/nframes;
   double dg = ((double)(fg - sg))/nframes;
   double db = ((double)(fb - sb))/nframes;
   double r = sr;
   double g = sg;
   double b = sb;
   int col;
   for (int i = frame_start ; i < frame_stop ; ++i) {
      col = makecol((int)r , (int)g , (int)b);
      color_array[i] = col;
      r += dr;
      g += dg;
      b += db;
   }
}



AnimationBase::AnimationBase(ANIMATION_TYPE atype) :
      frames(0),
      free_frame(0),
      xposition(0),
      yposition(0),
      rotation(0),
      scale(0),
      color1(0),
      color2(0),
      base_x(0),
      base_y(0),
      anim_type(atype),
      duration(0.0),
      frames_per_sec(0.0),
      frametime(0.0),
      time_factor(1.0),
      num_frames(0),
      frame_num(-1),
      loop_num(-1),
      loop_stop_num(-1),
      num_trails(0),
      frames_per_trail(1),
      paused(false)
{}



void AnimationBase::Free() {
   vector<BITMAP*> freed_list;
   if (frames) {
      if (free_frame) {
         for (int i = 0 ; i < num_frames ; ++i) {
            if (frames[i] && free_frame[i]) {
               bool freed = false;
               for (unsigned int j = 0 ; j < freed_list.size() ; ++j) {
                  if (freed_list[j] == frames[i]) {
                     freed = true;
                  }
               }
               if (!freed) {
                  destroy_bitmap(frames[i]);
                  freed_list.push_back(frames[i]);
               }
               frames[i] = 0;
            }
         }
      }
      delete [] frames;
      frames = 0;
   }
   if (free_frame) {
      delete [] free_frame;
      free_frame = 0;
   }
   if (xposition) {
      delete [] xposition;
      xposition = 0;
   }
   if (yposition) {
      delete [] yposition;
      yposition = 0;
   }
   if (rotation) {
      delete [] rotation;
      rotation = 0;
   }
   if (scale) {
      delete [] scale;
      scale = 0;
   }
   if (color1) {
      delete [] color1;
      color1 = 0;
   }
   if (color2) {
      delete [] color2;
      color2 = 0;
   }
   num_frames = 0;
   frames_per_sec = 0.0;
   frame_num = 0;
   loop_num = 0;
}



int AnimationBase::AdvanceFrameTime(double delta_time) {
   if (paused) {return loop_num;}
   return SetFrameTime(frametime + time_factor*delta_time);
}



int AnimationBase::SetFrameTime(double frame_time) {
   frametime = frame_time;
   double short_frame_time = frametime;
   double num_loops = 0.0;
   switch (anim_type) {
      case FORWARD_ONCE :
         if ((frametime < 0.0) || (frametime >= duration)) {
            frame_num = -1;
            loop_num = -1;
         } else {
            frame_num = (int)(frametime*frames_per_sec);
            loop_num = 0;
         }
         break;
      case BACKWARD_ONCE :
         if ((frametime < 0.0) || (frametime >= duration)) {
            frame_num = -1;
            loop_num = -1;
         } else {
            frame_num = (num_frames - 1) - (int)(frametime*frames_per_sec);
            loop_num = 0;
         }
         break;
      case FORWARD_AND_BACK_ONCE :
         if ((frametime < 0.0) || (frametime >= 2.0*duration)) {
            frame_num = -1;
            loop_num = -1;
         } else {
            if (frametime < duration) {
               frame_num = (int)(frametime*frames_per_sec);
            } else {
               frame_num = (num_frames - 1) - (int)((frametime-duration)*frames_per_sec);
            }
            loop_num = 0;
         }
         break;
      case FORWARD_REPEAT :
         loop_num = 0;
         if (frametime < 0.0) {
            frame_num = -1;
            loop_num = -1;
         } else {
            if (frametime >= duration) {
               num_loops = frametime/duration;
               loop_num = (int)num_loops;
               short_frame_time = frametime - duration*(double)(loop_num);
            }
            frame_num = (int)(short_frame_time*frames_per_sec);
         }
         break;
      case BACKWARD_REPEAT :
         loop_num = 0;
         if (frametime < 0.0) {
            frame_num = -1;
            loop_num = -1;
         } else {
            if (frametime >= duration) {
               num_loops = frametime/duration;
               loop_num = (int)num_loops;
               short_frame_time = frametime - duration*(double)(loop_num);
            }
            frame_num = (num_frames - 1) - (int)(short_frame_time*frames_per_sec);
//            frame_num = (int)(short_frame_time*frames_per_sec);
         }
         break;
      case FORWARD_AND_BACK_REPEAT :
         loop_num = 0;
         if (frametime < 0.0) {
            frame_num = -1;
            loop_num = -1;
         } else {
            if (frametime >= 2.0*duration) {
               num_loops = frametime/(2.0*duration);
               loop_num = (int)num_loops;
               short_frame_time = frametime - (2.0*duration)*(double)loop_num;
            }
            if (short_frame_time < duration) {
               frame_num = (int)(short_frame_time*frames_per_sec);
            } else {
               frame_num = (num_frames - 1) - (int)((short_frame_time-duration)*frames_per_sec);
            }
         }
         break;
   }
   if (loop_stop_num != -1) {
      if (loop_num > loop_stop_num) {
         frame_num = -1;
         loop_num = -1;
      }
   }
   return loop_num;
}



int AnimationBase::SetFrameNumber(int frame_number) {
   ASSERT(frames_per_sec != 0.0);
   frametime = (double)frame_number/frames_per_sec;
   frame_num = frame_number;
   loop_num = 0;
   if ((frame_num < 0) || (frame_num >= num_frames)) {
      frame_num = -1;
      loop_num = -1;
   }
   return loop_num;
}



double AnimationBase::TimeBetweenFrames(int frame_start , int frame_stop) {
   double nframes = (double)(frame_stop - frame_start);
   return nframes*SecondsPerFrame();
}



void AnimationBase::SetTimeFactor(double factor) {
   time_factor = factor;
}



void AnimationBase::SetLoopStop(int loop_stop) {
   if (loop_stop < -1) {loop_stop = -1;}
   loop_stop_num = loop_stop;
   SetFrameTime(frametime);
}



BITMAP* AnimationBase::GetBitmap(int frame_number) {
   ASSERT(frames);// make sure the frames array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   return frames[frame_number];
}



int AnimationBase::GetXpos(int frame_number) {
   ASSERT(xposition);// make sure the xposition array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   return xposition[frame_number];
}



int AnimationBase::GetYpos(int frame_number) {
   ASSERT(yposition);// make sure the yposition array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   return yposition[frame_number];
}



float AnimationBase::GetRotation(int frame_number) {
   ASSERT(rotation);// make sure the rotation array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   return rotation[frame_number];
}



float AnimationBase::GetScale(int frame_number) {
   ASSERT(scale);// make sure the scale array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   return scale[frame_number];
}



int AnimationBase::GetColor1(int frame_number) {
   ASSERT(color1);// make sure the color array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   return color1[frame_number];
}



int AnimationBase::GetColor2(int frame_number) {
   ASSERT(color2);// make sure the color array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   return color2[frame_number];
}



void AnimationBase::SetNumTrails(int numtrails , int num_frames_per_trail) {
   if (numtrails < 0) {numtrails = 0;}
   if (num_frames_per_trail < 1) {num_frames_per_trail = 1;}
   num_trails = numtrails;
   frames_per_trail = num_frames_per_trail;
}



void AnimationBase::SetBitmap(BITMAP* bmp , bool free_bmp) {
   ASSERT(frames);// make sure the frames array is allocated
   ASSERT(free_frame);// make sure the free_frame array is allocated
   for (int i = 0 ; i < num_frames ; ++i) {
      frames[i] = bmp;
      free_frame[i] = free_bmp;
   }
}



void AnimationBase::SetXpos(int x) {
   ASSERT(xposition);// make sure the xposition array is allocated
   for (int i = 0 ; i < num_frames ; ++i) {
      xposition[i] = x;
   }
}



void AnimationBase::SetYpos(int y) {
   ASSERT(yposition);// make sure the yposition array is allocated
   for (int i = 0 ; i < num_frames ; ++i) {
      yposition[i] = y;
   }
}



void AnimationBase::SetPos(int x , int y) {
   ASSERT(xposition);// make sure the xposition array is allocated
   ASSERT(yposition);// make sure the yposition array is allocated
   for (int i = 0 ; i < num_frames ; ++i) {
      xposition[i] = x;
      yposition[i] = y;
   }
}



void AnimationBase::SetRotation(double Rotation) {
   ASSERT(rotation);// make sure the rotation array is allocated
   double extra = Rotation/360.0;
   Rotation -= 360.0*(double)(int)extra;
   if (Rotation < 0.0) {Rotation += 360.0;}
   for (int i = 0 ; i < num_frames ; ++i) {
      rotation[i] = Rotation;
   }
}



void AnimationBase::SetScale(double Scale) {
   ASSERT(scale);// make sure the scale array is allocated
   ASSERT(Scale > 0.0);
   for (int i = 0 ; i < num_frames ; ++i) {
      scale[i] = Scale;
   }
}



void AnimationBase::SetColor1(int Color) {
   ASSERT(color1);// make sure the color array is allocated
   for (int i = 0 ; i < num_frames ; ++i) {
      color1[i] = Color;
   }
}



void AnimationBase::SetColor2(int Color) {
   ASSERT(color2);// make sure the color array is allocated
   for (int i = 0 ; i < num_frames ; ++i) {
      color2[i] = Color;
   }
}



void AnimationBase::SetBitmap(BITMAP* bmp , bool free_bmp , int frame_number) {
   ASSERT(frames);// make sure the frames array is allocated
   ASSERT(free_frame);// make sure the free_frame array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   frames[frame_number] = bmp;
   free_frame[frame_number] = free_bmp;
}



void AnimationBase::SetXpos(int x , int frame_number) {
   ASSERT(xposition);// make sure the xposition array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   xposition[frame_number] = x;
}



void AnimationBase::SetYpos(int y , int frame_number) {
   ASSERT(yposition);// make sure the yposition array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   yposition[frame_number] = y;
}



void AnimationBase::SetPos(int x , int y , int frame_number) {
   ASSERT(xposition);// make sure the xposition array is allocated
   ASSERT(yposition);// make sure the yposition array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   xposition[frame_number] = x;
   yposition[frame_number] = y;
}



void AnimationBase::SetRotation(double Rotation , int frame_number) {
   ASSERT(rotation);// make sure the rotation array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   double extra = Rotation/360.0;
   Rotation -= 360.0*(double)(int)extra;
   if (Rotation < 0.0) {Rotation += 360.0;}
   rotation[frame_number] = Rotation;
}



void AnimationBase::SetScale(double Scale , int frame_number) {
   ASSERT(scale);// make sure the scale array is allocated
   ASSERT(Scale > 0.0);
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   scale[frame_number] = Scale;
}



void AnimationBase::SetColor1(int Color , int frame_number) {
   ASSERT(color1);// make sure the color array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   color1[frame_number] = Color;
}



void AnimationBase::SetColor2(int Color , int frame_number) {
   ASSERT(color2);// make sure the color array is allocated
   ASSERT((frame_number >= 0) && (frame_number < num_frames));
   color2[frame_number] = Color;
}



void AnimationBase::SetBitmap(BITMAP* bmp , bool free_bmp , int frame_start , int frame_stop) {
   ASSERT(frames);// make sure the frames array is allocated
   ASSERT(free_frame);// make sure the free_frame array is allocated
   ASSERT((frame_start >= 0) && (frame_start < num_frames));
   ASSERT(frame_stop > frame_start);
   ASSERT((frame_stop > 0) && (frame_stop <= num_frames));
   for (int i = frame_start ; i < frame_stop ; ++i) {
      frames[i] = bmp;
      free_frame[i] = free_bmp;
   }
}



void AnimationBase::SetXpos(int x , int frame_start , int frame_stop) {
   ASSERT(xposition);// make sure the xposition array is allocated
   ASSERT((frame_start >= 0) && (frame_start < num_frames));
   ASSERT(frame_stop > frame_start);
   ASSERT((frame_stop > 0) && (frame_stop <= num_frames));
   for (int i = frame_start ; i < frame_stop ; ++i) {
      xposition[i] = x;
   }
}



void AnimationBase::SetYpos(int y , int frame_start , int frame_stop) {
   ASSERT(yposition);// make sure the yposition array is allocated
   ASSERT((frame_start >= 0) && (frame_start < num_frames));
   ASSERT(frame_stop > frame_start);
   ASSERT((frame_stop > 0) && (frame_stop <= num_frames));
   for (int i = frame_start ; i < frame_stop ; ++i) {
      yposition[i] = y;
   }
}



void AnimationBase::SetPos(int x , int y , int frame_start , int frame_stop) {
   ASSERT(xposition);// make sure the xposition array is allocated
   ASSERT(yposition);// make sure the yposition array is allocated
   ASSERT((frame_start >= 0) && (frame_start < num_frames));
   ASSERT(frame_stop > frame_start);
   ASSERT((frame_stop > 0) && (frame_stop <= num_frames));
   for (int i = frame_start ; i < frame_stop ; ++i) {
      xposition[i] = x;
      yposition[i] = y;
   }
}



void AnimationBase::SetRotation(double Rotation , int frame_start , int frame_stop) {
   ASSERT(rotation);// make sure the rotation array is allocated
   ASSERT((frame_start >= 0) && (frame_start < num_frames));
   ASSERT(frame_stop > frame_start);
   ASSERT((frame_stop > 0) && (frame_stop <= num_frames));
   double extra = Rotation/360.0;
   Rotation -= 360.0*(double)(int)extra;
   if (Rotation < 0.0) {Rotation += 360.0;}
   for (int i = frame_start ; i < frame_stop ; ++i) {
      rotation[i] = Rotation;
   }
}



void AnimationBase::SetScale(double Scale , int frame_start , int frame_stop) {
   ASSERT(scale);// make sure the scale array is allocated
   ASSERT(Scale > 0.0);
   ASSERT((frame_start >= 0) && (frame_start < num_frames));
   ASSERT(frame_stop > frame_start);
   ASSERT((frame_stop > 0) && (frame_stop <= num_frames));
   for (int i = frame_start ; i < frame_stop ; ++i) {
      scale[i] = Scale;
   }
}



void AnimationBase::SetColor1(int Color , int frame_start , int frame_stop) {
   ASSERT(color1);// make sure the color array is allocated
   ASSERT((frame_start >= 0) && (frame_start < num_frames));
   ASSERT(frame_stop > frame_start);
   ASSERT((frame_stop > 0) && (frame_stop <= num_frames));
   for (int i = frame_start ; i < frame_stop ; ++i) {
      color1[i] = Color;
   }
}



void AnimationBase::SetColor2(int Color , int frame_start , int frame_stop) {
   ASSERT(color2);// make sure the color array is allocated
   ASSERT((frame_start >= 0) && (frame_start < num_frames));
   ASSERT(frame_stop > frame_start);
   ASSERT((frame_stop > 0) && (frame_stop <= num_frames));
   for (int i = frame_start ; i < frame_stop ; ++i) {
      color2[i] = Color;
   }
}



void AnimationBase::SetBasePos(int xpos , int ypos) {
   base_x = xpos;
   base_y = ypos;
}



void AnimationBase::SetXPos(double xstart , double xspeed , double xaccel , int frame_start , int frame_stop) {
   // x = xstart + xspeed*t + (xaccel/2)*t^2
   ASSERT((frame_start >= 0) && (frame_start < num_frames));
   ASSERT(frame_stop > frame_start);
   ASSERT((frame_stop >= 1) && (frame_stop <= num_frames));
   ASSERT(frames_per_sec != 0.0);
   ASSERT(xposition);// make sure the xposition array is allocated
   double t = 0.0;
   double spf = 1.0/frames_per_sec;// seconds per frame
   double x;
   for (int i = frame_start ; i < frame_stop ; ++i) {
      x = xstart + xspeed*t + (xaccel/2.0)*t*t;
      xposition[i] = (int)x;
      t += spf;
   }
}



void AnimationBase::SetYPos(double ystart , double yspeed , double yaccel , int frame_start , int frame_stop) {
   // y = ystart + yspeed*t + (yaccel/2)*t^2
   ASSERT((frame_start >= 0) && (frame_start < num_frames));
   ASSERT(frame_stop > frame_start);
   ASSERT((frame_stop >= 1) && (frame_stop <= num_frames));
   ASSERT(frames_per_sec != 0.0);
   ASSERT(yposition);// make sure the yposition array is allocated
   double t = 0.0;
   double spf = 1.0/frames_per_sec;// seconds per frame
   double y;
   for (int i = frame_start ; i < frame_stop ; ++i) {
      y = ystart + yspeed*t + (yaccel/2.0)*t*t;
      yposition[i] = (int)y;
      t += spf;
   }
}



void AnimationBase::SetRotation(double rdeg , double rspeed , double raccel , int frame_start , int frame_stop) {
   // r = rdeg + rspeed*t + (raccel/2)*t^2
   ASSERT((frame_start >= 0) && (frame_start < num_frames));
   ASSERT(frame_stop > frame_start);
   ASSERT((frame_stop >= 1) && (frame_stop <= num_frames));
   ASSERT(frames_per_sec != 0.0);
   ASSERT(rotation);// make sure the rotation array is allocated
   double t = 0.0;
   double spf = 1.0/frames_per_sec;// seconds per frame
   double r;
   double extra;
   for (int i = frame_start ; i < frame_stop ; ++i) {
      r = rdeg + rspeed*t + (raccel/2.0)*t*t;
      // r = 0 + 900*4.0 + -225*16 = 3600 - 2250 - 1200 - 150 = 0
      // rv = 900 + -450*4 = -1800
      extra = r/360.0;
      r -= 360.0*(double)(int)extra;
      if (r < 0.0) {r += 360.0;}
      rotation[i] = (int)r;
      t += spf;
   }
}



void AnimationBase::SetColor1HSV(double hue , double huespeed , double sat , double satspeed ,
                  double val , double valspeed , int frame_start , int frame_stop) {
   ASSERT(color1);// make sure the color1 array is allocated
   SetColorHSV(color1 , hue , huespeed , sat , satspeed , val , valspeed , frame_start , frame_stop);
}



void AnimationBase::SetColor2HSV(double hue , double huespeed , double sat , double satspeed ,
                  double val , double valspeed , int frame_start , int frame_stop) {
   ASSERT(color2);// make sure the color2 array is allocated
   SetColorHSV(color2 , hue , huespeed , sat , satspeed , val , valspeed , frame_start , frame_stop);
}



void AnimationBase::SetColor1RGB(int start_color , int finish_color , int frame_start , int frame_stop) {
   ASSERT(color1);
   SetColorRGB(color1 , start_color , finish_color , frame_start , frame_stop);
}



void AnimationBase::SetColor2RGB(int start_color , int finish_color , int frame_start , int frame_stop) {
   ASSERT(color2);
   SetColorRGB(color2 , start_color , finish_color , frame_start , frame_stop);
}



/// ----------------------------------     SimpleAnimation     -------------------------------------------



void SimpleAnimation::Reallocate(int number_of_frames , double total_play_time) {
   ASSERT(total_play_time > 0.0);
   ASSERT(number_of_frames > 0);
   if ((total_play_time > 0.0) && (number_of_frames > 0)) {
      Free();
      num_frames = number_of_frames;
      duration = total_play_time;
      frames_per_sec = (double)num_frames / duration;
      SetFrameTime(0.0);
      
      frames = new BITMAP*[num_frames];
      free_frame = new bool[num_frames];
      xposition = new int[num_frames];
      yposition = new int[num_frames];
      
      for (int i = 0 ; i < num_frames ; ++i) {
         frames[i] = 0;
         free_frame[i] = false;
         xposition[i] = 0;
         yposition[i] = 0;
      }
      
   }
}
   
void SimpleAnimation::DrawOn(BITMAP* bmp , int x , int y) {
   if (frames) {
      if (frame_num == -1) {
         return;
      }
      BITMAP* src = frames[frame_num];
      if (src) {
         int w = src->w;
         int h = src->h;
         int xp = x + xposition[frame_num] - w/2;
         int yp = y + yposition[frame_num] - h/2;
         switch (drawtype) {
            case BMP_DRAW_SOLID :
               blit(src , bmp , 0 , 0 , xp , yp , w , h);
               break;
            case BMP_DRAW_MASKED :
               draw_sprite(bmp , src , xp , yp);
               break;
            case BMP_DRAW_TRANSLUCENT :
               set_alpha_blender();
               draw_trans_sprite(bmp , src , xp , yp);
               break;
         }
      }
   }
}


