#include <allegro5/allegro5.h>
#include <allegro5/allegro_opengl.h>
#include <allegro5/allegro_direct3d.h>


#define AL_SHADER_SUPPORT_CG  // This #define should be used by CMake to know if the user wants CG support. If not defined, CG won't
                              // be necessary to build the add-on.


#ifdef AL_SHADER_SUPPORT_CG
  // Necessary CG Includes.
  #include <cg/cg.h>
  #include <cg/cgD3D9.h>
  #include <cg/cgGL.h>
#endif

#ifndef AL_SHADER_H_INCLUDED
#define AL_SHADER_H_INCLUDED


// The default length of each string in a shader file.
#define AL_SHADER_DEFAULT_LENGTH            512

// Max Length of the main program string. This one's used to know what's the pixel shader program name is.
#define AL_SHADER_MAX_MAIN                  512

// A string used to determine what's the name of the pixel shader program.
extern char al_shader_program_name[];


/*
  ALLEGRO_SHADER_GL holds the handles to a GLSL Pixel Shader program.
  The functions for handling them are followed by _gl.
*/
typedef struct ALLEGRO_SHADER_GL {
  GLhandleARB handle;         // Handle to the fragment shader program.
  GLhandleARB handle_source;  // Handle to the source of the fragment shader program.
} ALLEGRO_SHADER_GL;


#ifdef AL_SHADER_SUPPORT_CG
  /*
    ALLEGRO_SHADER_CG_PS holds a CG Pixel shader program.
    The functions for handling them are followed by _cg_ps.
    Two different loading functions are avaible for this type of shaders.
  */
  typedef struct ALLEGRO_SHADER_CG_PS {
    CGprogram    program;   // The CG Handle to the pixel shader program.
  } ALLEGRO_SHADER_CG_PS;
#endif


/*
  ALLEGRO_SHADER is the struct the user would normally use for accesing these programs. They aren't followed by any
  suffixes, and will call the routines of gl_shader and cg_ps_shader if needed.
*/

typedef struct ALLEGRO_SHADER {
  ALLEGRO_SHADER_GL *gl_shader;            // Holds the value of a GLSL shader.
  #ifdef AL_SHADER_SUPPORT_CG
    ALLEGRO_SHADER_CG_PS *cg_ps_shader;    // Holds the value of a CG shader.
  #endif
} ALLEGRO_SHADER;


#ifdef AL_SHADER_SUPPORT_CG

  /*
    Internal method used for loading a CG Pixel shader program from a text array.

    char **text :  A 2 Dimensional array that holds the program.
    int size    :  The amount of strings that the text array holds.
  */
  ALLEGRO_SHADER_CG_PS *i_al_create_shader_cg_ps(char **text, int size);

  /*
    Internal method used for loading a CG Pixel shader program from a text array that holds a CG Effect.
    The CG Effect isn't used, only the Pixel shader program is loaded. Notice that this function will look
    for the program by the contents of al_shader_program_name, which can be set with al_set_shader_main_name.

    char **text :  A 2 Dimensional array that holds the program.
    int size    :  The amount of strings that the text array holds.
  */
  ALLEGRO_SHADER_CG_PS *i_al_create_shader_cg_ps_from_effect(char **text, int size);


  /*
    External method used for loading a CG Pixel shader program from a text array.

    char **text :  A 2 Dimensional array that holds the program.
    int size    :  The amount of strings that the text array holds.
  */
  ALLEGRO_SHADER *al_create_shader_cg_ps(char **text, int size);

  /*
    External method used for loading a CG Pixel shader program from a text array that holds a CG Effect.
    The CG Effect isn't used, only the Pixel shader program is loaded. Notice that this function will look
    for the program by the contents of al_shader_program_name, which can be set with al_set_shader_main_name.

    char **text :  A 2 Dimensional array that holds the program.
    int size    :  The amount of strings that the text array holds.
  */
  ALLEGRO_SHADER *al_create_shader_cg_ps_from_effect(char **text, int size);


  /*
    External method used for deleting the contents of a CG Pixel Shader program.

    ALLEGRO_SHADER_CG_PS *shader : Shader to delete.
  */
  void al_delete_shader_cg_ps(ALLEGRO_SHADER_CG_PS *shader);

  /*
    External method used for initializing the CG Support.
  */
  int al_init_cg_addon();


  /*
    External method for setting a texture used by a CG Pixel Shader program.

    ALLEGRO_SHADER_CG_PS *shader :   A CG Pixel shader program.
    const char *name             :   Name of the texture to look up for in the program.
    ALLEGRO_BITMAP *bt           :   Texture to be used in the program.
  */

  void al_set_shader_texture_cg_ps(ALLEGRO_SHADER_CG_PS *shader, const char *name, ALLEGRO_BITMAP *bt);

  /*
    External method for setting a float used by a CG Pixel Shader program.

    ALLEGRO_SHADER_CG_PS *shader :   A CG Pixel shader program.
    const char *name             :   Name of the float to look up for in the program.
    float v                      :   Float value to be used in the program.
  */
  void al_set_shader_float_cg_ps(ALLEGRO_SHADER_CG_PS *shader, const char *name, float v);

  /*
    Puts to use a CG Pixel Shader program. To reset using any pixel shaders, al_clear_shaders_cg, or al_clear_shaders
    should be called.

    ALLEGRO_SHADER_CG_PS *shader :   A CG Pixel shader program.
  */
  void al_apply_shader_cg_ps(ALLEGRO_SHADER_CG_PS *shader);


  /*
    If any shaders are active, this method will clear the usage of any of them. This one's used to clear the usage of
    CG Pixel Shader programs.
  */
  void al_clear_shaders_cg();


  /*
    Exits and cleans any stuff needed by CG.
  */
  void al_exit_shader_addon_cg();
#endif



/*
  Internal method used for loading a GLSL Pixel shader program from a text array.

  char **text :  A 2 Dimensional array that holds the program.
  int size    :  The amount of strings that the text array holds.
*/
ALLEGRO_SHADER_GL *i_al_create_shader_gl(char **text, int size);

/*
  External method used for loading a GLSL Pixel shader program from a text array.

  char **text :  A 2 Dimensional array that holds the program.
  int size    :  The amount of strings that the text array holds.
*/
ALLEGRO_SHADER *al_create_shader_gl(char **text, int size);




/*
  External method for setting a texture used by a GLSL Pixel Shader program.

  ALLEGRO_SHADER_GL *s         :   A GLSL Pixel shader program.
  const char *name             :   Name of the texture to look up for in the program.
  ALLEGRO_BITMAP *bt           :   Texture to be used in the program.
*/

void al_set_shader_texture_gl(ALLEGRO_SHADER_GL *s, const char *name, ALLEGRO_BITMAP *bt);


/*
  External method for setting a float used by a GLSL Pixel Shader program.

  ALLEGRO_SHADER_GL *s         :   A GLSL Pixel shader program.
  const char *name             :   Name of the float to look up for in the program.
  float v                      :   Float value to be used in the program.
*/

void al_set_shader_float_gl(ALLEGRO_SHADER_GL *s, const char *name, float v);


/*
  Puts to use a CG Pixel Shader program. To reset using any pixel shaders, al_clear_shaders_cg, or al_clear_shaders
  should be called.

  ALLEGRO_SHADER_GL *s         :   A GLSL Pixel shader program.
*/

void al_apply_shader_gl(ALLEGRO_SHADER_GL *s);



/*
  If any shaders are active, this method will clear the usage of any of them. This one's used to clear the usage of
  GLSL Pixel Shader programs.
*/

void al_clear_shaders_gl();


/*
  External method used for deleting the contents of a GLSL Pixel Shader program.

  ALLEGRO_SHADER_GL *shader : Shader to delete.
*/

void al_delete_shader_gl(ALLEGRO_SHADER_GL *shader);


//-----------------------------------------------------------------------------------------------------------------
// IMPORTANT NOTE: The following methods are the ones that should be used by the user.
//-----------------------------------------------------------------------------------------------------------------


/*
  External method used for loading a Pixel shader program from a file. This method should be the one used by
  the user, because it reads the file extension to determine which type of shader to load.

  const char *filename   :  Name of the file to be loaded.
  int max_line_length    :  Max length of each line in the pixel shader program. If you're not sure what number to use,
                            use AL_SHADER_DEFAULT_LENGTH.

  returns an ALLEGRO_SHADER program to use with any of the following methods.
*/

ALLEGRO_SHADER *al_load_shader(const char *filename, int max_line_length);


/*
  External method used for loading a Pixel shader program from a file. The difference is that you force the shader to be
  loaded as if the extension was another. The extension should be something like ".glsl", ".cg", ".cgfx".

  const char *filename   :  Name of the file to be loaded.
  const char *extension  :  Extension desired.
  int max_line_length    :  Max length of each line in the pixel shader program. If you're not sure what number to use,
                            use AL_SHADER_DEFAULT_LENGTH.

  returns an ALLEGRO_SHADER program to use with any of the following methods.
*/

ALLEGRO_SHADER *al_load_shader_as(const char *filename, const char *extension, int max_line_length);


/*
  Clears the contents of an ALLEGRO_SHADER. This one's called normally when loading a new shader, so you don't
  need to use it.

  ALLEGRO_SHADER *shader   : Program to clear.
*/

void al_clear_shader(ALLEGRO_SHADER *shader);


/*
  External method used for deleting the contents of an ALLEGRO_SHADER program.

  ALLEGRO_SHADER *shader   : Program to delete.
*/

void al_delete_shader(ALLEGRO_SHADER *shader);



/*
  External method for applying a Pixel shader effect to any following calls for drawing. You should clear the usage of
  any shaders when you no longer need them with al_clear_shaders();

  ALLEGRO_SHADER *shader  : Program to use.
*/

void al_apply_shader(ALLEGRO_SHADER *shader);

/*
  External Method for clearing the usage of any current shaders.
*/
void al_clear_shaders();



/*
  External method for setting a texture used by a Pixel Shader program.

  ALLEGRO_SHADER *shader       :   A Pixel shader program.
  const char *name             :   Name of the texture to look up for in the program.
  ALLEGRO_BITMAP *bt           :   Texture to be used in the program.
*/

void al_set_shader_texture(ALLEGRO_SHADER *shader, const char *name, ALLEGRO_BITMAP *bt);


/*
  External method for setting a float used by a Pixel Shader program.

  ALLEGRO_SHADER *shader       :   A Pixel shader program.
  const char *name             :   Name of the float to look up for in the program.
  float v                      :   Float value to be used in the program.
*/

void al_set_shader_float(ALLEGRO_SHADER *shader, const char *name, float v);


/*
  Method used to initialize the Shader addon. This should be called AFTER creating the display, because the Addon
  needs to know which video driver to use.
*/

int al_init_shader_addon();


/*
  Exit the shader Add-on. This is registered to be used at exit normally by al_init_shader_addon(),
  so you don't need to call it.
*/

void al_exit_shader_addon();


/*
  Sets the name of the Pixel shader program to look for. Normally, this is defaulted to "ps_main",
  and you can change it here if you don't want your functions to be called like that.

  char *name   : Copies this string to the program name.
*/

void al_set_shader_main_name(char *name);

/*
  Returns a string pointing to the actual name used for locating pixel shader programs.
*/
char *al_get_shader_main_name();


#endif
