#include "al_shader.h"
#include <stdio.h>
#include <d3dx9core.h>
#include "cg/cg.h"
#include <allegro5/internal/aintern.h>


char al_shader_program_name[AL_SHADER_MAX_MAIN] =  "ps_main";


void al_clear_shader(ALLEGRO_SHADER *shader) {
  shader->gl_shader = NULL;
#ifdef AL_SHADER_SUPPORT_CG
  shader->cg_ps_shader = NULL;
#endif
}




char **al_load_text_from_file(const char *filename, int max_line_length, int *size) {
  ALLEGRO_FILE *fp;
  char **text;
  int sz;
  fp = al_fopen(filename, "r+");  // Open the file read only

  if (!fp) return NULL;


  // Determine the total length of the shader text.
  sz=0;
  char str[max_line_length];
  while (al_fgets(fp, str, max_line_length) != NULL) {
    sz++;
  }

  // Allocate the text array
  text = (char **) malloc(sizeof(char *)*sz);
  int i=0;

  // Rewind the file back at the start
  al_fseek(fp, 0, ALLEGRO_SEEK_SET);

  // Read each text line from the shader
  while (al_fgets(fp, str, max_line_length) != NULL) {
    text[i] = (char *) malloc(sizeof(char)*max_line_length);
    strcpy(text[i], str);
    i++;
  }

  *size = sz;

  // Close the file
  al_fclose(fp);

  return text;
}


/*
  Loads a shader from a text file and compiles it.
*/


ALLEGRO_SHADER *al_load_shader(const char *filename, int max_line_length) {


  // Create the shader based on the text array
  char **text;
  int sz;
  text = al_load_text_from_file(filename, max_line_length, &sz);
  if (!text) {
    TRACE("Couldn't load %s shader file.", filename);
    return NULL;
  }

  ALLEGRO_PATH *path;
  path = al_create_path(filename);


  char ext[16];
  strcpy(ext, al_get_path_extension(path));
  al_free_path(path);



  ALLEGRO_SHADER *shad=NULL;
  #ifdef AL_SHADER_SUPPORT_CG
    if ((!strcmp(ext, ".fx")) || (!strcmp(ext, ".cgfx"))) {
      shad=al_create_shader_cg_ps_from_effect(text, sz);
    }
    if ((!strcmp(ext, ".cg")) || (!strcmp(ext, ".hlsl"))) {
      shad=al_create_shader_cg_ps(text, sz);
    }
  #endif


  if (!strcmp(ext, ".glsl")) {
    if (al_get_display_flags() & ALLEGRO_OPENGL) {
      shad=al_create_shader_gl(text, sz);
    }
    else {
      TRACE("Can't load %s if the driver used isn't OpenGL", filename);
    }
  }

  // Free the text array.
  for  (int i=0; i!=sz; i++) {
    free(text[i]);
  }
  free(text);

  // Return the new shader
  return shad;
}



ALLEGRO_SHADER *al_load_shader_as(const char *filename, const char *ext, int max_line_length) {
  // Create the shader based on the text array
  char **text;
  int sz;
  text = al_load_text_from_file(filename, max_line_length, &sz);
  if (!text) {
    TRACE("Couldn't load %s shader file.", filename);
    return NULL;
  }

  ALLEGRO_SHADER *shad=NULL;
  #ifdef AL_SHADER_SUPPORT_CG
    if ((!strcmp(ext, ".fx")) || (!strcmp(ext, ".cgfx"))) {
      shad=al_create_shader_cg_ps_from_effect(text, sz);
    }
    if ((!strcmp(ext, ".cg")) || (!strcmp(ext, ".hlsl"))) {
      shad=al_create_shader_cg_ps(text, sz);
    }
  #endif


  if (!strcmp(ext, ".glsl")) {
    if (al_get_display_flags() & ALLEGRO_OPENGL) {
      shad=al_create_shader_gl(text, sz);
    }
    else {
      TRACE("Can't load %s if the driver used isn't OpenGL", filename);
    }
  }

  // Free the text array.
  for  (int i=0; i!=sz; i++) {
    free(text[i]);
  }
  free(text);

  // Return the new shader
  return shad;
}



int al_init_shader_addon() {
  int status;
  bool d3d_flag, ogl_flag;
  d3d_flag = al_get_display_flags() & ALLEGRO_DIRECT3D;
  ogl_flag = al_get_display_flags() & ALLEGRO_OPENGL;

  status = 1;
  #ifdef AL_SHADER_SUPPORT_CG
    al_init_cg_addon();
  #endif


  if (ogl_flag) {
    if (!al_is_opengl_extension_supported("GL_EXT_framebuffer_object")) status = 0;
    if (!al_is_opengl_extension_supported("GL_ARB_fragment_shader")) status = 0;
  }

  _al_add_exit_func(al_exit_shader_addon, "al_exit_shader_addon");

  return status;
}

void al_exit_shader_addon() {
  #ifdef AL_SHADER_SUPPORT_CG
    al_exit_shader_addon_cg();
  #endif
}



void al_apply_shader(ALLEGRO_SHADER *shader) {
  if (shader == NULL) return;

  #ifdef AL_SHADER_SUPPORT_CG
    if (ALLEGRO_SHADER_CG_PS *s=shader->cg_ps_shader) {
      al_apply_shader_cg_ps(s);
    }
  #endif

  if (ALLEGRO_SHADER_GL *s=shader->gl_shader) {
    al_apply_shader_gl(s);
  }
}


void al_clear_shaders() {
  #ifdef AL_SHADER_SUPPORT_CG
    al_clear_shaders_cg();
  #endif

  if (al_get_display_flags() & ALLEGRO_OPENGL) {
    al_clear_shaders_gl();
  }
}



void al_set_shader_texture(ALLEGRO_SHADER *shader, const char *name, ALLEGRO_BITMAP *bt) {
  if (shader == NULL) return;


  #ifdef AL_SHADER_SUPPORT_CG
    if (ALLEGRO_SHADER_CG_PS *s=shader->cg_ps_shader) {
      al_set_shader_texture_cg_ps(s, name, bt);
    }
  #endif

  if (ALLEGRO_SHADER_GL *s=shader->gl_shader) {
    al_set_shader_texture_gl(s, name, bt);
  }
}


void al_set_shader_float(ALLEGRO_SHADER *shader, const char *name, float v) {
  if (shader == NULL) return;

  #ifdef AL_SHADER_SUPPORT_CG
    if (ALLEGRO_SHADER_CG_PS *s=shader->cg_ps_shader) {
      al_set_shader_float_cg_ps(s, name, v);
    }
  #endif

  if (ALLEGRO_SHADER_GL *s=shader->gl_shader) {
    al_set_shader_float_gl(s, name, v);
  }
}



void al_delete_shader(ALLEGRO_SHADER *shader) {
  if (shader == NULL) return;

  if (shader->gl_shader != NULL ) al_delete_shader_gl(shader->gl_shader);

  #ifdef AL_SHADER_SUPPORT_CG
    if (shader->cg_ps_shader != NULL) al_delete_shader_cg_ps(shader->cg_ps_shader);
  #endif

  free(shader);
}



void al_set_shader_main_name(char *name) {
  if (name == NULL) return;
  strcpy(al_shader_program_name, name);
}

char *al_get_shader_main_name() {
  return al_shader_program_name;
}




/* --------------------

         GLSL

-----------------------*/





ALLEGRO_SHADER_GL *i_al_create_shader_gl(char **text, int size) {
  ALLEGRO_SHADER_GL *shader;
  if (text == NULL) return NULL;

  shader = (ALLEGRO_SHADER_GL *) malloc(sizeof(ALLEGRO_SHADER_GL));

  shader->handle_source = glCreateShaderObjectARB(GL_FRAGMENT_SHADER_ARB);


  glShaderSourceARB(shader->handle_source, size, (const GLcharARB **)text, NULL);
  glCompileShaderARB(shader->handle_source);
  shader->handle = glCreateProgramObjectARB();
  glAttachObjectARB(shader->handle, shader->handle_source);
  glLinkProgramARB(shader->handle);

  return shader;
}




/*
  Creates a shader based on a text array.
*/



ALLEGRO_SHADER *al_create_shader_gl(char **text, int size) {
  ALLEGRO_SHADER *shader;
  if (text == NULL) return NULL;

  shader = (ALLEGRO_SHADER *) malloc(sizeof(ALLEGRO_SHADER));
  al_clear_shader(shader);

  shader->gl_shader = i_al_create_shader_gl(text, size);
  return shader;
}




void al_set_shader_texture_gl(ALLEGRO_SHADER_GL *s, const char *name, ALLEGRO_BITMAP *bt) {
  if (s == NULL) return;
  /* TODO:
    Here, I need the Add-on to set the sampler2D with char *name.
    ALLEGRO_SHADER_GL is the struct that holds the GLSL Pixel Shader program. The GLhandleARB to the pixel
    shader program is the ->handle member.

    As you can see, I've tried different things below. I get the index of the texture with al_get_opengl_texture, and
    use the glUniform1iARB to set the sampler2D. For some reason, any sampler I call in the shaders return the bitmap being
    drawn, and other behaviour that I have no idea of.

    Probaly some extension is not being used, or OGL Draw bitmaps reset something?

    The glActiveTextureARB and glBindTexture are my failed attempts at trying to correct the situtation :/
  */

  /*
    GLint loc;
    loc = glGetUniformLocationARB(s->handle, name);

    glUniform1iARB(loc, al_get_opengl_texture(bt));

    glActiveTextureARB(GL_TEXTURE0+loc);
    glBindTexture(GL_TEXTURE_2D, al_get_opengl_texture(bt));
  */
}



void al_set_shader_float_gl(ALLEGRO_SHADER_GL *s, const char *name, float v) {
  GLint loc;
  loc = glGetUniformLocationARB(s->handle, name);
  glUniform1fARB(loc, v);
}



void al_apply_shader_gl(ALLEGRO_SHADER_GL *s) {
  if (s==NULL) return;
  glUseProgramObjectARB(s->handle);
}



void al_clear_shaders_gl() {
  glUseProgramObjectARB(0);
}


void al_delete_shader_gl(ALLEGRO_SHADER_GL *shader) {
  if (shader == NULL) return;

  glDetachObjectARB(shader->handle, shader->handle_source);
  glDeleteObjectARB(shader->handle_source);

  free(shader);
}
