#include <sys/time.h>
#include <stdint.h>
#include "assert.h"
#include "game.h"
#include "timer.h"

/* returns the current value of the counter, in micro seconds */
uint64_t get_timer(void)
{
   struct timeval t;
   gettimeofday(&t, NULL);
   
   return t.tv_sec*1000000 + t.tv_usec; 
}

/* Start the clock for the current player */
void start_clock(gamestate_t *game)
{
   assert(game);
   game->start_time = get_timer();
}

/* Get the time elapsed since the last call to start_clock(), in ms */
int peek_timer(const gamestate_t *game)
{
   return (get_timer() - game->start_time) / 1000;
}

/* Time management: check if a fixed time per move has passed since the
 * clock was started.
 */
static bool check_time_per_move_clock(const gamestate_t *game)
{
   int time_passed = peek_timer(game);

   if (time_passed >= game->time_per_move)
      return true;

   return false;
}

static bool check_time_for_game(const gamestate_t *game)
{
   int time_passed = peek_timer(game);
   int num_moves = game->movestogo;
   int time_for_move;

   if (num_moves == 0) {
      num_moves = 40 - game->root_moves_played % 40;
   }

   if (num_moves < 0) num_moves = 10;

   time_for_move = game->time_left[game->running_clock]/(num_moves+1);
   if (game->time_left[game->running_clock] >
      game->time_inc[game->running_clock]/2)
      time_for_move += game->time_inc[game->running_clock]/2;

   if (time_passed >= time_for_move)
      return true;

   return false;

   if (game->time_left[game->running_clock] >
      2*game->time_inc[game->running_clock])
      time_for_move += game->time_inc[game->running_clock];

   /* Very little time left at all */
   if (game->time_left[game->running_clock] < 500)
      return true;

   if (time_passed && game->time_left[game->running_clock] < 1000)
      return true;

   /* Check if we've used our aallocated time for this move */
   if ( time_passed >= (time_for_move+game->extra_time) )
      return true;

   return false;
}

void set_time_per_move(gamestate_t *game, int msec)
{
   game->time_per_move = msec;
   game->check_clock = check_time_per_move_clock;
}

void set_infinite_time(gamestate_t *game)
{
   game->check_clock = NULL;
}

void set_time_for_game(gamestate_t *game)
{
   game->check_clock = check_time_for_game;
}
