#include "alphabeta.h"
#include "board.h"
#include "movegen.h"
#include "game.h"
#include "inline/game.h"
#include "evaluate.h"

#define HARD_HORIZON 0

/* Quiescence search */
alphabeta_state_t quiescence(gamestate_t *game, int horizon_distance, int alpha, int beta)
{
   alphabeta_state_t state;
   movelist_t movelist;
   int best_score;
   int legal_moves, n;
   int player;

   /* We're at the search horizon, return static evaluation.
    * This is a hard limit to prevent the program from locking up in a
    * perpetual check. These will eventually repeat, of course, but until
    * we can detect that it's best to have this check in.
    * We also abort if the static evaluation gives a beta cutoff.
    */
   best_score = state.score = static_evaluation(game->board, game->player);
   if ( (horizon_distance == HARD_HORIZON) || (state.score >= beta) ) {
      return state;
   }
   
   player = game->player;

   /* If we're in check, we need to generate all moves */
   if (player_in_check(game, player)) {
      generate_moves(&movelist, game->board, game->player);
   } else {
      generate_quiescence_moves(&movelist, game->board, game->player);
   }
   game->board->in_check = movelist.in_check;

   /* Now search all moves */
   legal_moves = movelist.num_moves;
   for (n=0; n<movelist.num_moves; n++) {
      playmove(game, movelist.move[n]);
      if (player_in_check(game, player)) { /* Illegal move! */
         legal_moves--;
         takeback(game);
         continue;
      }

      state = quiescence(game, horizon_distance-1, -beta, -alpha);
      state.score = -state.score;
      takeback(game);

      if (state.score > best_score) best_score = state.score;


      if (best_score >= beta)             /* Beta cutoff */
         break;

      if (best_score > alpha)
         alpha = best_score;
   }

   /* FIXME: does not distinguish between checkmate and stalemate.
    * We're in a bit of trouble if the quiescence search reports a
    * checkmate, because we don't know that the variation is forced in any
    * way... but we can simply extend the search one level up.
    */
   if (legal_moves == 0) {
      if (game->board->in_check) {
         //best_score = -CHECKMATE - horizon_distance;
         //print_bitboards(game->board);
         //exit(0);
      }
   }
   
   state.score = best_score;
   return state;

}

alphabeta_state_t alphabeta(gamestate_t *game, int horizon_distance, int alpha, int beta)
{
   movelist_t movelist;
   alphabeta_state_t state;
   alphabeta_state_t new_state;
   int best_score, a;
   int legal_moves, n;
   int player;

   state.score = 0;
   state.size_of_variation = 0;

   /* We're at the search horizon, return static evaluation */
   if (horizon_distance == 0) {
      state = quiescence(game, horizon_distance, alpha, beta);
      //state.score = static_evaluation(game->board, game->player);
      if (abs(state.score) > (CHECKMATE + HARD_HORIZON)) {
         /* We're in trouble...
          * We should re-search this node normally.
          */
         printf("Quiescence search found mate! %d\n", state.score);
         horizon_distance = abs(state.score) - CHECKMATE+2;
         printf("New horizon distance: %d\n", horizon_distance);
         state = alphabeta(game, horizon_distance, alpha, beta);
         printf("Research: %d\n", state.score);
         //exit(0);
      }

      /* Don't return variation from quiescence search; it's useless */
      state.size_of_variation = 0;

      return state;
   }

   player = game->player;
   best_score = -2*CHECKMATE;

   /* Begin minimax search */
   generate_moves(&movelist, game->board, game->player);
   game->board->in_check = movelist.in_check;

   state.mainline[0] = movelist.move[0];
   state.size_of_variation = 1;
   legal_moves = movelist.num_moves;
   for (n=0; n<movelist.num_moves; n++) {
      playmove(game, movelist.move[n]);
      if (player_in_check(game, player)) { /* Illegal move! */
         legal_moves--;
         takeback(game);
         if (moves_are_equal(movelist.move[n], state.mainline[0]))
            state.mainline[0] = movelist.move[n+1];
         continue;
      }

      new_state = alphabeta(game, horizon_distance-1, -beta, -alpha);
      new_state.score = -new_state.score;
      takeback(game);

      if (new_state.score > best_score) best_score = new_state.score;

      if (best_score >= beta)             /* Beta cutoff */
         break;

      if (best_score > alpha) {  /* New best continuation found */
         alpha = best_score;
         state.mainline[0] = movelist.move[n];
         if (new_state.size_of_variation)
            memcpy(&state.mainline[1], &new_state.mainline[0],
                              new_state.size_of_variation*sizeof(move_t));
         state.size_of_variation = new_state.size_of_variation+1;
      }
   }

   /* No legal moves, either checkmate or stalemate */
   if (legal_moves == 0) {
      if (game->board->in_check) {
         best_score = -CHECKMATE - horizon_distance;
      } else {
         best_score = - horizon_distance;
      }
   }
   
   state.score = best_score;
   return state;
}

