#include "config.h"
#include "sbr.h"
#include "alphabeta.h"
#include "board.h"
#include "movegen.h"
#include "game.h"
#include "inline/game.h"
#include "evaluate.h"
#include "see.h"

#undef TEST_MOVES_BEFORE_SEARCH
#undef DEBUG_HASHTABLE_ACCES

/* When taking back a move here, we don't have to be afraid of going back
 * beyond the first move in the game - so it's safe to skip the test for
 * that in the takeback() function.
 */
#define takeback takeback_no_check

/* Function call to check how much time we have left
 * We do the check every time clock_positions nodes have been searched.
 * Don't do this during QS - there aren't that many QS nodes anyway.
 */
static const int clock_positions = 0x7FFF;
#define check_clock() \
   if (game->check_clock && ((positions_evaluated & clock_positions)==0))\
      abort_search |= game->check_clock(game)

volatile bool abort_search = 0;
int positions_evaluated;
int moves_searched;
int positions_in_hashtable;
int branches_pruned;

/* quicksort callback function */
static int movelist_sorter(void *score_ptr, const void *mi1, const void *mi2)
{
   int *score = (int*)score_ptr;
   int i1 = *(int*)mi1;
   int i2 = *(int*)mi2;

   return score[i2] - score[i1];
}

/* Return the number of repetitions of the current position in the game
 * tree.
 */
inline int count_repetition(gamestate_t *game)
{
   int n;
   int count;
   count = 0;
   for (n=game->moves_played-1; n>=0; n--) {
      if (is_capture(game->move_list[n]) || (game->move_list[n].piece&PAWN))
         return count;

      count += (game->board_list[n].hash == game->board->hash);
   }

   return count;
}

/* Check if the current position is a draw by repetition. If so, there are
 * no legal moves in it - the position should be treated as stalemate.
 */
inline bool draw_by_repetition(gamestate_t *game)
{
   int n;
   int count;
   count = 1;
   for (n=game->moves_played-1; n>=0; n--) {
      //printf("%d %s\n", n, move_string(game->move_list[n], NULL));
      if (is_capture(game->move_list[n]) || (game->move_list[n].piece&PAWN))
         return false;

      //printf("%d %d %p %p\n", game->moves_played, n, game->board_list+n, game->board);
      count += (game->board_list[n].hash == game->board->hash);
   }
   //printf("%d\n", count);

   return count >= 3;
}

/* Quiescence search */
alphabeta_state_t quiescence(gamestate_t *game, int horizon_distance, int alpha, int beta)
{
   alphabeta_state_t state;
   movelist_t movelist;
   move_t hash_move;
   hash_table_entry_t *hash = NULL;
   int move_perm[MAX_MOVES];
   int score[MAX_MOVES];
   int best_score;
   int legal_moves, n;
   int player;
   int qdepth;
   int hash_flag = HASH_TYPE_ALPHA;
   int max_gain = 0;

   /* Clear memory */
  memset(&state, 0, sizeof state);
  qdepth = -horizon_distance;
  state.qdepth = qdepth;

   /* We're at the search horizon, return static evaluation.
    * This is a hard limit to prevent the program from locking up in a
    * perpetual check. These will eventually repeat, of course, but until
    * we can detect that it's best to have this check in.
    * We also abort if the static evaluation gives a beta cutoff.
    */
   best_score = state.score = static_evaluation(game, game->player);
   if ( (horizon_distance == HARD_HORIZON) || (state.score >= beta) ) {
      return state;
   }

   /* Look up this position in the hash table. It's not likely, but it
    * might be there - this is quick to check anyway.
    * Only one caveat: we can't return a "checkmate" result here because we
    * can't be sure that the variation up to now has been forced (we can if
    * all moves up to now have been checks).
    * If we do find a checkmate, the position is probably not "quiet"
    * though...
    */
   hash = query_table_entry(game->qs_table, game->board->hash);
   if (hash) {
      positions_in_hashtable++;
      /* If we searched this position before from the same depth or better,
       * just return the previous score. Otherwise we can still use the
       * recorded move to improve our move ordering.
       */
      if (hash->depth >= horizon_distance && abs(hash->score)<(CHECKMATE-60)) {
         if (hash->flags & HASH_TYPE_EXACT) {
            state.score = hash->score;
            return state;
         } else if ((hash->flags & HASH_TYPE_ALPHA) && (hash->score<=alpha)) {
            state.score = hash->score;
            return state;
         } else if ((hash->flags & HASH_TYPE_BETA) && (hash->score>=beta)) {
            state.score = hash->score;
            return state;
         }
      }
   }
   player = game->player;
   positions_evaluated++;

   /* If we're in check, we need to generate all moves */
   if (game->board->in_check) {
      generate_moves(&movelist, game->board, game->player);
      if (!movelist.num_moves) {
         state.score = -CHECKMATE;
         return state;
      }
   } else {
      generate_quiescence_moves(&movelist, game->board, game->player);
      /* If there are no captures or promotions, simply return the static
       * score - the position is quiet.
       */
      if (!movelist.num_moves)
         return state;
   }

   /* Sort the moves */
   for (n=0; n<movelist.num_moves; n++) {
      move_perm[n] = n;

      /* Preserve the order of the moves put out by the move generator, if
       * we have no reason to change it. This seems to work best for late
       * move ordering...
       */
      score[n] = 255-n;

      /* Give the highest priority to the move from the hash table, if it
       * exists.
       */
      if (hash && moves_are_equal(hash->best_move, movelist.move[n])) {
         score[n] += CHECKMATE;
         continue;
      }

      /* Promotion moves */
      if (is_promotion(movelist.move[n])) {
         int gain = static_exchange_evaluation(game->board, movelist.move[n]);
         score[n] += piece_value(get_promotion_piece(movelist.move[n]));
         if (gain+875 > max_gain)
            max_gain = gain+875;
         continue;
      }

      /* Capture; use SEE to estimate relative worth of captures */
      if (is_capture(movelist.move[n])) {
         int gain = static_exchange_evaluation(game->board, movelist.move[n]);
         score[n] += gain;
         if (gain > max_gain)
            max_gain = gain;
         continue;
      }
   }

#ifdef USE_DELTA_PRUNING
   /* "Delta pruning" or futility pruning during quiescence: if the
    * expected gain from any move in the current position cannot improve
    * alpha, then this node is poor and we might as well skip the rest of
    * QS.
    */
   if (state.score+max_gain + 500 < alpha) {
      return state;
   }
#endif
   qsort_r(move_perm, movelist.num_moves, sizeof *move_perm, score,
                                                         movelist_sorter);

   /* Check if the best move (estimated through SEE) looks like it might
    * win material. If not, just return the static score, since we don't expect
    * to gain much here. This is not "safe", but this is the quiescence
    * search, and this suggests that we can't improve our score by
    * captures.
    */
   if (score[move_perm[0]] < 0) {
      return state;
   }
      
   /* Now search all moves */
   legal_moves = movelist.num_moves;
   for (n=0; n<movelist.num_moves; n++) {
      int local_beta = beta;

#ifdef USE_QS_PVS
      if (n>0) local_beta = alpha+1;
#endif

#ifdef PRUNE_LOSING_CAPTURES
      /* Don't look at losing captures */
      if (score[move_perm[n]] < 0)
         continue;
#endif
      playmove(game, movelist.move[move_perm[n]]);
      if (player_in_check(game, player)) { /* Illegal move! */
         legal_moves--;
         takeback(game);
         continue;
      }
      moves_searched++;

      state = quiescence(game, horizon_distance-1, -local_beta, -alpha);
      state.score = -state.score;
#ifdef USE_QS_PVS
      /* PVS: detect failed searches */
      if (state.score > alpha && state.score < beta) {
         state = quiescence(game, horizon_distance-1, -beta, -alpha);
         state.score = -state.score;
      }
#endif
      takeback(game);

      if (state.score >= CHECKMATE+HARD_HORIZON) {
         state.score--;
      } else if (state.score <= -CHECKMATE-HARD_HORIZON) {
         state.score++;
      }

      if (qdepth > state.qdepth)
         state.qdepth = qdepth;
      else
         qdepth = state.qdepth;

      if (state.score > best_score) {
         hash_move = movelist.move[move_perm[n]];
         best_score = state.score;
      }

      if (best_score >= beta) {           /* Beta cutoff */
         hash_flag = HASH_TYPE_BETA;
         break;
      }

      if (best_score > alpha) {
         hash_flag = HASH_TYPE_EXACT;
         alpha = best_score;
      }
   }

   state.score = best_score;

   /* Detect forced check mates: if the position is a check mate and all
    * moves leading up to this position have been check evasions, the line
    * was actually forced (since all moves are generated when considering
    * check evasions). There might be a simpler mate in the position, but
    * this is better than not detecting the forced line in the first place.
    * If the line is not based on check evasions at every move, we assume
    * that the line was not forced and simply return the static score.
    * Alternatively, we should maybe flag this branch as "dangerous" and
    * let upstream decide what to do about that (eg, research).
    */
   if (legal_moves == 0 && game->board->in_check && !(abs(horizon_distance)&1)) {
      /* If the side to play encountered checks at all nodes up to this
       * point during QS, then the sequence of moves leading to this
       * position was forced and we can return the mate score.
       */
      for (n=horizon_distance; n<=0; n+=2) {
         if (!game->board[horizon_distance - n].in_check) {
            return state;
         }
      }

      state.score = -CHECKMATE;
   }

   store_table_entry(game->qs_table, game->board->hash, 
                     horizon_distance,  best_score, hash_flag, hash_move);
   return state;
}

/* Store a "counter move"; this is a depth-independent extension of the
 * killer principle.
 */
static void inline store_counter_move(gamestate_t *game, move_t prev_move, move_t move)
{
   if (game->board->in_check || is_promotion(move) || is_capture(move) ||
      !prev_move.m) {
      return;
   }
   game->counter[prev_move.from][prev_move.to].m = move.m;
}

/* Store moves that kill a branch in the killer slots, but only if:
 *  - we were not in check at the beginning of this move
 *  - the move is not a promotion (already high in the tree)
 *  - the move was not a capture (already high in the tree)
 */
static inline void store_killer(gamestate_t *game, move_t move, int depth)
{
   if (game->board->in_check || is_promotion(move) || is_capture(move)) {
      return;
   }

#ifdef DEBUG_KILLER_TABLE
   if ((game->killer[0][depth].m == game->killer[1][depth].m)
         && game->killer[0][depth].m) {
      printf("Error: killers are equal but non-zero:");
      printf(" %s ", move_string(game->killer[0][depth], NULL));
      printf("%s\n", move_string(game->killer[1][depth], NULL));
   }
#endif
   if (moves_are_equal(move, game->killer[0][depth])) {
      /* The move was the first killer - do nothing */
#ifdef SHOW_KILLER_VERBOSE_OUTPUT
      printf("Killer 1 (%s", move_string(game->killer[0][depth], NULL));
      printf(" %s) killed a tree at depth %d.\n",
            move_string(game->killer[1][depth], NULL), depth);
#endif
#ifdef USE_THIRD_KILLER
   } else if (moves_are_equal(move, game->killer[1][depth])) {
      /* The move was the second killer (out of three) - promote it to
       * first killer, leaving the third killer untouched.
       */
      game->killer[1][depth].m=game->killer[0][depth].m;
      game->killer[0][depth].m=move.m;
#endif
   } else {
#ifdef SHOW_KILLER_VERBOSE_OUTPUT
      if (moves_are_equal(move, game->killer[1][depth])) {
         printf("Killer 2 (%s", move_string(game->killer[0][depth], NULL));
         printf(" %s) killed a tree at depth %d.\n",
               move_string(game->killer[1][depth], NULL), depth);
      } else {
         printf("Store new killer at ply %d: %s",depth,move_string(move,NULL));
         printf(" (%s ", move_string(game->killer[0][depth], NULL));
         printf("%s)\n", move_string(game->killer[1][depth], NULL));
      }
#endif
      /* This was either the last killer (out of 2 or 3), or it's a new
       * move. Either way, Degrade first killer to second killer (etc) and
       * store the new first killer.
       */
#ifdef USE_THIRD_KILLER
      game->killer[2][depth].m=game->killer[1][depth].m;
#endif
      game->killer[1][depth].m=game->killer[0][depth].m;
      game->killer[0][depth].m=move.m;
   }
}

static inline int get_extension(const gamestate_t *game, int horizon_distance, move_t move, move_t prev_move)
{
   int local_extension = 0;
#ifdef RECAPTURE_EXTENSION
   /* If this move is a recaputre move, then extend the search by one
    * ply. Previously we looked at all recaptures, now we only look at
    * recaptures between pieces of the same value; otherwise the search
    * tree just explodes.
    * Another idea is only to extend catures where SEE tells us we lose
    * material that would otherwise fall outside teh search horizon. The
    * idea behind this is that we can look for compensation.
    */
   if (is_capture(prev_move) && prev_move.to == move.to &&
         static_exchange_evaluation(game->board, move)<=0
//         piece_value(prev_move.piece&PIECE) == piece_value(move.piece&PIECE) 
         && horizon_distance < 3*PLY) {
      local_extension+=PLY;
   }
#endif
#ifdef PAWN_PUSH_EXTENSION
   /* Extend the search by two ply if a pawn moves to the seventh row -
    * assuming the square in front of it is free or it can capture into
    * a promotion.
    */
   if (move.piece&PAWN && ( (move.to>=A7 && move.to<=H7) ||
            ( (move.to>=A2 && move.to<=H2) )) ) {
      bitboard_t occ = game->board->bbc[0]|game->board->bbc[1];
      bitboard_t promo_bitboard;
      int promotion_square = move.to + 8 - ((move.piece & BLACK)>>3);
      promo_bitboard = make_bitboard_square(promotion_square);
      if ( (occ & promo_bitboard)==0) {
         local_extension += 2*PLY;
      } else {
         bitboard_t cbp;
         cbp  = (promo_bitboard & board_minus_a)>>1;
         cbp |= (promo_bitboard & board_minus_h)<<1;
         if ( (game->board->bbc[1-((move.piece&BLACK)>>7)] & cbp))
            local_extension += 2*PLY;
      }
   }
#endif

   return local_extension;
}

alphabeta_state_t alphabeta(gamestate_t *game, int horizon, int horizon_distance, int alpha, int beta, bool in_pv)
{
   movelist_t movelist;
   alphabeta_state_t state;
   alphabeta_state_t new_state;
   hash_table_entry_t *hash;
   move_t hash_move;
   move_t best_move;
   move_t prev_move;
   move_t my_prev_move;
   int move_perm[MAX_MOVES];
   int score[MAX_MOVES];
   int best_score;
   int legal_moves, n, c;
   int player;
   int depth;
   int extension = 0;
   int hash_flag = HASH_TYPE_ALPHA;
   bool have_hash_move = false;
   bool mate_threat = false;

   memset(&state, 0, sizeof state);

   /* The very first thing we do is to check if this position has occurred
    * three times in the game tree, or whether no irreversible move has
    * been played in 50 moves (100 ply). Either of these conditions means
    * we should flag a draw and return immediately.
    * During the search, we simply score positions as a draw if they have
    * occurred at least once before. From the top level, we don't claim a
    * draw until three repetitions have been made.
    */
   if (game->fifty_counter[game->moves_played] >= 100 ||
       (game->repetition_hash_table[game->board->hash&0xFFFF]>=2
         && count_repetition(game)>=1)) {
      state.score = 0;
      return state;
   }

   /* Actual search depth since the root of the tree */
   depth = horizon - horizon_distance;

   state.score = 0;
   state.size_of_variation = 0;
   state.depth = depth;
   state.qdepth = 0;
#ifdef PRINCIPLE_VARIATION_IN_GAME_STRUCT
   game->length_of_variation[depth+1] = depth+1;
#endif

   /* We're in trouble if we ever get here... not that this is likely! */
   if (depth/PLY >= MAX_SEARCH_DEPTH) {
      state.score = static_evaluation(game, game->player);
      return state;
   }

   /* Check whether we still have enough time for this ply */
   check_clock();

   if (horizon/PLY>3 && abort_search) return state;

   /* Look up this position in the hash table */
   hash_move.m = 0;
   hash = query_table_entry(game->transposition_table, game->board->hash);
   if (hash) {
      hash_move.m = hash->best_move.m;
      positions_in_hashtable++;
#ifdef DEBUG_HASHTABLE_ACCES
      printf("This position was searched before.\n");
      printf("Score was %d at distance %d (now %d) and best move was %s\n",
               hash->score, hash->depth, horizon_distance,
               move_string(hash->best_move, NULL));
#endif
      /* If we searched this position before from the same depth or better,
       * just return the previous score. Otherwise we can still use the
       * recorded move to improve our move ordering.
       */
      /* FIXME: we should not ignore the hahs table if we find a mate in
       * there... just need to make sure that the score is correct!
       */
      if (hash->depth >= horizon_distance/PLY) {
         if (hash->flags & HASH_TYPE_EXACT) {
            state.score = hash->score;
            state.mainline[0] = hash->best_move;
            state.size_of_variation = 1;
            state.depth += hash->depth;

            return state;
         } else if ((hash->flags & HASH_TYPE_ALPHA) && (hash->score<=alpha)) {
            state.score = hash->score;
            state.mainline[0] = hash->best_move;
            state.size_of_variation = 1;
            state.depth += hash->depth;

            return state;
         } else if ((hash->flags & HASH_TYPE_BETA) && (hash->score>=beta)) {
            state.score = hash->score;
            state.mainline[0] = hash->best_move;
            state.size_of_variation = 1;
            state.depth += hash->depth;

            return state;
         }
      }
   } else {
#ifdef DEBUG_HASHTABLE_ACCES
      printf("This position is new\n");
#endif
      /* We don't have a move now, which is bad... try to get one from
       * interbal iterative deepening. This improves the move ordering
       * considerably and makes a large difference in speed at larger
       * depths.
       */
#ifdef USE_IID
      if (horizon_distance > 4*PLY) {
         int iid_depth = horizon_distance / 2 + PLY;
         int iid_horizon = horizon-horizon_distance+iid_depth;
         new_state = alphabeta(game, iid_horizon, iid_depth, alpha, beta, in_pv);

         if (new_state.size_of_variation) {
            hash_move = new_state.mainline[0];
         }
      }
#endif
   }

   /* We're at the search horizon, return static evaluation */
   if (horizon_distance < PLY) {
      /* If we're in check don't go into quiescence search but extend the
       * depth of the search by one ply.
       */
      if (game->board->in_check) {
         state = alphabeta(game, horizon+PLY, horizon_distance+PLY, -beta, -alpha, in_pv);
         return state;
      }

      state = quiescence(game, 0, alpha, beta);
      //state.score = static_evaluation(game, game->player);

      /* Don't return variation from quiescence search; it's useless */
      state.size_of_variation = 0;
      state.depth = depth;

      return state;
   }

   player = game->player;
   best_score = -2*CHECKMATE;
   positions_evaluated++;

   /* Store the previous move: this is used to change move ordering on this
    * move. If there is no previous move, then this move will be null.
    */
   prev_move.m = 0;
   if (game->moves_played>1)
      prev_move = game->move_list[game->moves_played-1];
   my_prev_move.m = 0;
   if (game->moves_played>2)
      my_prev_move = game->move_list[game->moves_played-2];

#ifdef ALLOW_NULL_MOVE
   /* Do a NULL-move search - unless in the situations where this should be
    * avoided:
    *  - we're in check (obviously)
    *  - risk of zugzwang 
    *  - too close to the horizon (or on the first ply)
    *  - previous move was a NULL move
    */
#undef NULL_MOVE_PRUNING
#ifdef NULL_MOVE_PRUNING
    if (!game->board->in_check && (prev_move.m != 0) &&
         (game->board->material[game->player>>7]>1800) &&
         (horizon_distance > 3*PLY) ) {
      int r = 2;

      /* play null-move */
      play_null_move(game);
      new_state = alphabeta(game, horizon-(r-1)*PLY,
                            horizon_distance-r*PLY,
                            -beta, -alpha, in_pv);
      new_state.score = -new_state.score;
      takeback(game);
      if (new_state.score >= beta) { /* null-move causes a cut-off */
         return new_state;
      }
    }
#else
    /* NULL move reductions; an alternative approach to NULL moves.
     * Disallow NULL moves if
     *  - we're in check (obviously)
     *  - the last move was a move out-of-check (too dangerous, we may have
     *    something much better that we could otherwise miss)
     *  - we're too close to the horizon (don't skip to QS from a NULL
     *    move)
     *  - the previous move was a null move (the search would collapse into
     *    nothing and we'd just end up researching the position at a
     *    shallower depth - which we've already done thanks to ID)
     */
    if (!game->board->in_check && !game->board[-1].in_check &&
        (prev_move.m != 0) && (horizon_distance > 3*PLY) ) {
      int r = 4;
      if (horizon_distance < 6*PLY) r = 3;

      /* play null-move */
      play_null_move(game);
      new_state = alphabeta(game, horizon-(r-1)*PLY, horizon_distance-r*PLY,
                            -beta, 1-beta, false);
      new_state.score = -new_state.score;
      takeback(game);
      if (new_state.score >= beta) { /* null-move causes a cut-off */
         extension -= 4*PLY;
      }

      /* Mate threat detection */
      if (new_state.score <= (-CHECKMATE+10) && extension<=0 && horizon_distance <=5*PLY) {
         extension += PLY;
         mate_threat = true;
      }
    }
#endif
#endif

   /* Generate moves, then sort them based on their estimated value: the
    * move generator gives a crude guess for this, and we can use the
    * killer tables to refine this.
    * The 0-th element of the killer tables is at the root of the tree,
    * rather than at the leaves. The reason for this is simple: we want to
    * reuse the killers as much as possible between iterative deepening.
    */
   generate_moves(&movelist, game->board, game->player);
   legal_moves = movelist.num_moves;
   if (legal_moves == 0) {
      state.size_of_variation = 0;
      if (game->board->in_check) {
         best_score = -CHECKMATE;
      } else {
         best_score = 0;
      }
   }

#ifdef FEW_MOVES_EXTENSION
#define LATE_FEW_MOVES_EXTENSION
//#define EARLY_FEW_MOVES_EXTENSION
#endif

#ifdef EARLY_FEW_MOVES_EXTENSION
   if ( (legal_moves < 3) && (horizon_distance <= 2*PLY) &&
         !game->board->in_check) {
      extension++;
   }
#endif

   /* Enhanced transposition cutoff: check if any of the children are in
    * the hash table and would cause a cutoff
    */
#ifdef ENHANCED_TRANSPOSITION_CUTOFF
   for (n=0; n<movelist.num_moves; n++) {
      uint64_t new_key = update_hashkey(game->board, movelist.move[n]);
      hash_table_entry_t *hash = query_table_entry(game->transposition_table, new_key);
      if (hash) {
         if (hash->depth >= (horizon_distance/PLY-1)) {
            if (hash->flags & (HASH_TYPE_EXACT|HASH_TYPE_ALPHA)) {
               if (-hash->score >= beta) {
                  state.score = -hash->score;
                  return state;
               }
            }
         }
      }
   }
#endif

   for (n=0; n<movelist.num_moves; n++) {
      move_t move = movelist.move[n];
      move_perm[n] = n;

      /* Preserve the order of the moves put out by the move generator, if
       * we have no reason to change it. This seems to work best for late
       * move ordering...
       */
      score[n] = 255-n;
      
      /* Give the highest priority to the move from the hash table, if it
       * exists.
       */
      if (moves_are_equal(hash_move, move)) {
         score[n] += CHECKMATE;
         have_hash_move = true;
         continue;
      }

#ifdef USE_MATE_KILLER
      /* Mate killer: a killer move that produced a mate score */
      if (moves_are_equal(game->mate_killer[depth], move)) {
         score[n] += CHECKMATE-100;
         continue;
      }
#endif

      /* Capture on promotion square, probably a fairly good move */
      if (is_promotion(prev_move) &&
          (get_destination(prev_move) == get_destination(move))) {
         score[n] += 2100;
         continue;
      }

      /* Promotion moves */
      if (is_promotion(movelist.move[n])) {
         score[n] += 1050 + piece_value(get_promotion_piece(move));
         continue;
      }

      /* Regular captures
       * The score is 950+the SSE score, which sorts winning captures below
       * promotion moves (with a score >= 1050) and losing captures with a
       * score <1000 below everything else.
       */
      if (is_capture(move)) {
         int see_score = static_exchange_evaluation(game->board, move);
         score[n] += 950+see_score;
         continue;
      }

      /* Award killers a higher score, so they are searched first
       * Given the weights here, winning captures are normally searched
       * before killer moves, losing captures are searched after killer
       * moves.
       */
      if (moves_are_equal(game->killer[0][depth], move)) {
         score[n] += 1100;
         continue;
      }
      if (moves_are_equal(game->killer[1][depth], move)) {
         score[n] += 1090;
         continue;
      }
#ifdef USE_THIRD_KILLER
      if (moves_are_equal(game->killer[2][depth], move)) {
         score[n] += 1090;
         continue;
      }
#endif

      /* Counter moves, a generalisation of the Killer concept.
       * The question is whether these should come before, or after
       * killers...
       * Doesn't seem to make much of a difference, but it sometimes hurts
       * badly if they are sorted before the killer moves, so they should
       * probably go between the killers and the losing captures.
       */
#ifdef USE_COUNTER_MOVES
      if (moves_are_equal(game->counter[prev_move.from][prev_move.to],
          move)) {
         score[n] += 1000;
         continue;
      }
#endif

      /* Also look at killers from earlier positions. This doesn't make a
       * big difference, but it speeds things up very slightly still and
       * isn't likely to hurt.
       */
      if (depth>2 && moves_are_equal(game->killer[0][depth-2], move)) {
         score[n] += 1080;
         continue;
      }
      if (depth>2 && moves_are_equal(game->killer[1][depth-2], move)) {
         score[n] += 1070;
         continue;
      }

#if 0
      /* Penalise moves that simply undo the previous move */
      if (depth>2 && movelist.move[n].to==my_prev_move.from &&
         movelist.move[n].from==my_prev_move.to)
         score[n] -= 100;
#endif

#if 0
      if (move.piece & BLACK) {
         score[n] += (unpack_row(move.from) - unpack_row(move.to));
      } else {
         score[n] += (unpack_row(move.to) - unpack_row(move.from));
      }
      score[n] += piece_value(move.piece&WOOD)/10;
#endif

      /* Finally, add history scores to the move */
#ifdef USE_HISTORY_HEURISTIC
      if (game->max_history) {
         int history_score =
            game->history[player>>7][movelist.move[n].from][movelist.move[n].to];
         score[n] += 400*history_score/game->max_history;
      }
#endif
   }

#ifdef CHECK_EXTENSION
   /* If there are two checks in a row close to the leaves of the tree,
    * best extend - there might be a mate thread!
    */
   if (game->board->in_check && //horizon_distance <= 10*PLY &&
       game->moves_played>2 && horizon<20*PLY) {// (MAX_SEARCH_DEPTH-1)) {
#ifdef EXTEND_ALL_CHECKS
      extension += PLY;
#else
      int check = game->board[-2].in_check;
      extension += check * PLY;
#endif
   }
#endif

   /* If the hash move is available, search it first, before calling qsort
    * to sort the other moves. This speeds up the search a bit.
    * The duplication of code here is a bit ugly though and this could be
    * cleaned up.
    */
   if (have_hash_move) {
      int local_extension = extension;
      move_t move;
      move.m = hash_move.m;
      local_extension += get_extension(game, horizon_distance, move, prev_move);

      /* Play move and evaluate the resulting position recursively -
       * assuming the move was legal.
       */
      playmove(game, move );
      if (player_in_check(game, player)) { /* Illegal move! */
         legal_moves--;
         takeback(game);
      } else {
      moves_searched++;

      new_state = alphabeta(game, horizon+local_extension,
            horizon_distance+local_extension-PLY, -beta, -alpha, in_pv);
      new_state.score = -new_state.score;

      /* For forced mates further down the tree, we should adjust the score
       * by one ply: mate one node down is a mate-in-one-ply from this ply
       * and a mate-in-two-ply one node further up.
       * We need two checks: one to see if we're mating and one to see if
       * we're being mated.
       */
      new_state.score -= (new_state.score >=  CHECKMATE-1000);
      new_state.score += (new_state.score <= -CHECKMATE+1000);
      takeback(game);

      if (new_state.score > best_score) {
         best_score = new_state.score;
         state.depth = new_state.depth;
         state.qdepth = new_state.qdepth;

         if (best_score > alpha) {           /* New best line found */
            hash_flag = HASH_TYPE_EXACT;
            alpha = best_score;

            /* Store good moves in the killer slots, but only if:
             *  - we were not in check at the beginning of this move
             *  - the move is not a promotion (already high in the list)
             *  - the move was not a capture (already high in the list)
             */
            store_killer(game, move, depth / PLY);
            store_counter_move(game, prev_move, move);
#ifdef USE_MATE_KILLER
            if (best_score >= (CHECKMATE-1000))
               game->mate_killer[depth].m = move.m;
#endif

            /* Update and store principle variation */
            state.mainline[0] = move;
            state.size_of_variation = new_state.size_of_variation+1;
            if (new_state.size_of_variation)
               memcpy(&state.mainline[1], &new_state.mainline[0],
                     new_state.size_of_variation*sizeof(move_t));

#ifdef PRINCIPLE_VARIATION_IN_GAME_STRUCT
            /* Copy principle variation */
            game->principle_variation[depth+1][depth+1] = move;
            for (c=depth+2; c < game->length_of_variation[depth+2]; c++) {
               game->principle_variation[depth+1][c] =
                  game->principle_variation[depth+2][c];
            }
            game->length_of_variation[depth+1] =
               game->length_of_variation[depth+2];
#endif

#ifdef USE_HISTORY_HEURISTIC
            int history_score;

            game->history[player>>7][move.from][move.to]+=depth*depth/(PLY*PLY);
            history_score = game->history[player>>7][move.from][move.to];
            if (history_score > game->max_history) {
               game->max_history = history_score;
            }
#endif

            if (best_score >= beta) {           /* Beta cutoff */
               hash_flag = HASH_TYPE_BETA;
               branches_pruned++;
            }
         }
      }
      }
      n = 1;
   } else {
      n = 0;
      hash_move.m = 0;
      best_move = state.mainline[0] = movelist.move[0];
      state.size_of_variation = 0;
   }

   /* Begin minimax search */
   if (best_score < beta) {
   qsort_r(move_perm, movelist.num_moves, sizeof *move_perm, score,
                                                         movelist_sorter);

   for (; n<movelist.num_moves; n++) {
      int local_extension = extension;
      move_t move = movelist.move[move_perm[n]];
      int local_beta = beta;
      bool move_is_futile = false;

      if (n>0)
         in_pv = false;

      /* Principle variation search: use full window only for first move. */
#ifdef USE_PVS
      if (n>0) local_beta = alpha+1;
#endif

      /* Local extensions for this move only */
      local_extension += get_extension(game, horizon_distance, move, prev_move);

#ifdef LATE_MOVE_REDUCTION
      /* Reduce the search depth for "uninteresting" moves.
       * The criterion "n>5" is rather crude and should probably be
       * replaced by "n>number of interesting moves", which lists the move
       * from the hash table, killer moves and captures.
       * We should NOT do this in the main variation, however.
       * Another idea is to remove the "&& !is_capture" condition. Removing
       * this will reduce all losing captures by 2 ply, which might be
       * dangerous...
       * The code below uses a middle-ground approach: it reduces moves
       * that seem poor by 2 ply, but it will only reduce losing captures
       * by 1 ply. This seems to be a good compromise. In addition, we
       * don't reduce moves where the opponent was in check on the move
       * before quite as much. This means that if this move is a
       * capture following an out-of-check move, we do not reduce at all.
       */
      if ((score[move_perm[n]]<1000) && local_extension==0 && !mate_threat
          && horizon_distance>2*PLY && !game->board->in_check &&
          prev_move.m != 0 && !in_pv) {
         local_extension = -2*PLY;      /* -1 or -2? */
         local_extension += is_capture(move)*PLY;
         local_extension += game->board[-1].in_check*PLY;
      }
#endif

      /* Play move and evaluate the resulting position recursively -
       * assuming the move was legal.
       */
      playmove(game, move );
      if (player_in_check(game, player)) { /* Illegal move! */
         legal_moves--;
         takeback(game);
         /* Prevent illegal moves from polluting the main variation */
         if (moves_are_equal(move, state.mainline[0])
            && (n<(movelist.num_moves-1)))
            best_move = state.mainline[0] = movelist.move[move_perm[n+1]];
         continue;
      }
      moves_searched++;

      /* Futility pruning: if the static evaluation is very poor, then try
       * to weed out moves that do not improve the static score.
       */
#ifdef FUTILITY_PRUNING
      move_is_futile = false;
      bool reduction_candidate = false;
      if (horizon_distance <= 4*PLY)
         reduction_candidate = game->moves_played>1 &&
            !game->board->in_check && !game->board[-1].in_check &&
            local_extension==0;
      if (reduction_candidate) {
         static const int margins[6] = {0,0,300,500,900,1500};
         static const int new_depth[6] = {0,0,0,0,2,0};
         int score = static_evaluation(game, player);
         //int score = lazy_evaluation(game, player)+100;
         int margin = margins[horizon_distance / PLY];
         int d = new_depth[horizon_distance / PLY];
         if ((score + margin) < alpha) { /* Poor move, go directly to QS */
            new_state = alphabeta(game, horizon, d, -local_beta, -alpha, in_pv);
            move_is_futile = true;
         }
      }

      if (!move_is_futile)
#endif
         new_state = alphabeta(game, horizon+local_extension,
               horizon_distance+local_extension-PLY, -local_beta, -alpha, in_pv);
      new_state.score = -new_state.score;

#ifdef USE_PVS
      /* PVS: detect failed searches
       * However, there's no point in doing this if the top-level window is
       * already a null window!
       */
      if (new_state.score > alpha && new_state.score < beta && abs(alpha-beta)>1) {
         new_state = alphabeta(game, horizon+local_extension,
                               horizon_distance+local_extension-PLY,
                               -beta, -alpha, in_pv);
         new_state.score = -new_state.score;
      }
#endif
      /* Research moves that were reduced but now fail high (improve alpha)
       * on normal depth. A quick test suggests that this overcomes the
       * downside of LMR (being blind to threats in side lines) without
       * eliminating the benefit of LMR (deep/fast search along the PV).
       */
      if (new_state.score > alpha && local_extension < 0) {
         new_state = alphabeta(game, horizon, horizon_distance-PLY,
                               -beta, -alpha, in_pv);
         new_state.score = -new_state.score;
      }

      /* For forced mates further down the tree, we should adjust the score
       * by one ply: mate one node down is a mate-in-one-ply from this ply
       * and a mate-in-two-ply one node further up.
       * We need two checks: one to see if we're mating and one to see if
       * we're being mated.
       */
      new_state.score -= (new_state.score >=  CHECKMATE-1000);
      new_state.score += (new_state.score <= -CHECKMATE+1000);
      takeback(game);

      /* Abort if we've exceeded our allowed time to think */
      if (horizon/PLY>3 && abort_search) break;

      if (new_state.score > best_score) {
         best_score = new_state.score;
         hash_move = best_move = movelist.move[move_perm[n]];
         state.depth = new_state.depth;
         state.qdepth = new_state.qdepth;

         if (best_score > alpha) {           /* New best line found */
            hash_flag = HASH_TYPE_EXACT;
            alpha = best_score;

            /* Store good moves in the killer slots, but only if:
             *  - we were not in check at the beginning of this move
             *  - the move is not a promotion (already high in the list)
             *  - the move was not a capture (already high in the list)
             */
            store_killer(game, move, depth / PLY);
            store_counter_move(game, prev_move, move);
#ifdef USE_MATE_KILLER
            if (best_score >= (CHECKMATE-1000))
               game->mate_killer[depth].m = move.m;
#endif

            /* Update and store principle variation */
            state.mainline[0] = move;
            state.size_of_variation = new_state.size_of_variation+1;
            if (new_state.size_of_variation)
               memcpy(&state.mainline[1], &new_state.mainline[0],
                     new_state.size_of_variation*sizeof(move_t));

#ifdef USE_HISTORY_HEURISTIC
            int history_score;

            game->history[player>>7][move.from][move.to]+=depth*depth/(PLY*PLY);
            history_score = game->history[player>>7][move.from][move.to];
            if (history_score > game->max_history) {
               game->max_history = history_score;
            }
#endif

#ifdef PRINCIPLE_VARIATION_IN_GAME_STRUCT
            /* Copy principle variation */
            game->principle_variation[depth+1][depth+1] = move;
            for (c=depth+2; c < game->length_of_variation[depth+2]; c++) {
               game->principle_variation[depth+1][c] =
                  game->principle_variation[depth+2][c];
            }
            game->length_of_variation[depth+1] =
               game->length_of_variation[depth+2];
#endif

            if (best_score >= local_beta) {           /* Beta cutoff */
               hash_flag = HASH_TYPE_BETA;
               branches_pruned++;
               break;
            }
         }
      }
   }
   }

   if (horizon/PLY>3 && abort_search) return state;

   /* No legal moves - either checkmate or stalemate */
   if (legal_moves == 0) {
      state.size_of_variation = 0;
      if (game->board->in_check) {
         best_score = -CHECKMATE;
      } else {
         best_score = 0;
      }
#ifdef LATE_FEW_MOVES_EXTENSION
   } else if ( (legal_moves < 3) && (horizon_distance <= PLY) &&
               !game->board->in_check && (best_score < beta)) {
      /* Research this ply if we only have a few moves and are close to the
       * horizon. Really, we should test this *before* searching it, but on
       * the other hand, we're very close to the node... so it might not be
       * that bad.
       * NB: don't do this is we're already in check or if this move
       * already generated a beta-cutoff!
       */
      state = alphabeta(game, horizon+PLY, 2*PLY, alpha, beta, in_pv);
      return state;
#endif
   }

   /* store evaluation in the transposition table */
   if (!(hash_flag & (HASH_TYPE_BETA|HASH_TYPE_EXACT)))
      best_move.m = 0;
   if (legal_moves && !abort_search)
   store_table_entry(game->transposition_table, game->board->hash, 
                     horizon_distance/PLY, best_score, hash_flag, hash_move);
   
   state.score = best_score;
   return state;
}

