#include <stdbool.h>
#include "sbr.h"
#include "move.h"
#include "board.h"
#include "bitboard.h"
#include "square.h"
#include "names.h"
#include "movegen.h"

#undef WRITE_OUTPUT_FROM_MOVE_GENERATOR
#undef RANDOM_ORDER_CAPTURES
#define QUIESCENCE_ALL_CAPTURES_RANDOM_ORDER

/* Attack bitboards: for a given square for each piece and for selected
 * occupation numbers, return all squares attacked by a piece (ie, all
 * possible moves).
 * We have row/file bitboards, diagonal bitboards, knight and king
 * bitboards.
 * We can safe quite a bit of space (important for caching more than
 * anything else) by discarding the end points on the bitboard (a piece
 * will always attack those and can't pass beyond them anyway).
 * NB: although we conceptually use rotated bitboards to store
 * row/column/diagonal occupation numbers, the bitboards in the below data
 * structure are strictly regular.
 */
static bitboard_t king_attack[64];
static bitboard_t knight_attack[64];
static bitboard_t attack_row[64][64];
static bitboard_t attack_file[64][64];
static bitboard_t attack_diagonal_a8h1[64][64];
static bitboard_t attack_diagonal_a1h8[64][64];
static bitboard_t ep_neighbours[16];

/* Discovered or indirect attacks: squares along the direction beyond the
 * first blocker. This is useful for detecting discovered attacks, X-ray
 * attacks and pins. The logical and between these bitboards and the normal
 * attack bitboards indicates the blockers.
 *  1. For a rook attack, if a blocker is a rook or queen, we have X-ray
 *     attacks of the rook through the queen and the queen through the rook.
 *  2. For a bishop, if a blocker is a queen (or bishop, but that is
 *     unlikely) then there is an X-ray attack of the bishop through the
 *     queen and of the queen through the bishop
 *  3. For a rook, if a blocker is an enemy rook, there is possibly an
 *     X-ray defence of squares behind the enemy rook. Vice versa for
 *     bishops and queens.
 *  4. If there is an 'X-ray' looking at the opposing king, we have either
 *     a pin or a discovered check.
 */
static bitboard_t xattack_row[64][64];
static bitboard_t xattack_file[64][64];
static bitboard_t xattack_diagonal_a8h1[64][64];
static bitboard_t xattack_diagonal_a1h8[64][64];

static void print_uint8_t(uint8_t b)
{
   int c;
   
   for (c=0; c<8; c++) {
      printf ("%d", (b>>c)&1);
   }
}

//#define DEBUG_DIAGONAL_BITBOARD_MAPPING
/* Compute the bitboards and other static data structures needed for move
 * generation.
 */
void generate_movement_tables(void)
{
   int row, file, square;
   int occ, bit;
   int attack, xattack;

#ifdef DEBUG_DIAGONAL_BITBOARD_MAPPING
   for (row=7; row>=0; row--) {
      /* Write sequential number of diagonal */
      for (file=0; file<8; file++) {
         int square = pack_row_column(row, file);
         int diag_nr = get_a8h1_diagonal(square);
         printf(" %02d", diag_nr);
      }
      printf("  ");
      /* The position of each square within the diagonal */
      for (file=0; file<8; file++) {
         int square = pack_row_column(row, file);
         printf(" %02d", get_a8h1_diagonal_ord(square));
      }
      printf("  ");
      /* The offset of the start of the diagonal in the rotated bitboard */
      for (file=0; file<8; file++) {
         int square = pack_row_column(row, file);
         printf(" %02d", get_a8h1_diagonal_offset(square));
      }
      printf("  ");
      /* The permutation of numbers along the bitboard */
      for (file=0; file<8; file++) {
         int square = pack_row_column(row, file);
         printf(" %02d", get_a8h1_diagonal_square(square));
      }
      printf("\n");
   }

   printf("\n");
   for (row=7; row>=0; row--) {
      /* Write sequential number of diagonal */
      for (file=0; file<8; file++) {
         int square = pack_row_column(row, file);
         int diag_nr = get_a1h8_diagonal(square);
         printf(" %02d", diag_nr);
      }
      printf("  ");
      /* The position of each square within the diagonal */
      for (file=0; file<8; file++) {
         int square = pack_row_column(row, file);
         printf(" %02d", get_a1h8_diagonal_ord(square));
      }
      printf("  ");
      /* The offset of the start of the diagonal in the rotated bitboard */
      for (file=0; file<8; file++) {
         int square = pack_row_column(row, file);
         printf(" %02d", get_a1h8_diagonal_offset(square));
      }
      printf("  ");
      /* The permutation of numbers along the bitboard */
      for (file=0; file<8; file++) {
         int square = pack_row_column(row, file);
         printf(" %02d", get_a1h8_diagonal_square(square));
      }
      printf("\n");
   }
   exit(0);
#endif

   /* Generate movement tables for the king and the knight.
    * The movement tables for the knight are relatively simple,
    * because knights can't be blocked. On the discrete grid of the chess
    * board, the possible destinations of the knight lie on a circle with
    * radius 5.
    * The king likewise can't be blocked, because it can only move one
    * square at a time.
    */
   for (square=0; square<64; square++) {
      int dx, dy;
      row = unpack_row(square);
      file = unpack_column(square);

      king_attack[square] = board_empty;
      for (dx=-1; dx<2; dx++)
         for (dy=-1; dy<2; dy++) {
            if ( (dx||dy) &&
                  (row+dx>=0 && row+dx<=7 && file+dy>=0 && file+dy<=7) ) {
               set_bitboard(&(king_attack[square]), pack_row_column(row+dx,file+dy));
            }
         }

      knight_attack[square] = board_empty;
      for (dx=-3; dx<4; dx++)
         for (dy=-3; dy<4; dy++) {
            if ( (dx*dx + dy*dy == 5) &&
                  (row+dx>=0 && row+dx<=7 && file+dy>=0 && file+dy<=7) ) {
               set_bitboard(&(knight_attack[square]), pack_row_column(row+dx,file+dy));
            }
         }
   }

   /* Generate row and file attack bitboards */
   for (file=0; file<8; file++) {
      for (occ=0; occ<256; occ++) {
         int occupied = occ & ~(1<<file);
         attack = 0;
         xattack = 0;
         /* Normal attacks */
         bit = file;
         while ( bit>0 && !(occupied&1<<bit) ) {
            attack |= 1<<(bit-1);
            bit--;
         }
         /* X-ray/indirect attacks */
         bit--;
         while ( bit>0 && !(occupied&1<<bit) ) {
            xattack |= 1<<(bit-1);
            bit--;
         }

         /* Normal attacks */
         bit = file;
         while ( bit<7 && !(occupied&1<<bit) ) {
            attack |= 1<<(bit+1);
            bit++;
         }
         /* X-ray/indirect attacks */
         bit++;
         while ( bit<7 && !(occupied&1<<bit) ) {
            xattack |= 1<<(bit+1);
            bit++;
         }

         /* store the generated bitboards */
         for(row=0; row<8; row++) {
            attack_row[pack_row_column(row, file)][(occ>>1)&63] =
                                    make_bitboard_row(attack, row);
            attack_file[pack_row_column(file, row)][(occ>>1)&63] =
                                    make_bitboard_file(attack, row);

            xattack_row[pack_row_column(row, file)][(occ>>1)&63] =
                                    make_bitboard_row(xattack, row);
            xattack_file[pack_row_column(file, row)][(occ>>1)&63] =
                                    make_bitboard_file(xattack, row);
         }
      }
   }

   /* Generate attack bitboards for the diagonals */
   for (square=0; square<64; square++) {
      for (occ=0; occ<256; occ++) {
         int diag_nr = get_a8h1_diagonal(square);
         /* Find the length of the diagonal: this is the number of bits in
          * the occupancy number that we actually care about.
          */
         int diag_length = 8 - abs(7 - diag_nr);
         int occupied = 0;

         bit = get_a8h1_diagonal_ord(square);

         /* Compute effective occupancy number: we don't care about the
          * square the piece we're computing the moves for is on (we know
          * there's a piece there) and we don't care about bits that
          * are on the extension of diagonal beyond the edge of the board.
          * We unset the bit corresponding to the current square and set
          * the bits that are off the board. This makes the code below a
          * little bit simpler.
          */
         occupied = (uint8_t)((occ & ~(1<<bit)) | ~((1<<diag_length)-1));
         attack = 0;
         xattack = 0;

         /* Scan both directions of the ray */
         /* Normal attacks */
         bit = get_a8h1_diagonal_ord(square);
         while ( bit>0 && !(occupied&1<<bit) ) {
            attack |= 1<<(bit-1);
            bit--;
         }
         /* X-ray attacks */
         bit--;
         while ( bit>0 && !(occupied&1<<bit) ) {
            xattack |= 1<<(bit-1);
            bit--;
         }

         /* Normal attacks */
         bit = get_a8h1_diagonal_ord(square);
         while ( bit<7 && !(occupied&1<<bit) ) {
            attack |= 1<<(bit+1);
            bit++;
         }
         /* X-ray attacks */
         bit++;
         while ( bit<7 && !(occupied&1<<bit) ) {
            xattack |= 1<<(bit+1);
            bit++;
         }

         /* We don't care about the edge bits; they re irrelevant. */
         attack_diagonal_a8h1[square][(occ>>1)&63] =
                     make_bitboard_a8h1_diagonal(attack, square);
         xattack_diagonal_a8h1[square][(occ>>1)&63] =
                     make_bitboard_a8h1_diagonal(xattack, square);

         /* Now we repeat this exercise for diagonals parallel to a1-h8 */
         diag_nr = get_a1h8_diagonal(square);
         /* Find the length of the diagonal: this is the number of bits in
          * the occupancy number that we actually care about.
          */
         diag_length = 8 - abs(7 - diag_nr);
         occupied = 0;

         bit = get_a1h8_diagonal_ord(square);

         /* Compute effective occupancy number: we don't care about the
          * square the piece we're computing the moves for is on (we know
          * there's a piece there) and we don't care about bits that
          * are on the extension of diagonal beyond the edge of the board.
          * We unset the bit corresponding to the current square and set
          * the bits that are off the board. This makes the code below a
          * little bit simpler.
          */
         occupied = (uint8_t)((occ & ~(1<<bit)) | ~((1<<diag_length)-1));
         attack = 0;
         xattack = 0;

         /* Scan both directions of the ray */
         /* Normal attacks */
         bit = get_a1h8_diagonal_ord(square);
         while ( bit>0 && !(occupied&1<<bit) ) {
            attack |= 1<<(bit-1);
            bit--;
         }
         /* X-ray attacks */
         bit--;
         while ( bit>0 && !(occupied&1<<bit) ) {
            xattack |= 1<<(bit-1);
            bit--;
         }

         /* Normal attacks */
         bit = get_a1h8_diagonal_ord(square);
         while ( bit<7 && !(occupied&1<<bit) ) {
            attack |= 1<<(bit+1);
            bit++;
         }
         bit++;
         while ( bit<7 && !(occupied&1<<bit) ) {
            xattack |= 1<<(bit+1);
            bit++;
         }

         /* We don't care about the edge bits; they re irrelevant. */
         attack_diagonal_a1h8[square][(occ>>1)&63] =
                     make_bitboard_a1h8_diagonal(attack, square);
         xattack_diagonal_a1h8[square][(occ>>1)&63] =
                     make_bitboard_a1h8_diagonal(xattack, square);
      }
   }

   /* Generate test bitboards for en-passant captures: neighbouring squares
    * for each file (shifted such that A4 = 0)
    */
   for (square = 0; square < 16; square++)
      ep_neighbours[square] = 0;
   for (square = 1; square < 8; square++) {
      ep_neighbours[square] |= make_bitboard_square(A4+square-1);
      ep_neighbours[8+square] |= make_bitboard_square(A5+square-1);
   }
   for (square = 0; square < 7; square++) {
      ep_neighbours[square] |= make_bitboard_square(A4+square+1);
      ep_neighbours[8+square] |= make_bitboard_square(A5+square+1);
   }

   /* Construct bitmask for "irreversible" moves, so we can check against
    * the "m" field quickly.
    * Unfortunately, we can't use static initiaisation for this!
    */
   irreversible_move.m = 0;
   irreversible_move.piece = PAWN;
   irreversible_move.is_capture = 1;
   irreversible_move.castle = 0xFF;
}

/* Simplified version of the move generator that only checks if any piece
 * belonging to the given player can reach the square indicated.
 */
bool square_is_attacked(const board_t *board, int square, int player)
{
   bitboard_t bb_where = make_bitboard_square(square);
   bitboard_t occupied;
   bitboard_t pawns;
   bitboard_t knights;
   bitboard_t bishops_or_queens;
   bitboard_t rooks_or_queens;
   bitboard_t rooks;
   bitboard_t king;
   bitboard_t bb;
   bitboard_t moves;
   bitboard_t own_movers;
   int c_index = (player != 0);
   int row_nr, col_nr, a8h1_nr, a1h8_nr;

   /* Initilize appropriate bitboards */
   own_movers = board->bbc[c_index];
   occupied = or_bitboards(board->bbc[0], board->bbc[1]);

   /* Bitboards representing the different piece types */
   pawns = and_bitboards(own_movers, board->bbp[0]);
   rooks_or_queens = and_bitboards(own_movers,
                                    or_bitboards(board->bbp[3], board->bbp[4]));
   bishops_or_queens = and_bitboards(own_movers,
                                    or_bitboards(board->bbp[2], board->bbp[4]));
   rooks = and_bitboards(own_movers, board->bbp[3]);
   knights = and_bitboards(own_movers, board->bbp[1]);
   king = and_bitboards(own_movers, board->bbp[5]);
   
   /* The idea in each of these checks is to reverse the problem: see
    * whether a rook on the requested square can move to a square holding a
    * rook (or queen). This only requires one check.
    */

   /* Check rook moves and horizontal/vertical queen moves */
   row_nr = (get_bitboard_row(occupied, unpack_row(square)) >> 1) & 63;
   col_nr = (get_bitboard_row(board->bbf, unpack_column(square)) >> 1) & 63;
   moves = or_bitboards(attack_row[square][row_nr],
                        attack_file[square][col_nr]);
   bb = and_bitboards(moves, rooks_or_queens);
   if (bb) return true;

   /* Check bishop moves and diagonal queen moves */
   a8h1_nr = (get_diagonal_a8h1_occupancy_number(board, square) >> 1) & 63;
   a1h8_nr = (get_diagonal_a1h8_occupancy_number(board, square) >> 1) & 63;
   moves = or_bitboards(attack_diagonal_a8h1[square][a8h1_nr],
                        attack_diagonal_a1h8[square][a1h8_nr]);
   bb = and_bitboards(moves, bishops_or_queens);
   if (bb) return true;

   /* Check knight and king moves in one go */
   bb = and_bitboards(knights, knight_attack[square]);
   bb = or_bitboards(bb, and_bitboards(king, king_attack[square]));
   if (bb) return true;

   /* Finally check for pawns. */
   /* Generate pawn moves. These do depend on the current player, but the
    * code is identical apart from a few sign and shift operators - so we
    * use a macro to generate these.
    */
   if (player==WHITE) {
      bb = and_bitboards(bb_where, shift_bitboard_right_up(pawns));
      if (bb) return true;
      bb = and_bitboards(bb_where, shift_bitboard_left_up(pawns));
      if (bb) return true;
   } else {
      bb = and_bitboards(bb_where, shift_bitboard_right_down(pawns));
      if (bb) return true;
      bb = and_bitboards(bb_where, shift_bitboard_left_down(pawns));
      if (bb) return true;
   }

   return false;
}


/* Helper to generate pawn moves, except double moves and en-passant moves */
#define generate_pawn_single_move(bb, disp, cap)               \
      while(promote_pawns) {                                   \
         int dest = sbr64(promote_pawns);                      \
         int pp = allowed_promotion_pieces;                    \
         int p;                                                \
         unset_bitboard(&promote_pawns, dest);                 \
         while (pp) {                                          \
            p = sbr32(pp);                                     \
            pp ^= (1<<p);                                           \
         *move = encode_promotion_move((1<<p)|player, dest + disp, dest);\
         cap; move++;                                          \
         }\
      }                                                        \
                                                               \
      while(bb) {                                              \
         int dest = sbr64(bb);                                 \
         unset_bitboard(&bb, dest);                            \
         *move = encode_normal_move(PAWN|player, dest + disp, dest);\
         cap;                                                  \
         move++;                                               \
      }

/* Helper macro to generate pawn moves.
 * Apart from a few signs and constants, the code for these is exactly the
 * same for white and for black.
 */
#define generate_pawn_moves(shift_f,shift_l,shift_r,sign,dl,dr,rank3,rank8)\
{                                                                       \
   /* First we separate the pawns into three groups: those that can move\
    * two paces, those that can promote and the rest.                   \
    * shift all pawns forward one row, check against empty squares. This\
    * gives us all pawns that can move.                                 \
    */                                                                  \
   bb = and_bitboards(empty, shift_f(pawns));                           \
   /* Keep seperate track of pawns that can do a double move or promote \
    * Only pawns that can move to the third row could move to the fourth\
    */                                                                  \
   double_pawns = and_bitboards(empty, shift_f(and_bitboards(bb, rank3)));\
                                                                        \
   /* Do not double check pawns we already check for promotion */       \
   bb = and_bitboards(bb, destination_mask);                            \
   promote_pawns = and_bitboards(rank8, bb);                            \
   bb = xor_bitboards(bb, promote_pawns);                               \
                                                                        \
   /* Now generate pawn moves */                                        \
   /* Single steps */                                                   \
   generate_pawn_single_move(bb, sign 8, (void)0);                      \
   /* Double steps */                                                   \
   double_pawns = and_bitboards(double_pawns, destination_mask);        \
   while(double_pawns) {                                                \
      int dest;                                                         \
      int ep = 0;                                                       \
                                                                        \
      dest = sbr64(double_pawns);                                       \
      /* Only set E-P square if enemy pawns are present */              \
      if (enemy_pawns & ep_neighbours[dest-A4]) ep = dest sign 8;       \
      *move = encode_doublep_move(PAWN|player,                          \
                                       dest sign 16, dest, ep);         \
      move++;                                                           \
                                                                        \
      unset_bitboard(&double_pawns, dest);                              \
   }                                                                    \
   /* Captures right */                                                 \
   bb = and_bitboards(enemy, shift_r(pawns));                           \
   bb = and_bitboards(bb, destination_mask);                            \
   promote_pawns = and_bitboards(rank8, bb);                            \
   bb = xor_bitboards(bb, promote_pawns);                               \
   generate_pawn_single_move(bb, sign dr, (*move).is_capture = 1);      \
   /* Captures left */                                                  \
   bb = and_bitboards(enemy, shift_l(pawns));                           \
   bb = and_bitboards(bb, destination_mask);                            \
   promote_pawns = and_bitboards(rank8, bb);                            \
   bb = xor_bitboards(bb, promote_pawns);                               \
   generate_pawn_single_move(bb, sign dl, (*move).is_capture = 1);      \
   /* en-passant captures, right */                                     \
   bb = and_bitboards(en_passant, shift_r(pawns));                      \
   if(bb) {                                                             \
      int dest = sbr64(bb);                                             \
      bb ^= ((bitboard_t)1) << dest;                                    \
      *move = encode_enpassant_capture(PAWN|player, dest sign dr,       \
                                                   dest, dest sign 8);  \
      /*printf("%2s x %2sep (%2s)\n", square_str[dest sign dr],           \
                           square_str[dest], square_str[dest sign 8]);*/  \
      move++;                                                           \
   }                                                                    \
   /* en-passant captures, left */                                      \
   bb = and_bitboards(en_passant, shift_l(pawns));                      \
   if(bb) {                                                             \
      int dest = sbr64(bb);                                             \
      bb ^= ((bitboard_t)1) << dest;                                    \
      /*printf("%2s x %2sep (%2s)\n", square_str[dest sign dl],           \
                           square_str[dest], square_str[dest sign 8]);*/  \
      *move = encode_enpassant_capture(PAWN|player, dest sign dl,       \
                                                   dest, dest sign 8);  \
      move++;                                                           \
   }                                                                    \
}

/* General move generation function: generate all pseudo-legal moves for
 * pieces on the board under the 'source_mask' to the 'destination_mask'
 */
static void generate_moves_mask(movelist_t *movelist, const board_t *board, bitboard_t source_mask, bitboard_t destination_mask, int player, int allowed_promotion_pieces)
{
   bitboard_t own_movers;
   bitboard_t own;
   bitboard_t enemy;
   bitboard_t enemy_attacks;
   bitboard_t occupied;
   bitboard_t empty;
   bitboard_t pawns;
   bitboard_t enemy_pawns;
   bitboard_t knights;
   bitboard_t bishops_or_queens;
   bitboard_t rooks_or_queens;
   bitboard_t rooks;
   bitboard_t king;
   bitboard_t double_pawns;
   bitboard_t promote_pawns;
   bitboard_t en_passant;
   bitboard_t castle_row;
   bitboard_t bb;
   bitboard_t bp;
   bitboard_t moves;
   bitboard_t attacked_squares;
   int square;
   int piece;
   int c_index = (player != 0);
   move_t *move;

   /* Bookkeeping: we keep a pointer to the next move in the move list, and
    * update the number of moves in the list at the end of this function
    */
   move = &(movelist->move[movelist->num_moves]);

   /* Initilize appropriate bitboards */
   en_passant = and_bitboards(board_ep, make_bitboard_square(board->ep_square));
   en_passant = and_bitboards(en_passant, destination_mask);

   own = board->bbc[c_index];
   enemy = board->bbc[1-c_index];
   enemy_attacks = 0;
   attacked_squares = 0;

   occupied = or_bitboards(own, enemy);
   empty = invert_bitboard(occupied);

   /* Now apply bitmasks: we don't generate moves for pieces that are not
    * on the source mask.
    */
   own_movers = and_bitboards(own, source_mask);

   /* Bitboards representing the different piece types */
   enemy_pawns = and_bitboards(enemy, board->bbp[0]);
   pawns = and_bitboards(own_movers, board->bbp[0]);
   rooks_or_queens = and_bitboards(own_movers,
                                    or_bitboards(board->bbp[3], board->bbp[4]));
   bishops_or_queens = and_bitboards(own_movers,
                                    or_bitboards(board->bbp[2], board->bbp[4]));
   rooks = and_bitboards(own_movers, board->bbp[3]);
   knights = and_bitboards(own_movers, board->bbp[1]);
   king = and_bitboards(own_movers, board->bbp[5]);

   /* Return if we can't generate any moves with these masks */
   if ((own_movers == 0) || ( (empty==0) && (enemy==0)))
      return;
   
   /* Generate rook moves and horizontal/vertical queen moves */
   bb = rooks_or_queens;
   while(bb) {
      int row_nr, col_nr;
      square = sbr64(bb);
      bb ^= ((bitboard_t)1) << square;
      piece = get_piece(board, square);

      /* Get bitboard representing all possible moves for this piece by
       * making use of row and file occupancy numbers.
       * Mask out the boundary bits, since they're irrelevant.
       */
      row_nr = (get_bitboard_row(occupied, unpack_row(square)) >> 1) & 63;
      col_nr = (get_bitboard_row(board->bbf, unpack_column(square)) >> 1) & 63;
      moves = or_bitboards(attack_row[square][row_nr],
                           attack_file[square][col_nr]);
      attacked_squares |= moves;
      moves = mask_bitboards(moves, own);

      /* "moves" now has all possible moves for this piece encoded, but we
       * should only generate the ones that satisfy the destination mask.
       */
      moves = and_bitboards(moves, destination_mask);

      /* Loop over all bits to read off all destination squares */
      while(and_bitboards(moves, board_all)) {
         int dest = sbr64(moves);
         moves ^= ((bitboard_t)1) << dest;

         *move = encode_normal_move(piece, square, dest);
         if (get_piece(board, dest)) move->is_capture = 1;
         move++;
      }
   }

   /* Generate bishop moves and diagonal queen moves */
   bb = bishops_or_queens;
   while(bb) {
      int a8h1_nr, a1h8_nr;
      square = sbr64(bb);
      bb ^= ((bitboard_t)1) << square;
      piece = get_piece(board, square);
      
      /* Get bitboard representing all possible moves for this piece by
       * making use of the diagonal occupancy numbers.
       * Mask out the boundary bits, since they're irrelevant.
       */
      a8h1_nr = (get_diagonal_a8h1_occupancy_number(board, square) >> 1) & 63;
      a1h8_nr = (get_diagonal_a1h8_occupancy_number(board, square) >> 1) & 63;
      moves = or_bitboards(attack_diagonal_a8h1[square][a8h1_nr],
                           attack_diagonal_a1h8[square][a1h8_nr]);
      attacked_squares |= moves;
      moves = mask_bitboards(moves, own);

      /* "moves" now has all possible moves for this piece encoded, but we
       * should only generate the ones that satisfy the destination mask.
       */
      moves = and_bitboards(moves, destination_mask);

      /* Loop over all bits to read off all destination squares */
      while(and_bitboards(moves, board_all)) {
         int dest = sbr64(moves);
         moves ^= ((bitboard_t)1) << dest;

         *move = encode_normal_move(piece, square, dest);
         if (get_piece(board, dest)) move->is_capture = 1;
         move++;
      }
   }

   /* Generate knight moves */
   bb = knights;
   while(bb) {
      square = sbr64(bb);
      bb ^= ((bitboard_t)1) << square;
      piece = get_piece(board, square);
      
      attacked_squares |= knight_attack[square];
      moves = mask_bitboards(knight_attack[square], own);

      /* "moves" now has all possible moves for this piece encoded, but we
       * should only generate the ones that satisfy the destination mask.
       */
      moves = and_bitboards(moves, destination_mask);

      /* Loop over all bits to read off all destination squares */
      while(and_bitboards(moves, board_all)) {
         int dest = sbr64(moves);
         moves ^= ((bitboard_t)1) << dest;

         *move = encode_normal_move(piece, square, dest);
         if (get_piece(board, dest)) move->is_capture = 1;
         move++;
      }
   }

   /* Generate pawn moves. These do depend on the current player, but the
    * code is identical apart from a few sign and shift operators - so we
    * use a macro to generate these.
    */
   if (player==WHITE) {
      attacked_squares |= shift_bitboard_right_up(pawns);
      attacked_squares |= shift_bitboard_left_up(pawns);
      enemy_attacks |= shift_bitboard_right_down(enemy_pawns);
      enemy_attacks |= shift_bitboard_left_down(enemy_pawns);
      generate_pawn_moves(shift_bitboard_row_up,
                          shift_bitboard_right_up,
                          shift_bitboard_left_up, 
                          -,9,7,board_rank3,board_rank8);
      castle_row = and_bitboards(board_rank1, destination_mask);
   } else {
      attacked_squares |= shift_bitboard_right_down(pawns);
      attacked_squares |= shift_bitboard_left_down(pawns);
      enemy_attacks |= shift_bitboard_right_up(enemy_pawns);
      enemy_attacks |= shift_bitboard_left_up(enemy_pawns);
      generate_pawn_moves(shift_bitboard_row_down,
                          shift_bitboard_right_down,
                          shift_bitboard_left_down,
                          +, 7, 9, board_rank6, board_rank1);
      castle_row = and_bitboards(board_rank8, destination_mask);
   }

   /* Generate (legal) king moves; we mask out any squares attacked by the
    * opponent.
    */
   //enemy_attacks = get_attack_bitboard(board, player ^ BLACK);
   /* Locate the enemy king */
#if 0
   square = sbr64(board->bbp[5]&enemy);
   enemy_attacks |= king_attack[square];
   /* Also mask out squares surrounding the enemy queen; this is relatively
    * easy to do and trivially illegal to move to.
    */
   bb = board->bbp[4]&enemy;
   while (bb) {
      square = sbr64(bb);
      bb ^= ((bitboard_t)1) << square;
      enemy_attacks |= king_attack[square];
   }
   bb = board->bbp[1]&enemy;
   while (bb) {
      square = sbr64(bb);
      bb ^= ((bitboard_t)1) << square;
      enemy_attacks |= knight_attack[square];
   }
#endif

   /* It seems that the if statement should not be necessary because there
    * is always a king on the board, but this is false: the king may be
    * outside of the origin mask and therefore be "invisible"
    */
   if(king) {
      square = sbr64(king);
      piece = get_piece(board, square);

      attacked_squares |= king_attack[square];
      moves = mask_bitboards(king_attack[square], own|enemy_attacks);

      /* Here, we could already mask out squares attacked by the enemy
       * king, pawns and knights. Sliders are a bit more tricky though.
       */

      /* "moves" now has all possible moves for this piece encoded, but we
       * should only generate the ones that satisfy the destination mask.
       */
      moves = and_bitboards(moves, destination_mask);

      /* Loop over all bits to read off all destination squares */
      while(and_bitboards(moves, board_all)) {
         int dest = sbr64(moves);
         unset_bitboard(&moves, dest);

         /* Don't generate moves that will put the king in check; this test
          * is fairly cheap here and it saves time later, especially in
          * positions where the king is threatened.
          * NB: by checking one of these squares, we actually (usually) get
          * information about some of the others, which we could cache and
          * reuse... TODO
          */
         if (square_is_attacked(board, dest, player^BLACK))
            continue;

         *move = encode_normal_move(piece, square, dest);
         if (get_piece(board, dest)) move->is_capture = 1;
         move++;
      }

      /* Castling, king side */
      bp = and_bitboards(castle_row, board_castleki);
      bb = and_bitboards(bp, empty);
      if (bb && (bb == bp)) {
         bb = or_bitboards(rooks, king);
         bb = and_bitboards(board_castlek, and_bitboards(bb, board->init));
         bp = and_bitboards(castle_row, board_castlek);
         if (bb && (bb == bp)) {
            uint8_t from = sbr64(king);
            /* make sure the King does not pass through check */
            if (!square_is_attacked(board, from+1, player^BLACK)&&
                  !square_is_attacked(board, from, player^BLACK)) {
               *move=encode_castling(KING|player, from, from+2, castlek_rook_bits);
               move++;
            }
         }
      }

      /* Castling, queen side */
      bp = and_bitboards(castle_row, board_castleqi);
      bb = and_bitboards(bp, empty);
      if (bb && (bb == bp)) {
         bb = or_bitboards(rooks, king);
         bb = and_bitboards(board_castleq, and_bitboards(bb, board->init));
         bp = and_bitboards(castle_row, board_castleq);
         if (bb && (bb == bp)) {
            uint8_t from = sbr64(king);
            /* make sure the King does not pass through check */
            if (!square_is_attacked(board, from-1, player^BLACK)&&
                  !square_is_attacked(board, from, player^BLACK)) {
               *move=encode_castling(KING|player, from, from-2, castleq_rook_bits);
               move++;
            }
         }
      }
   }

   /* Store the number of moves currently in the list */
   movelist->num_moves = (move - &(movelist->move[0]));
}

static inline bitboard_t get_super_attacks(const board_t *board, int
square, int colour)
{
   bitboard_t attack;
   bitboard_t occupied = or_bitboards(board->bbc[0], board->bbc[1]);
   uint8_t row_nr, col_nr, a8h1_nr, a1h8_nr;
   row_nr = (get_bitboard_row(occupied, unpack_row(square)) >> 1) & 63;
   col_nr = (get_bitboard_row(board->bbf, unpack_column(square)) >> 1) & 63;
   a8h1_nr = (get_diagonal_a8h1_occupancy_number(board, square) >> 1) & 63;
   a1h8_nr = (get_diagonal_a1h8_occupancy_number(board, square) >> 1) & 63;

   attack = attack_row[square][row_nr] |
            attack_file[square][col_nr] |
            attack_diagonal_a8h1[square][a8h1_nr] |
            attack_diagonal_a1h8[square][a1h8_nr] |
            (board->bbc[1-(colour>>7)]&knight_attack[square]);

   return attack;
}

/* Generate all check evasion voves */
static void inline generate_check_evasion_moves(movelist_t *movelist, const board_t *board, int colour)
{
   static const int dbl_pawn_check_row[2] = {3, 4};
   bitboard_t destination_mask = board_all;
   bitboard_t source_mask = board_all;
   bitboard_t attackers;
   int king_square;
   int n;

   king_square = sbr64(and_bitboards(board->bbc[colour>>7], board->bbp[5]));
   destination_mask = get_super_attacks(board, king_square, colour);
   destination_mask = mask_bitboards(destination_mask, board->bbc[colour>>7]);

   /* Get a list of all attackers; try to generate only legal moves out
    * of check
    */
   attackers = get_attackers(board, colour^BLACK, king_square);
   if (!is_power_of_two(attackers)) {  /* More than one attacker */
      /* Only king moves out-of-check will do */
      source_mask = board->bbp[5];
      destination_mask &= ~get_attack_bitboard(board, colour^BLACK);
   } else {
      /* Generate captures first
       * Take care that we should also generate en-passant captures if the
       * checking piece is a pawn that just made a double push!
       */
      if (board->ep_square &&
         unpack_row(king_square)==dbl_pawn_check_row[colour>>7] &&
         (board->bbp[0]&board->bbc[1-(colour>>7)]&king_attack[king_square])) {

         generate_moves_mask(movelist, board, board_all,
               make_bitboard_square(board->ep_square),
               colour, OFFICER);
      }

      generate_moves_mask(movelist, board, board_all,
            destination_mask&attackers,
            colour, OFFICER);
      destination_mask &= ~attackers;

      /* Generate king moves out-of-check */
      generate_moves_mask(movelist, board, board->bbp[5],
            destination_mask & ~get_attack_bitboard(board, colour^BLACK),
            colour, OFFICER);
      source_mask &= ~board->bbp[5];

      /* Finally, generate check interceptions. These are by definition
       * not captures.
       */
      destination_mask &= ~board->bbc[1-(colour>>7)];
   }

   if (source_mask && destination_mask)
      generate_moves_mask(movelist, board, source_mask, destination_mask,
            colour, OFFICER);
   
   /* Assign "scores" to the moves, essentially just give the first move
    * the highest score and go down from there. This is used to seed the
    * initial move ordering and preserve it when passing the move list
    * through quicksort.
    * This value has nothing to do with the score assigned to moves in the
    * course of the search process.
    */
   for (n=0; n<movelist->num_moves; n++) {
      movelist->move[n].score = 255 - n;
   }
}

/* Generate all pseudo-legal moves on the board for the specified colour */
void generate_moves(movelist_t *movelist, const board_t *board, int colour)
{
   bitboard_t destination_mask = board_all;
   bitboard_t source_mask = board_all;
   int n;

#ifdef WRITE_OUTPUT_FROM_MOVE_GENERATOR
   printf("Generating moves for player %d\n", colour);
#endif
   movelist->num_moves = 0;

   /* If the king is in check, then only moves that end in the region of
    * allowed moves for a "super piece" at the location of the king can be
    * valid. A "super piece" is a piece that moves as a knight or a queen.
    * The idea behind this is that we only need to consider moves that can
    * break the check, all other moves will be discarded anyway so why
    * generate them?
    * We don't actually have to test for this here in order to play legal
    * chess, the search algorithm will automatically cull moves that leave
    * the king in check, but this will avoid some unnecessary evaluations.
    * Note that the search function tests whether the king is in check at
    * the beginning of the move anyway in order to decide whether to tag on
    * a search extension anyway, so it doesn't cost us extra to do the test
    * here and store the result.
    */
   if (board->in_check) {
      generate_check_evasion_moves(movelist, board, colour);
      return;
   } else {

      /* Generate all moves by setting the "source" and "destination" masks
       * to the full board.
       */

      /* Test: First look at captures, then look for moves towards the
       * centre, then look at all other moves.
       * First generate queen captures, then rook, bishop, knight, pawn
       * captures.
       */
#ifdef RANDOM_ORDER_CAPTURES
      generate_moves_mask(movelist, board, board_all,
                                    board->bbc[1-(colour>>7)],
                                    colour, OFFICER);
#else
      generate_moves_mask(movelist, board, board_all,
                              board->bbp[4]&board->bbc[1-(colour>>7)],
                              colour, OFFICER);
      generate_moves_mask(movelist, board, board_all,
                              board->bbp[3]&board->bbc[1-(colour>>7)],
                              colour, OFFICER);
      generate_moves_mask(movelist, board, board_all,
                              board->bbp[2]&board->bbc[1-(colour>>7)],
                              colour, OFFICER);
      generate_moves_mask(movelist, board, board_all,
                              board->bbp[1]&board->bbc[1-(colour>>7)],
                              colour, OFFICER);
      generate_moves_mask(movelist, board, board_all,
                              board->bbp[0]&board->bbc[1-(colour>>7)],
                              colour, OFFICER);
#endif
      destination_mask &= ~board->bbc[1-(colour>>7)];

      generate_moves_mask(movelist, board, board_all,
                  (board_centre|board_xcentre)&destination_mask,
                  colour, OFFICER);
      destination_mask &= ~(board_centre|board_xcentre);
   }

   if (source_mask && destination_mask)
      generate_moves_mask(movelist, board, source_mask, destination_mask,
            colour, OFFICER);
   
   /* Assign "scores" to the moves, essentially just give the first move
    * the highest score and go down from there. This is used to seed the
    * initial move ordering and preserve it when passing the move list
    * through quicksort.
    * This value has nothing to do with the score assigned to moves in the
    * course of the search process.
    */
   for (n=0; n<movelist->num_moves; n++) {
      movelist->move[n].score = 255 - n;
   }

#ifdef WRITE_OUTPUT_FROM_MOVE_GENERATOR
   printf_bitboard(destination_mask);
   printf("Generated %d moves\n", movelist->num_moves);

   for (n=0; n<movelist->num_moves; n++) {
      printf("%s ", move_string(movelist->move[n], NULL));
   }
   printf("\n");
#endif
}

/* Generate moves for quiescence search. Generally, this is restricted to
 * captures, check evasions and queen promotions.
 */
void generate_quiescence_moves(movelist_t *movelist, const board_t *board, int colour)
{
   bitboard_t destination_mask = board_all;
   bitboard_t origin_mask = board_all;
   bitboard_t source_mask = board_all;
   int n;

#ifdef WRITE_OUTPUT_FROM_MOVE_GENERATOR
   printf("Generating moves for player %d\n", colour);
#endif

   /* Generate all moves by setting the "source" and "destination" masks to
    * the full board.
    */
   movelist->num_moves = 0;

   /* If the king is in check, then only moves that end in the region of
    * allowed moves for a "super piece" at the location of the king can be
    * valid. A "super piece" is a piece that moves as a knight or a queen.
    * The idea behind this is that we only need to consider moves that can
    * break the check, all other moves will be discarded anyway so why
    * generate them?
    * We don't actually have to test for this here in order to play legal
    * chess, the search algorithm will automatically cull moves that leave
    * the king in check, but this will avoid some unnecessary evaluations.
    * Note that the search function tests whether the king is in check at
    * the beginning of the move anyway in order to decide whether to tag on
    * a search extension anyway, so it doesn't cost us extra to do the test
    * here and store the result.
    * TODO: work out whether this is actually something we should detect
    * here, or before the move generator is called so that we read it as
    * *input* from the board struct.
    */
   if (board->in_check) {
      generate_check_evasion_moves(movelist, board, colour);
      return;
   } else {
      /* Generate only captures */
      /* FIXME: it's sufficient (according to Ed Schroeder) to only
       * consider "winning" captures or exchanges.
       * NB: the code below is flawed in that it will not consider QxP even
       * if the pawn is hanging... however, it does give a tremendous
       * speedup to the QS.
       * What we really need though is static exchange evaluation and a
       * bitboard with hanging pieces...
       */
#ifdef QUIESCENCE_ALL_CAPTURES_RANDOM_ORDER
      destination_mask = board->bbc[1-(colour>>7)];
      generate_moves_mask(movelist, board, board_all, destination_mask,
         colour, QUEEN);
#else
      /* Queen captures */
      destination_mask = board->bbp[4]&board->bbc[1-(colour>>7)];
      if (destination_mask) {
         int n;
         for (n=0; n<5; n++) {
            origin_mask = board->bbp[n];
            generate_moves_mask(movelist, board, origin_mask, destination_mask,
                  colour, QUEEN);
         }
         origin_mask = board->bbp[5];
         generate_moves_mask(movelist, board, origin_mask, destination_mask,
               colour, QUEEN);
      }

      /* Rook captures */
      destination_mask = board->bbp[3]&board->bbc[1-(colour>>7)];
      if (destination_mask) {
         int n;
         for (n=0; n<4; n++) {
            origin_mask = board->bbp[n];
            generate_moves_mask(movelist, board, origin_mask, destination_mask,
                  colour, QUEEN);
         }
         origin_mask = board->bbp[5];
         generate_moves_mask(movelist, board, origin_mask, destination_mask,
               colour, QUEEN);
      }

      /* Bishop captures */
      destination_mask = board->bbp[2]&board->bbc[1-(colour>>7)];
      if (destination_mask) {
         int n;
         for (n=0; n<3; n++) {
            origin_mask = board->bbp[n];
            generate_moves_mask(movelist, board, origin_mask, destination_mask,
                  colour, QUEEN);
         }
         origin_mask = board->bbp[5];
         generate_moves_mask(movelist, board, origin_mask, destination_mask,
               colour, QUEEN);
      }

      /* Knight captures */
      destination_mask = board->bbp[1]&board->bbc[1-(colour>>7)];
      if (destination_mask) {
         int n;
         for (n=0; n<2; n++) {
            origin_mask = board->bbp[n];
            generate_moves_mask(movelist, board, origin_mask, destination_mask,
                  colour, QUEEN);
         }
         origin_mask = board->bbp[5];
         generate_moves_mask(movelist, board, origin_mask, destination_mask,
               colour, QUEEN);
      }

      /* Pawn captures */
      destination_mask = board->bbp[0]&board->bbc[1-(colour>>7)];
      if (destination_mask) {
         int n;
         for (n=0; n<1; n++) {
            origin_mask = board->bbp[n];
            generate_moves_mask(movelist, board, origin_mask, destination_mask,
                  colour, QUEEN);
         }
         origin_mask = board->bbp[5];
         generate_moves_mask(movelist, board, origin_mask, destination_mask,
               colour, QUEEN);
      }
#endif
      /* Now generate all normal (non-capture) pawn promotions */
      destination_mask = mask_bitboards(or_bitboards(board_rank1, board_rank8),
                                 board->bbc[1-(colour>>7)]);
      destination_mask = or_bitboards(board_rank1, board_rank8);
      source_mask = and_bitboards(board->bbc[colour>>7], board->bbp[0]);
   }

   if (source_mask && destination_mask)
      generate_moves_mask(movelist, board, source_mask, destination_mask,
            colour, QUEEN);
   
   /* Assign "scores" to the moves, essentially just give the first move
    * the highest score and go down from there. This is used to seed the
    * initial move ordering and preserve it when passing the move list
    * through quicksort.
    */
   for (n=0; n<movelist->num_moves; n++) {
      movelist->move[n].score = 255 - n;
   }

#ifdef WRITE_OUTPUT_FROM_MOVE_GENERATOR
   printf_bitboard(destination_mask);
   printf("Generated %d moves\n", movelist->num_moves);

   for (n=0; n<movelist->num_moves; n++) {
      printf("%s ", move_string(movelist->move[n], NULL));
   }
   printf("\n");
#endif
}

/* Return the bitboard of attacked squares for the given side.
 * FIXME: if we need this information regularly: it is easy to get for the
 * side that is to move as a side-effect of the move generation...
 */
bitboard_t get_attack_bitboard(const board_t *board, int side)
{
   bitboard_t attackers;
   bitboard_t own;
   bitboard_t enemy;
   bitboard_t occupied;
   bitboard_t empty;
   bitboard_t pawns;
   bitboard_t knights;
   bitboard_t bishops_or_queens;
   bitboard_t rooks_or_queens;
   bitboard_t rooks;
   bitboard_t king;
   bitboard_t bb;
   int square;
   int piece;
   int c_index = (side != 0);

   own = board->bbc[c_index];
   enemy = board->bbc[1-c_index];

   occupied = or_bitboards(own, enemy);
   empty = invert_bitboard(occupied);

   /* Bitboards representing the different piece types */
   pawns = and_bitboards(own, board->bbp[0]);
   rooks_or_queens = and_bitboards(own,
                                    or_bitboards(board->bbp[3], board->bbp[4]));
   bishops_or_queens = and_bitboards(own,
                                    or_bitboards(board->bbp[2], board->bbp[4]));
   rooks = and_bitboards(own, board->bbp[3]);
   knights = and_bitboards(own, board->bbp[1]);
   king = and_bitboards(own, board->bbp[5]);

   attackers = 0;
   
   /* Return if we can't generate any moves with these masks */
   if ((own == 0) || ( (empty==0) && (enemy==0)))
      return 0;
   
   /* Generate rook moves and horizontal/vertical queen moves */
   bb = rooks_or_queens;
   while(bb) {
      int row_nr, col_nr;
      square = sbr64(bb);
      piece = get_piece(board, square);
      
      unset_bitboard(&bb, square);

      /* Get bitboard representing all possible moves for this piece by
       * making use of row and file occupancy numbers.
       * Mask out the boundary bits, since they're irrelevant.
       */
      row_nr = (get_bitboard_row(occupied, unpack_row(square)) >> 1) & 63;
      col_nr = (get_bitboard_row(board->bbf, unpack_column(square)) >> 1) & 63;
      attackers |= or_bitboards(attack_row[square][row_nr],
                           attack_file[square][col_nr])&
                           ~((bitboard_t)1<<square);
   }

   /* Generate bishop moves and diagonal queen moves */
   bb = bishops_or_queens;
   while(bb) {
      int a8h1_nr, a1h8_nr;
      square = sbr64(bb);
      piece = get_piece(board, square);
      
      unset_bitboard(&bb, square);

      /* Get bitboard representing all possible moves for this piece by
       * making use of the diagonal occupancy numbers.
       * Mask out the boundary bits, since they're irrelevant.
       */
      a8h1_nr = (get_diagonal_a8h1_occupancy_number(board, square) >> 1) & 63;
      a1h8_nr = (get_diagonal_a1h8_occupancy_number(board, square) >> 1) & 63;
      attackers |= or_bitboards(attack_diagonal_a8h1[square][a8h1_nr],
                           attack_diagonal_a1h8[square][a1h8_nr])&
                           ~((bitboard_t)1<<square);
   }

   /* Generate knight moves */
   bb = knights;
   while(bb) {
      square = sbr64(bb);
      unset_bitboard(&bb, square);
      piece = get_piece(board, square);
      
      attackers |= knight_attack[square];
   }

   /* Generate king moves */
   bb = king;
   while(bb) {
      square = sbr64(bb);
      unset_bitboard(&bb, square);
      piece = get_piece(board, square);
      
      attackers |= king_attack[square];
   }

   /* Generate pawn attacks. These do depend on the current player, but the
    * code is identical apart from a few sign and shift operators - so we
    * use a macro to generate these.
    */
   if (side==WHITE) {
      attackers |= shift_bitboard_right_up(pawns);
      attackers |= shift_bitboard_left_up(pawns);
   } else {
      attackers |= shift_bitboard_right_down(pawns);
      attackers |= shift_bitboard_left_down(pawns);
   }

   return attackers;
}

/* List all attackers for side for a given square */
void list_attackers(board_t *board, int player, int square)
{
   bitboard_t bb_where = make_bitboard_square(square);
   bitboard_t occupied;
   bitboard_t pawns;
   bitboard_t knights;
   bitboard_t bishops;
   bitboard_t queens;
   bitboard_t rooks;
   bitboard_t king;
   bitboard_t bb;
   bitboard_t moves;
   bitboard_t attackers;
   bitboard_t own_movers;
   int c_index = (player != 0);
   uint8_t row_nr, col_nr, a8h1_nr, a1h8_nr;

   /* Initilize appropriate bitboards */
   own_movers = board->bbc[c_index];
   occupied = or_bitboards(board->bbc[0], board->bbc[1]);

   /* Bitboards representing the different piece types */
   pawns = and_bitboards(own_movers, board->bbp[0]);
   queens = and_bitboards(own_movers, board->bbp[4]);
   bishops = and_bitboards(own_movers, board->bbp[2]);
   rooks = and_bitboards(own_movers, board->bbp[3]);
   knights = and_bitboards(own_movers, board->bbp[1]);
   king = and_bitboards(own_movers, board->bbp[5]);

   /* Pieces that can attack this square are pieces that can be captured by
    * a super piece -> generate all moves for a super piece on this square
    */
   row_nr = (get_bitboard_row(occupied, unpack_row(square)) >> 1) & 63;
   col_nr = (get_bitboard_row(board->bbf, unpack_column(square)) >> 1) & 63;
   a8h1_nr = (get_diagonal_a8h1_occupancy_number(board, square) >> 1) & 63;
   a1h8_nr = (get_diagonal_a1h8_occupancy_number(board, square) >> 1) & 63;

   /* attacking knights */
   attackers = and_bitboards(knights, knight_attack[square]);
   while(attackers) {
      int where = sbr64(attackers);

      printf("N%s ", square_str[where]);
      unset_bitboard(&attackers, where);
   }

   /* Attacking kings */
   attackers = and_bitboards(king, king_attack[square]);
   while(attackers) {
      int where = sbr64(attackers);

      printf("K%s ", square_str[where]);
      unset_bitboard(&attackers, where);
   }

   /* Attacking rooks */
   moves = or_bitboards(attack_row[square][row_nr],
                        attack_file[square][col_nr]);
   attackers = and_bitboards(moves, rooks);
   while(attackers) {
      int where = sbr64(attackers);

      printf("R%s ", square_str[where]);
      unset_bitboard(&attackers, where);
   }

   /* Check bishop moves */
   moves = or_bitboards(attack_diagonal_a8h1[square][a8h1_nr],
                        attack_diagonal_a1h8[square][a1h8_nr]);
   attackers = and_bitboards(moves, bishops);
   while(attackers) {
      int where = sbr64(attackers);

      printf("R%s ", square_str[where]);
      unset_bitboard(&attackers, where);
   }

   /* Queen */
   moves = or_bitboards( or_bitboards(attack_row[square][row_nr],
                                      attack_file[square][col_nr]),
                         or_bitboards(attack_diagonal_a8h1[square][a8h1_nr],
                                      attack_diagonal_a1h8[square][a1h8_nr]) );
   attackers = and_bitboards(moves, queens);
   while(attackers) {
      int where = sbr64(attackers);

      printf("Q%s ", square_str[where]);
      unset_bitboard(&attackers, where);
   }

   /* Finally check for pawns. */
   /* Generate pawn moves. These do depend on the current player, but the
    * code is identical apart from a few sign and shift operators - so we
    * use a macro to generate these.
    */
   if (player==WHITE) {
      bb = and_bitboards(bb_where, shift_bitboard_right_up(pawns));
      if (bb) printf("P%s ", square_str[sbr64(bb)]);
      bb = and_bitboards(bb_where, shift_bitboard_left_up(pawns));
      if (bb) printf("P%s ", square_str[sbr64(bb)]);
   } else {
      bb = and_bitboards(bb_where, shift_bitboard_right_down(pawns));
      if (bb) printf("P%s ", square_str[sbr64(bb)]);
      bb = and_bitboards(bb_where, shift_bitboard_left_down(pawns));
      if (bb) printf("P%s ", square_str[sbr64(bb)]);
   }
}

/* get all attackers for the target square */
bitboard_t get_attackers(const board_t *board, int player, int square)
{
   bitboard_t bb_where = make_bitboard_square(square);
   bitboard_t occupied;
   bitboard_t pawns;
   bitboard_t knights;
   bitboard_t bishops;
   bitboard_t queens;
   bitboard_t rooks;
   bitboard_t king;
   bitboard_t moves;
   bitboard_t attackers;
   bitboard_t own_movers;
   int c_index = (player != 0);
   uint8_t row_nr, col_nr, a8h1_nr, a1h8_nr;

   /* Initilize appropriate bitboards */
   own_movers = board->bbc[c_index];
   occupied = or_bitboards(board->bbc[0], board->bbc[1]);

   /* Bitboards representing the different piece types */
   pawns = and_bitboards(own_movers, board->bbp[0]);
   queens = and_bitboards(own_movers, board->bbp[4]);
   bishops = and_bitboards(own_movers, board->bbp[2]);
   rooks = and_bitboards(own_movers, board->bbp[3]);
   knights = and_bitboards(own_movers, board->bbp[1]);
   king = and_bitboards(own_movers, board->bbp[5]);

   /* Pieces that can attack this square are pieces that can be captured by
    * a super piece -> generate all moves for a super piece on this square
    */
   row_nr = (get_bitboard_row(occupied, unpack_row(square)) >> 1) & 63;
   col_nr = (get_bitboard_row(board->bbf, unpack_column(square)) >> 1) & 63;
   a8h1_nr = (get_diagonal_a8h1_occupancy_number(board, square) >> 1) & 63;
   a1h8_nr = (get_diagonal_a1h8_occupancy_number(board, square) >> 1) & 63;

   /* attacking knights */
   attackers = and_bitboards(knights, knight_attack[square]);

   /* Attacking kings */
   attackers |= and_bitboards(king, king_attack[square]);

   /* Attacking rooks and queens */
   moves = or_bitboards(attack_row[square][row_nr],
                        attack_file[square][col_nr]);
   attackers |= and_bitboards(moves, rooks|queens);

   /* Check bishop moves */
   moves = or_bitboards(attack_diagonal_a8h1[square][a8h1_nr],
                        attack_diagonal_a1h8[square][a1h8_nr]);
   attackers |= and_bitboards(moves, bishops|queens);

   /* Finally check for pawns. */
   /* Generate pawn moves. These do depend on the current player, but the
    * code is identical apart from a few sign and shift operators - so we
    * use a macro to generate these.
    */
   if (player==WHITE) {
      bitboard_t attacking_pawns = shift_bitboard_right_down(bb_where)|
                                   shift_bitboard_left_down(bb_where);
      attackers |= pawns&attacking_pawns;
   } else {
      bitboard_t attacking_pawns = shift_bitboard_right_up(bb_where)|
                                   shift_bitboard_left_up(bb_where);
      attackers |= pawns&attacking_pawns;
   }

   return attackers;
}

/* get all attackers for the target square, from either side */
bitboard_t get_all_attackers(const board_t *board, int square)
{
   bitboard_t bb_where = make_bitboard_square(square);
   bitboard_t occupied;
   bitboard_t pawns;
   bitboard_t knights;
   bitboard_t bishops;
   bitboard_t queens;
   bitboard_t rooks;
   bitboard_t king;
   bitboard_t moves;
   bitboard_t attackers;
   uint8_t row_nr, col_nr, a8h1_nr, a1h8_nr;

   /* Initilize appropriate bitboards */
   occupied = or_bitboards(board->bbc[0], board->bbc[1]);

   /* Bitboards representing the different piece types */
   pawns = board->bbp[0];
   queens = board->bbp[4];
   bishops = board->bbp[2];
   rooks = board->bbp[3];
   knights = board->bbp[1];
   king = board->bbp[5];

   /* Pieces that can attack this square are pieces that can be captured by
    * a super piece -> generate all moves for a super piece on this square
    */
   row_nr = (get_bitboard_row(occupied, unpack_row(square)) >> 1) & 63;
   col_nr = (get_bitboard_row(board->bbf, unpack_column(square)) >> 1) & 63;
   a8h1_nr = (get_diagonal_a8h1_occupancy_number(board, square) >> 1) & 63;
   a1h8_nr = (get_diagonal_a1h8_occupancy_number(board, square) >> 1) & 63;

   /* attacking knights */
   attackers = and_bitboards(knights, knight_attack[square]);

   /* Attacking kings */
   attackers |= and_bitboards(king, king_attack[square]);

   /* Attacking rooks and queens */
   moves = or_bitboards(attack_row[square][row_nr],
                        attack_file[square][col_nr]);
   attackers |= and_bitboards(moves, rooks|queens);

   /* Check bishop moves */
   moves = or_bitboards(attack_diagonal_a8h1[square][a8h1_nr],
                        attack_diagonal_a1h8[square][a1h8_nr]);
   attackers |= and_bitboards(moves, bishops|queens);

   /* Finally check for pawns. */
   /* Generate pawn moves. These do depend on the current player, but the
    * code is identical apart from a few sign and shift operators - so we
    * use a macro to generate these.
    */
   attackers |= shift_bitboard_right_up(bb_where)&pawns&board->bbc[1];
   attackers |= shift_bitboard_left_up(bb_where)&pawns&board->bbc[1];
   attackers |= shift_bitboard_right_down(bb_where)&pawns&board->bbc[0];
   attackers |= shift_bitboard_left_down(bb_where)&pawns&board->bbc[0];

   return attackers;
}

/* Returns the index in the movelist of a valid moves of the move
 * corresponding to the given origin/destination. Intended for player
 * interaction.
 * Returns -1 if no moves in the list match (ie, the requested move is
 * invalid). In the case of a promotion, there will be more than one move
 * that matches.
 */
int validate_move(const movelist_t *movelist, int source, int dest)
{
   int n;
   if (!movelist)
      return -1;

   for (n=0; n<movelist->num_moves; n++) {
      if (get_origin(movelist->move[n]) == source &&
          get_destination(movelist->move[n]) == dest) {
         return n;
      }
   }
   return -1;
}
