#include <stdio.h>
#include "config.h"
#include "square.h"
#include "board.h"
#include "game.h"
#include "hashtable.h"
#include "pawn_table.h"

#define PIECESTRING   " NBRQK"

static void (*default_iteration_output)(const char *, ...) = NULL;
static void (*default_uci_output)(const char *, ...) = NULL;

void playgame_init(void)
{
   default_iteration_output = printf;
   default_uci_output = NULL;
}

void playgame_shutdown(void)
{
}

void set_default_output_function(void (*func)(const char *, ...))
{
   default_iteration_output = func;
}

void set_uci_output_function(void (*func)(const char *, ...))
{
   default_uci_output = func;
}

gamestate_t *create_game(void)
{
   return calloc(1, sizeof(gamestate_t));
}

/* Initialize the game datastructure to correspond to a new game */
void start_new_game(gamestate_t *game)
{
   /* Load the starting position */
   game->max_moves = 1000;
   game->score = calloc(game->max_moves, sizeof *game->score);
   game->fifty_counter = calloc(game->max_moves, sizeof(*game->fifty_counter));
   game->move_list = calloc(game->max_moves, sizeof *game->move_list);
   game->board_list = calloc(game->max_moves, sizeof *game->board_list);
   game->board = game->board_list;

   game->transposition_table = create_hash_table(HASH_TABLE_SIZE);
   game->qs_table = create_hash_table(HASH_TABLE_SIZE/64);
   game->pawn_structure = create_pawn_hash_table(PAWN_TABLE_SIZE);

   clear_board(game->board);
   begin_board(game->board);

   game->player = WHITE;
   
   game->moves_played = 0;
   game->last_move = 0;

   game->output_iteration = default_iteration_output;
   game->uci_output = default_uci_output;
}

/* Frees the memory used by a gamestate */
void end_game(gamestate_t *game)
{
   if (game) {
      free(game->move_list);
      free(game->board_list);
      free(game->score);
      free(game->fifty_counter);
      destroy_hash_table(game->transposition_table);
      destroy_hash_table(game->qs_table);
      destroy_pawn_hash_table(game->pawn_structure);
      free(game);
   }
}
