#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>

#include "movegen.h"
#include "game.h"
#include "inline/game.h"
#include "hashtable.h"
#include "fen.h"
#include "computer.h"
#include "alphabeta.h"
#include "names.h"
#include "timer.h"

#ifdef __LP64__
#define ARCHSTR "(x86_64)"
#elif defined __i386__
#define ARCHSTR "(i386)"
#elif defined POWERPC
#define ARCHSTR "(powerpc)"
#else
#define ARCHSTR "(unknown)"
#endif 

char *chomp(char *s)
{
   char *p;
   p = strstr(s, "\n");
   if (p) *p = '\0';
   return s;
}

static bool streq(const char *s1, const char *s2)
{
   return strcmp(s1, s2) == 0;
}

static FILE *f = NULL;
static char *buf;

static void log_engine_output(const char *msg, ...)
{
   va_list ap;
   va_start(ap, msg);
   vsnprintf(buf, 4095, msg, ap);
   va_end(ap);

   if (f)
      fprintf(f, ">%s", buf);
}

/* Play a sequence of moves from the initial position */
void replay_game(gamestate_t *game, char *moves)
{
   movelist_t movelist;
   int from, to, n;
   char *s;
   /* First rewind to the beginning if not there yet */
   while (game->moves_played)
      takeback(game);

   /* Clear the repetition hash table - it will be rebuild as we play
    * through the game.
    */
   memset(game->repetition_hash_table, 0, sizeof game->repetition_hash_table);
   game->repetition_hash_table[game->board->hash&0xFFFF] = 1;

   s = moves;
   while(s && *s) {
      generate_moves(&movelist, game->board, game->player);
      while (*s == ' ') s++;
      if (*s) {
         move_t move;
         int col = *s - 'a'; s++;
         int row = *s - '1'; s++;
         from = pack_row_column(row, col);
         col = *s - 'a'; s++;
         row = *s - '1'; s++;
         to = pack_row_column(row, col);

         n = validate_move(&movelist, from, to);
         if (n<0) {
            int k;
            for (k=0; k<movelist.num_moves; k++) {
               fprintf(stderr, "%s ", move_string(movelist.move[k], NULL));
            }
            fprintf(stderr, "\n");
            break;
         }
         move.m = movelist.move[n].m;
         if (is_promotion(move)) {
            switch (*s) {
               case 'q':
                  move.tpiece = (move.tpiece & BLACK)|QUEEN;
                  break;
               case 'r':
                  move.tpiece = (move.tpiece & BLACK)|ROOK;
                  break;
               case 'n':
                  move.tpiece = (move.tpiece & BLACK)|KNIGHT;
                  break;
               case 'b':
                  move.tpiece = (move.tpiece & BLACK)|BISHOP;
                  break;
            }
            s++;
         }
         playmove(game, move);
      }
   }
}

int main(void)
{
   gamestate_t *game = NULL;
   char input[65536];
   const char *lower_piece_str[] = {
      "p", "n", "b", "r", "q", "k"
   };

   buf = malloc(4096);

   playgame_init();
   construct_inverse_diagonal_maps();
   generate_movement_tables();

   //set_default_output_function(log_engine_output);
   set_default_output_function(NULL);
   set_uci_output_function(printf);

   /* Turn off buffering for stdout */
   setvbuf(stdout, NULL, _IONBF, 0);

   /* Write log output (for debugging purposes) */
   f = fopen("uci.log", "a");

   while (true) {
      input[0] = '\0';
      if (!fgets(input, sizeof input, stdin))
         break;
      chomp(input);
      fprintf(f, "< %s\n", input);
      if (streq(input, "uci")) {
         printf("id name Jazz %s\n", ARCHSTR);
         printf("id author Evert Glebbeek\n");
         //printf("option name OwnBook type check default true\n");
         printf("uciok\n");
      } else if (streq(input, "isready")) {
         printf("readyok\n");
      } else if (streq(input, "ucinewgame")) {
         if (game) {
            end_game(game);
         }
         game = create_game();
         start_new_game(game);
      } else if (strstr(input, "position")) {
         char *p = strstr(input, " ");
         while (p && *p && p[0]==' ') p++;

         if (!game) {
            game = create_game();
            start_new_game(game);
         }

         if (strstr(p, "startpos") == p) {   /* Game from start position */
            /* We need to return to the first node in the game struct and
             * then replay all the moves from the beginning to the current
             * position.
             */
            p = strstr(p, "moves");
            if (p && *p) p+=6;
            replay_game(game, p);
         } else if (strstr(p, "fen") == p) { /* Game from FEN string */
            p+=4;
            setup_fen_position(game, p);

            p = strstr(p, "moves");
            if (p && *p) p+=6;
            replay_game(game, p);
         }
      } else if (strstr(input, "go")) {
         move_t move;
         const char *p = "";
         int depth = 40;
         char *s;

         /* Set defaults */
         set_time_per_move(game, 5000);
         game->movestogo = 0;
         game->time_inc[0] = game->time_inc[1] = 0;

         /* parse options */
         if (s = strstr(input, "ponder")) {   /* Can't ponder */
            continue;
         }
         if (s = strstr(input, "movetime")) {
            int time_per_move = 5000;
            sscanf(s+9, "%d", &time_per_move);
            set_time_per_move(game, time_per_move);
         }
         if (s = strstr(input, "movetogo")) {
            sscanf(s+9, "%d", &game->movestogo);
         }
         if (s = strstr(input, "infinite")) {
            set_infinite_time(game);
         }
         if (s = strstr(input, "wtime")) {
            sscanf(s+6, "%d", &game->time_left[0]);
            set_time_for_game(game);
         }
         if (s = strstr(input, "btime")) {
            sscanf(s+6, "%d", &game->time_left[1]);
            set_time_for_game(game);
         }
         if (s = strstr(input, "winc")) {
            sscanf(s+5, "%d", &game->time_inc[0]);
         }
         if (s = strstr(input, "binc")) {
            sscanf(s+5, "%d", &game->time_inc[1]);
         }
         if (s = strstr(input, "mate")) {  /* Just do a normal N-ply search instead */
            sscanf(s+5, "%d", &depth);
            set_infinite_time(game);
         }
         if (s = strstr(input, "depth")) {
            sscanf(s+6, "%d", &depth);
            set_infinite_time(game);
         }

         computer_play(game, depth);

         move.m = game->move_list[game->moves_played-1].m;
         if (is_promotion(move)) {
            p = lower_piece_str[piece_index[move.tpiece & PIECE]];
         }
         printf("bestmove %s%s%s\n", square_str[move.from],
               square_str[move.to], p);
      } else if (streq(input, "quit")) {
         exit(0);
      }
   }

   return 0;
}

