#include <stdio.h>
#include <math.h>
#include <allegro5/allegro.h>
#include <allegro5/allegro_image.h>
#include <allegro5/allegro_ttf.h>
#include <allegro5/allegro_native_dialog.h>

#include "sbr.h"
#include "genrand.h"
#include "movegen.h"
#include "names.h"
#include "game.h"
#include "inline/game.h"
#include "fen.h"
#include "computer.h"
#include "alphabeta.h"
#include "see.h"
#include "pawns.h"
#include "timer.h"

#ifndef M_PI
#define M_PI 3.14159265
#endif

#define TEXT_BUFFER_SIZE  100
#define TEXT_OUTPUT_LINES  10
#define ITERATION_FONT_SIZE 12

#define PROGRAM_NAME "Jazz"

#define DIALOG_FLAGS_SAVE_EPD    0x0001
#define DIALOG_FLAGS_LOAD_EPD    0x0002
#define DIALOG_FLAGS_SAVE_PGN    0x0004
#define DIALOG_FLAGS_LOAD_PGN    0x0008
#define DIALOG_FLAGS_SAVE        0x0005
#define DIALOG_FLAGS_LOAD        0x000A
#define DIALOG_FLAGS_EPD         0x0003
#define DIALOG_FLAGS_PGN         0x000C
typedef struct {
   ALLEGRO_NATIVE_DIALOG *file_dialog;
   ALLEGRO_EVENT_SOURCE event_source;
   ALLEGRO_THREAD *thread;
   ALLEGRO_DISPLAY *display;
   int flags;
} async_dialog_t;

/* To communicate from a separate thread, we need a user event. */
#define EVENT_FILE_DIALOG_CLOSE   ALLEGRO_GET_EVENT_TYPE('e', 'N', 'F', '1')

static async_dialog_t *file_dialog = NULL;
static const ALLEGRO_PATH *last_path = NULL;

static ALLEGRO_FONT *figurine_font;
static ALLEGRO_FONT *font;
static ALLEGRO_BITMAP *figurine_sheet;
static ALLEGRO_BITMAP *figurines[2][6];
static ALLEGRO_BITMAP *square[2];
static ALLEGRO_BITMAP *side_icon[2] = {NULL, NULL};
static ALLEGRO_BITMAP *cursor;

static ALLEGRO_DISPLAY *moves_display;
static ALLEGRO_DISPLAY *board_display;
static ALLEGRO_DISPLAY *iteration_display;
static ALLEGRO_DISPLAY *key_display;
static int board_x = 0, board_y = 0;

static int player_select = -1;
static int player_cursor = E4;

static int display_timer_counter;

/* Our thread to show the native file dialog. */
static void *async_file_dialog_thread_func(ALLEGRO_THREAD *thread, void *arg)
{
   async_dialog_t *data = arg;
   ALLEGRO_EVENT event;
   (void)thread;

   /* We need to set the current display for this thread becuse
    * al_show_native_file_dialog() shows the dialog on the current window.
    */
   al_set_current_display(data->display);

   /* The next line is the heart of this example - we display the
    * native file dialog.
    */
   al_show_native_file_dialog(data->file_dialog);

   /* We emit an event to let the main program now that the thread has
    * finished.
    */
   event.user.type = EVENT_FILE_DIALOG_CLOSE;
   al_emit_user_event(&data->event_source, &event, NULL);

   return NULL;
}

/* Function to start the new thread. */
async_dialog_t *spawn_async_file_dialog(const ALLEGRO_PATH *initial_path, int flags)
{
   async_dialog_t *data = malloc(sizeof *data);

   if (flags & DIALOG_FLAGS_LOAD) {
      data->file_dialog = al_create_native_file_dialog(
            initial_path, "Open EPD/FEN/PGN file", ".;*.pgn;*.epd;*.fen",
            ALLEGRO_FILECHOOSER_FILE_MUST_EXIST);
   } else {
      data->file_dialog = al_create_native_file_dialog(
            initial_path, "Save EPD file", ".;*.epd",
            ALLEGRO_FILECHOOSER_SAVE);
   }
   al_init_user_event_source(&data->event_source);
   data->display = al_get_current_display();
   data->thread = al_create_thread(async_file_dialog_thread_func, data);
   data->flags = flags;

   al_start_thread(data->thread);

   return data;
}


void stop_async_dialog(async_dialog_t *data)
{
   if (data) {
      al_destroy_thread(data->thread);
      al_destroy_user_event_source(&data->event_source);
      if (data->file_dialog) al_destroy_native_dialog(data->file_dialog);
      free(data);
   }
}

void load_figurine_set(int figurine_set)
{
   int size = 40;
   int x, y;

   al_set_current_display(board_display);
   if (!figurine_sheet)
      return;

   for (y=0; y<2; y++) {
      for (x=0; x<6; x++) {
         /* Allegro 4.9 seems to do something odd when it generates sub
          * bitmaps, because performance is really horrible if we use
          * them directly... this is a work-around until the problem is
          * fixed.
          */
         ALLEGRO_BITMAP *bmp = al_create_sub_bitmap(figurine_sheet, 
               x*size, (y+2*figurine_set)*size, size, size);
         if (figurines[y][x]) al_destroy_bitmap(figurines[y][x]);
         figurines[y][x] = bmp;
      }
   }

   for (y = 0; y<2; y++) {
      al_destroy_bitmap(side_icon[y]);
      side_icon[y] = al_clone_bitmap(figurines[y][1]);
   }
}

void draw_board(gamestate_t *game)
{
   static char title[256];
   ALLEGRO_COLOR white = al_map_rgba_f(1, 1, 1, 1);
   double score;
   int x, y;

   if (al_get_current_display() != board_display)
      al_set_current_display(board_display);
   al_clear_to_color(white);

   score = (game->board->material[0] - game->board->material[1])/100.0;
   score = game->score[game->moves_played]/100.0;
   if (game->player == BLACK) {
      snprintf(title, 255, PROGRAM_NAME" [ Black to play %.2f]", -score);
   } else {
      snprintf(title, 255, PROGRAM_NAME" [ White to play %.2f]", score);
   }
   al_set_window_title(title);
   al_set_display_icon(side_icon[game->player == BLACK]);
   //al_set_display_icon(figurines[game->player == BLACK][1]);

   for (y=0; y<8; y++) {
      for (x=0; x<8; x++) {
         int where = pack_row_column(y, x);
         int piece = get_piece(game->board, where);
         int col = (piece&BLACK) >> 7;
         int draw_x = board_x + x*40;
         int draw_y = board_y + (7-y)*40;
         bool blink_piece = false;

         /* Blink highlighted pieces */
         if ( (player_select == where) )// && ( (display_timer_counter/4)&1) )
            blink_piece = true;

         /* Draw background tile */
         al_draw_bitmap(square[1-((x^y)&1)], draw_x, draw_y, 0);

         /* Highlight cursor */
         if (player_cursor == where) {
            al_draw_bitmap(cursor, draw_x, draw_y, 0);
         }

         /* Piece */
         if (piece) {
            piece = piece_index[piece&PIECE];
            if (blink_piece) {
               float f = (display_timer_counter&31)/32.;
               f = cos(f*M_PI);
               al_set_blender(ALLEGRO_ALPHA, ALLEGRO_INVERSE_ALPHA,
                     al_map_rgba_f(1,1,1, f*f));
            }
            al_draw_bitmap(figurines[col][piece], draw_x, draw_y, 0);
            if (blink_piece)
               al_set_blender(ALLEGRO_ALPHA, ALLEGRO_INVERSE_ALPHA, white);
         }
      }
   }

   al_flip_display();
}

void show_move_list(gamestate_t *game)
{
   ALLEGRO_COLOR white = al_map_rgba_f(1, 1, 1, 1);
   int x, y=0, n;

   if (al_get_current_display() != moves_display)
      al_set_current_display(moves_display);
   al_clear_to_color(white);

   for(n=0; n<game->moves_played; n++) {
      /* Odd plies should normally have been played by black - unless we
       * changed the player, or loaded a position which had "black to play"
       */
      int first_move = get_move_player(game->move_list[0])>>7;
      int w = (n&1) ^ first_move;
      int dx = 0;

      x = w * 80 + 10;
      y = ((n+first_move)/2)*16;
      al_draw_textf(figurine_font, 8, y, 0, "% 3d", (n+first_move)/2+1);
      dx = 32;

      if (!(n&1)) {
         al_draw_textf(figurine_font, x+dx, y, 0, "%s",
                     move_string(game->move_list[n], NULL));
      } else {
         al_draw_textf(figurine_font, x+dx, y, 0, "%s",
                     move_string(game->move_list[n], NULL));
      }
   }

   //printf("%d %d\n", player_is_mate(game), player_in_check(game, game->player));
   if (player_is_mate(game)) {
      y+=16;
      x = 24;
      if (player_in_check(game, game->player)) {
         al_draw_textf(figurine_font, x, y, 0, "#");
         al_set_window_title("Jazz - checkmate");
      } else {
         al_draw_textf(font, x, y, 0, "=");
         al_set_window_title("Jazz - stalemate");
      }

   }

   al_flip_display();
}

static char **line = NULL;
static int cur_line = 0;
static int last_line = 0;
static int delta_line = 0;
static bool new_line = false;
static int text_window_size = TEXT_OUTPUT_LINES;

static void draw_iteration_window(void)
{
   ALLEGRO_STATE allegro_state;
   int n;
   int first;

   if (!line)
      return;

   al_store_state(&allegro_state, ALLEGRO_STATE_ALL);
   al_set_current_display(iteration_display);
   al_clear_to_color(al_map_rgb_f(1,1,1));
   al_set_blender(ALLEGRO_ALPHA, ALLEGRO_INVERSE_ALPHA, al_map_rgb(0,0,0));
   text_window_size = al_get_display_height()/(ITERATION_FONT_SIZE+2);
   first = last_line - text_window_size;
   if (first < 0) first = 0;
   for (n=0; n < text_window_size; n++) {
      int k = (first + n + TEXT_BUFFER_SIZE - delta_line) % TEXT_BUFFER_SIZE;

      if (line[k])
         al_draw_text(font, 1, (ITERATION_FONT_SIZE+2)*n, 0, line[k]);
   }
   al_flip_display();
   al_restore_state(&allegro_state);
}

static void iteration_display_message(const char *msg, ...)
{
   char *s = malloc(4096);
   char *eol;
   va_list ap;
   va_start(ap, msg);
   vsnprintf(s, 4095, msg, ap);
   va_end(ap);

   printf("%s", s);

   if (!line) {
      line = calloc(TEXT_BUFFER_SIZE, sizeof *line);
   }

   /* Clear the line if needed */
   if (new_line && line[cur_line]) {
      line[cur_line][0] = '\0';
   }

   eol = strchr(s, '\n');
   if (eol) *eol = '\0';
   new_line = (eol != NULL);

   /* Append string to existing text */
   if (!line[cur_line]) {
      line[cur_line] = s;
   } else {
      char *ss = strdup(line[cur_line]);
      snprintf(line[cur_line], 4095, "%s%s", ss, s);
      line[4095] = '\0';
      free(ss);
   }

   if (new_line) {
      cur_line = (cur_line+1) % TEXT_BUFFER_SIZE;
      last_line++;
      draw_iteration_window();
   }
}

int main(void)
{
   gamestate_t *game = NULL;
   movelist_t movelist;
   ALLEGRO_EVENT_QUEUE *queue;
   ALLEGRO_TIMER *display_timer;
   int n, x, y;
   int max_figurine_set = 0;
   int figurine_set = 1;
   bool redraw = true;
   bool done = false;

   if (!al_init()) {
      fprintf(stderr, "Couldn't initialise Allegro\n");
      return 1;
   }
   al_install_mouse();
   al_install_keyboard();
   al_init_image_addon();
   al_init_font_addon();
   al_init_ttf_addon();

   /* Seed random number generator */
   sgenrand(time(NULL));

   /* The *main* display is actually the board_display, but we create that
    * last because we also want it to be the key window.
    */

   /* Create display to display the list of moves played so far */
   moves_display = al_create_display(200, 320);
   if (!moves_display) {
      fprintf(stderr, "Can't open display\n");
      return 1;
   }
   al_get_window_position(moves_display, &x, &y);
   al_clear_to_color(al_map_rgb_f(1,1,1));
   al_flip_display();
   al_set_window_title(PROGRAM_NAME);

   /* Messages */
   al_set_new_window_position(x-324, y+344);
   //al_set_new_display_flags(al_get_new_display_flags()^ALLEGRO_NOFRAME);
   al_set_new_display_flags(al_get_new_display_flags()^ALLEGRO_RESIZABLE);
   iteration_display = al_create_display(524, (ITERATION_FONT_SIZE+2)*TEXT_OUTPUT_LINES);
   al_set_new_display_flags(al_get_new_display_flags()^ALLEGRO_RESIZABLE);
   //al_set_new_display_flags(al_get_new_display_flags()^ALLEGRO_NOFRAME);
   al_clear_to_color(al_map_rgb_f(1,1,1));
   al_flip_display();
   al_set_window_title(PROGRAM_NAME" - Iterations");


   /* Main (board) display */
   al_set_new_window_position(x-324, y);
   board_display = al_create_display(320, 320);
   if (!board_display) {
      fprintf(stderr, "Can't open display\n");
      return 1;
   }
   al_set_window_title("Jazz");

   key_display = board_display;

   memset(figurines, 0, sizeof figurines);
   square[0] = al_load_bitmap("gfx/ws.png");
   square[1] = al_load_bitmap("gfx/bs.png");
   figurine_sheet = al_load_bitmap("gfx/figurines.png");
   cursor = al_load_bitmap("gfx/indicator.png");

   if (!figurine_sheet) {
      figurines[0][0] = al_load_bitmap("gfx/wp.png");
      figurines[0][1] = al_load_bitmap("gfx/wn.png");
      figurines[0][2] = al_load_bitmap("gfx/wb.png");
      figurines[0][3] = al_load_bitmap("gfx/wr.png");
      figurines[0][4] = al_load_bitmap("gfx/wq.png");
      figurines[0][5] = al_load_bitmap("gfx/wk.png");

      figurines[1][0] = al_load_bitmap("gfx/bp.png");
      figurines[1][1] = al_load_bitmap("gfx/bn.png");
      figurines[1][2] = al_load_bitmap("gfx/bb.png");
      figurines[1][3] = al_load_bitmap("gfx/br.png");
      figurines[1][4] = al_load_bitmap("gfx/bq.png");
      figurines[1][5] = al_load_bitmap("gfx/bk.png");

      for (n=0; n<6; n++) {
         if (!figurines[0][n] || !figurines[1][n]) {
            fprintf(stderr, "Can't load figurines\n");
            return 1;
         }
         al_convert_mask_to_alpha(figurines[0][n], al_map_rgb_f(1, 0, 1));
         al_convert_mask_to_alpha(figurines[1][n], al_map_rgb_f(1, 0, 1));
      }
      side_icon[0] = al_clone_bitmap(figurines[0][1]);
      side_icon[1] = al_clone_bitmap(figurines[1][1]);
   } else {
      max_figurine_set = al_get_bitmap_height(figurine_sheet)/80 - 1;
      if (figurine_set > max_figurine_set)
         figurine_set = max_figurine_set;

      load_figurine_set(figurine_set);
   }
   font = al_load_font("gfx/DejaVuSansMono.ttf", ITERATION_FONT_SIZE, ALLEGRO_TTF_NO_KERNING);
   //font = al_load_font("gfx/monaco.ttf", ITERATION_FONT_SIZE, ALLEGRO_TTF_NO_KERNING);
   if (!font) {
      fprintf(stderr, "Can't load font\n");
      exit(0);
   }

   figurine_font = al_load_bitmap_font("gfx/figurines-font.png");
   if (!figurine_font) {
      fprintf(stderr, "Can't load figurine figurine_font\n");
      exit(0);
   }

   printf("Initialising Jazz engine\n");
   playgame_init();
   construct_inverse_diagonal_maps();
   generate_movement_tables();

   /* Redirect output to our own window rather than the terminal */
   set_default_output_function(iteration_display_message);

   printf("Initialising new game\n");
   game = create_game();
   start_new_game(game);
   generate_moves(&movelist, game->board, game->player);

   display_timer = al_install_timer(1.0 / 40.0);
   queue = al_create_event_queue();
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)al_get_mouse_event_source());
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)al_get_keyboard_event_source());
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)board_display);
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)moves_display);
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)iteration_display);
   al_register_event_source(queue, (ALLEGRO_EVENT_SOURCE *)display_timer);
   al_start_timer(display_timer);
   display_timer_counter = 0;

   draw_board(game);
   redraw = false;

   while (!done) {
      ALLEGRO_EVENT event;

      if (redraw && al_event_queue_is_empty(queue)) {
         draw_board(game);
         show_move_list(game);
         draw_iteration_window();
         redraw = false;

#if 0
         pawn_structure_t *ps = evaluate_pawn_structure(game);
         if (ps) {
            printf("Pawn structure:\n");
            int n;
            for (n=0; n<2; n++) {
               printf("%s\n", n?"Black":"White");
               printf("Outposts:\n");
               printf_bitboard(ps->outposts[n]);
               printf("Strong squares:\n");
               printf_bitboard(ps->strong_squares[n]);
            }
            
         }
#endif

         //printf("0x%016llxll\n", game->board->hash);
         //printf("%d %d\n", count_repetition(game), draw_by_repetition(game));
#if 0
         bitboard_t white_attacks = get_attack_bitboard(game->board, WHITE);
         bitboard_t black_attacks = get_attack_bitboard(game->board, BLACK);
         bitboard_t occupied = game->board->bbc[0]|game->board->bbc[1];
         bitboard_t defended_pieces = occupied&white_attacks&black_attacks;
         bitboard_t attacked_pieces = 0;

         attacked_pieces |= game->board->bbc[0] & black_attacks;
         attacked_pieces |= game->board->bbc[1] & white_attacks;
         printf("White attacks:\n");
         printf_bitboard( white_attacks );
         printf("Black attacks:\n");
         printf_bitboard( black_attacks );
         printf("Defended pieces:\n");
         printf_bitboard( defended_pieces );
         printf("Hanging pieces:\n");
         printf_bitboard( attacked_pieces^defended_pieces );
#endif
#undef SHOW_SEE_CAPTURES
#ifdef SHOW_SEE_CAPTURES
         int n;
         for (n=0; n<movelist.num_moves; n++) {
            if (is_capture(movelist.move[n])) {
               printf("%s %d\n", move_string(movelist.move[n], NULL), 
                     static_exchange_evaluation(game->board, movelist.move[n]));
            }
         }
#endif
         //printf("King safety: %d\n", evaluate_king_safety(game));
      }

      al_wait_for_event(queue, &event);
      switch (event.type) {
         case ALLEGRO_EVENT_DISPLAY_RESIZE:
            al_acknowledge_resize(event.display.source);
         case ALLEGRO_EVENT_DISPLAY_EXPOSE:
            redraw = true;
            break;

         case ALLEGRO_EVENT_DISPLAY_CLOSE:
            done = true;
            break;

         case ALLEGRO_EVENT_DISPLAY_SWITCH_IN:
            key_display = event.display.source;
            break;

         case ALLEGRO_EVENT_DISPLAY_SWITCH_OUT:
            key_display = NULL;
            break;

         case ALLEGRO_EVENT_MOUSE_AXES:
            if (key_display == iteration_display && event.mouse.dz) {
               int nd = delta_line + event.mouse.dz;
               if (nd <0) nd = 0;
               if (nd >= TEXT_BUFFER_SIZE) nd = TEXT_BUFFER_SIZE-1;
               if (nd > last_line-text_window_size)
                  nd = last_line-text_window_size;
               if (nd != delta_line) {
                  delta_line = nd;
                  redraw = true;
               }
            }
            break;

         case ALLEGRO_EVENT_MOUSE_BUTTON_UP:
            if (key_display == board_display) {
            player_cursor = pack_row_column((8*40-event.mouse.y)/40,
                                             event.mouse.x/40);
            if (player_select == -1) {
               player_select = player_cursor;
               display_timer_counter = 0;
            } else {
               /* Confirm that the player has entered a valid move */
               int n = validate_move(&movelist, player_select, player_cursor);

               if (n == -1)
                  n = validate_move(&movelist, player_cursor, player_select);
               if (n > -1) {
                  playmove_was_ok(game, movelist.move[n]);
                  generate_moves(&movelist, game->board, game->player);
               }
               player_select = -1;
            }
            redraw = true;
            }
            break;
         
         case ALLEGRO_EVENT_KEY_DOWN:
         case ALLEGRO_EVENT_KEY_REPEAT:
            switch (event.keyboard.keycode) {
               case ALLEGRO_KEY_ESCAPE:
                  done = true;
                  break;
               case ALLEGRO_KEY_F5:             /* Load FEN/PGN/EPD file */
                  if (!file_dialog) {
                     file_dialog = spawn_async_file_dialog(last_path, DIALOG_FLAGS_LOAD);
                     al_register_event_source(queue, &file_dialog->event_source);
                  }
                  break;
               case ALLEGRO_KEY_F6:             /* Load FEN/PGN/EPD file */
                  if (!file_dialog) {
                     file_dialog = spawn_async_file_dialog(last_path, DIALOG_FLAGS_SAVE);
                     al_register_event_source(queue, &file_dialog->event_source);
                  }
                  break;
               case ALLEGRO_KEY_DELETE:         /* New game */
                  /* FIXME: this leaks memory, because the internal
                   * pointers in the game struct are not freed, just
                   * overwritten.
                   */
                  start_new_game(game);
                  generate_moves(&movelist, game->board, game->player);
                  redraw = true;
                  break;
                  
               case ALLEGRO_KEY_FULLSTOP:       /* Switch figurine set */
                  if (figurine_set < max_figurine_set) {
                     figurine_set++;
                     load_figurine_set(figurine_set);
                     redraw = true;
                  }
                  break;
               case ALLEGRO_KEY_COMMA:
                  if (figurine_set > 0) {
                     figurine_set--;
                     load_figurine_set(figurine_set);
                     redraw = true;
                  }
                  break;
               case ALLEGRO_KEY_LEFT:
                  if ((player_cursor&7) != 0) player_cursor--;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_RIGHT:
                  if ((player_cursor&7) != 7) player_cursor++;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_UP:
                  if (player_cursor < 56) player_cursor += 8;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_DOWN:
                  if (player_cursor>7) player_cursor -= 8;
                  redraw = true;
                  break;
               case ALLEGRO_KEY_ENTER:
                  if (player_select == -1) {
                     player_select = player_cursor;
                     display_timer_counter = 0;
                  } else {
                     /* Confirm that the player has entered a valid move */
                     int n = validate_move(&movelist, player_select, player_cursor);

                     if (n == -1)
                        n = validate_move(&movelist, player_cursor, player_select);
                     if (n > -1) {
                        playmove_was_ok(game, movelist.move[n]);
                        generate_moves(&movelist, game->board, game->player);
                     }
                     player_select = -1;
                  }
                  redraw = true;
                  break;
               case ALLEGRO_KEY_TAB:
                  redo_last_move(game);
                  generate_moves(&movelist, game->board, game->player);
                  redraw = true;
                  break;
               case ALLEGRO_KEY_BACKSPACE:
                  takeback(game);
                  generate_moves(&movelist, game->board, game->player);
                  redraw = true;
                  break;
               case ALLEGRO_KEY_SPACE:
                  set_time_per_move(game, 5000);
                  computer_play(game, 40);
                  generate_moves(&movelist, game->board, game->player);
                  player_select = -1;
                  redraw = true;
                  break;
            }
            break;
         case ALLEGRO_EVENT_TIMER:
            display_timer_counter++;
            if (player_select > -1) redraw = true;
            break;
         case EVENT_FILE_DIALOG_CLOSE:
            al_unregister_event_source(queue, &file_dialog->event_source);

            /* If files were selected, we replace the old files list.
             * Otherwise the dialog was cancelled, and we keep the old results.
             */
            if (al_get_native_file_dialog_count(file_dialog->file_dialog) > 0) {
               ALLEGRO_NATIVE_DIALOG *dlg;
               dlg = file_dialog->file_dialog;
               last_path = al_get_native_file_dialog_path(dlg, 0);
               if (al_get_native_file_dialog_count(dlg)) {
                  const ALLEGRO_PATH *path;
                  const char *name;
                  gamestate_t *new_game = NULL;
                  path = al_get_native_file_dialog_path(dlg, 0);
                  name = al_path_cstr(path, '/');
                  if (file_dialog->flags & DIALOG_FLAGS_LOAD) {
                     /* Load file */
                     if (file_dialog->flags & DIALOG_FLAGS_EPD)
                        new_game = load_epd_file(name);
                     if (new_game) {
                        end_game(game);
                        game = new_game;
                        generate_moves(&movelist, game->board, game->player);
                     } else {
                        al_show_native_message_box( "Error", "",
                              "Could not import the selected file... sorry :(",
                              NULL, ALLEGRO_MESSAGEBOX_ERROR);
                     }
                     redraw = true;
                  } else {
                     /* Save file */
                     if (file_dialog->flags & DIALOG_FLAGS_EPD)
                        write_epd_file(game, name);
                  }
               }
            } else {
               stop_async_dialog(file_dialog);
            }
            file_dialog = NULL;
            break;
      }

   }

   end_game(game);

   return 0;
}
END_OF_MAIN()
