#include <string.h>
#include "hashtable.h"
#include "config.h"

#undef HASH_STRATEGY_ALWAYS_REPLACE

static inline size_t map_key_to_index(uint64_t key, size_t nelem)
{
   return key & ( nelem - 1);
}

hash_table_t *create_hash_table(size_t nelem)
{
   hash_table_t *table = calloc(1, sizeof *table);
   table->data = calloc(nelem, sizeof *table->data);
   table->depth_data = calloc(nelem, sizeof *table->data);
   table->number_of_elements = nelem;
   return table;
}

void destroy_hash_table(hash_table_t *table)
{
   if (table) {
      free(table->data);
      free(table->depth_data);
      free(table);
   }
}

void store_table_entry(hash_table_t *table, uint64_t key, int depth,  int score, unsigned int flags, move_t best_move)
{
   hash_table_entry_t *data;
   size_t index;

   if (!table)
      return;

   /* Map the key onto the array index, check if entry is there */
   index = map_key_to_index(key, table->number_of_elements);
   data = table->data + index;

   /* Check where to store this data: the depth priority table or the
    * always replace table.
    */
#ifdef HASH_STRATEGY_ALWAYS_REPLACE
   if ( (flags & HASH_TYPE_EXACT) || (flags & HASH_TYPE_BETA) ||
      !table->data[index].flags) {
      table->data[index].key = key;
      table->data[index].depth = depth;
      table->data[index].score = score;
      table->data[index].flags = flags;
      table->data[index].best_move = best_move;
   }
#else

   /* The data in the table was obtained at a smaller horizon distance.
    * Push the data from the depth priority table into the normal table,
    * then replace the data in the depth priority table.
    */
   if (table->data[index].depth <= depth || table->data[index].age == 0) {
      memcpy(data, table->depth_data+index, sizeof(hash_table_entry_t));
      data = table->depth_data + index;
   }

   data->key = key;
   data->depth = depth;
   data->score = score;
   data->flags = flags;
   data->best_move = best_move;
   data->age = 10;
#endif
}

hash_table_entry_t *query_table_entry(hash_table_t *table, uint64_t key)
{
   size_t index;

   if (!table)
      return NULL;

   /* Map the key onto the array index, check if entry is there */
   index = map_key_to_index(key, table->number_of_elements);
   
//#ifndef HASH_STRATEGY_ALWAYS_REPLACE
   /* Check the "depth priority" table */
   if (table->depth_data[index].key == key){
      table->depth_data[index].age = 10;
      return &table->depth_data[index];
   }
//#endif

   /* Check the "always replace" table */
   if (table->data[index].key == key) {
      table->data[index].age = 10;
      return &table->data[index];
   }

   return NULL;
}
