#include <string.h>
#include "pieces.h"
#include "movegen.h"
#include "board.h"
#include "see.h"
#include "sbr.h"

#undef PRINT_SEE_DEBUG_MESSAGES

static inline int max(int x, int y)
{
   return (x>y)?x:y;
}

static inline int find_least_valued_piece(const board_t *board, bitboard_t own)
{
   int n;

   for (n=0; n<5; n++)
      if (board->bbp[n] & own)
         return sbr64(board->bbp[n] & own);

   /* Final option: king */
   return sbr64(board->bbp[5] & own);
}

/* Perform static exchange evaluation for a given capture move.
 * Uses the "swap" algorithm
 * (http://chessprogramming.wikispaces.com/SEE+-+The+Swap+Algorithm).
 * returns the evaluation in centipawns.
 */
int static_exchange_evaluation(const board_t *const_board, const move_t move)
{
   int score[32];
   int depth;
   int side = get_move_player(move);
   int piece = get_move_piece(move);
   int square = get_destination(move);
   int origin = get_origin(move);
   bitboard_t attackers;
   bitboard_t xray_update;
   bitboard_t own;
   board_t board;

   assert(is_capture(move));

   /* Make a copy of the board */
   memcpy(&board, const_board, sizeof *const_board);

   /* We need to update the list of attackers if the moving piece is a
    * slider or a pawn.
    * FIXME: this is only necessary if there is a slider attacking one of
    * these squares.
    */
   xray_update = board.bbp[0]|board.bbp[2]|board.bbp[3]|board.bbp[4];

   /* Initial gain: the piece on the target square */
   depth = 0;
   score[depth] = piece_value(get_piece(&board, square));

   /* Perform the capture on the board */
   remove_piece(&board, square);
   move_piece(&board, piece, origin, square);

#ifdef PRINT_SEE_DEBUG_MESSAGES
   printf("SEE: %d %s %02x\n", depth, side?"BLACK":"White", piece);
#endif

   /* Get a new list of all attackers/defenders */
   attackers = get_all_attackers(&board, square);

   do {
      /* Consider next move - one ply deeper, flip side */
      depth++;
      side ^= BLACK;
      /* Update score (speculative) */
      score[depth] = piece_value(get_piece(&board, square)) - score[depth-1];
      /* Perform LVA capture */
      own = attackers&board.bbc[side>>7];
      if (own) {
         /* Find the least valued attacker */
         origin = find_least_valued_piece(&board, own);
         piece = get_piece(&board, origin);

#ifdef PRINT_SEE_DEBUG_MESSAGES
         printf("SEE: %d %s %02x %d\n", depth, side?"Black":"White", piece&PIECE, piece_value(piece));
#endif

         /* Perform the capture on the board */
         remove_piece(&board, square);
         move_piece(&board, piece, origin, square);

         /* Update the list of attackers/defenders */
         if (get_bitboard(&xray_update, origin))
            attackers = get_all_attackers(&board, square);
         else
            unset_bitboard(&attackers, origin);

         /* If the piece is a king, the list of attackers should be empty
          * now, otherwise we just performed an illegal capture (putting
          * the king in check), meaning we should terminate the evaluation
          * at this point. Since the king is always the last piece we check
          * for each side, any remaining attackers belong to the
          * opposing side.
          */
         if ( (piece&KING) && attackers ) {
            break;
         }
      }
   } while(own);

   /* Compute final score */
   while (--depth) {
      score[depth-1] = -max(-score[depth-1], score[depth]);
   }
   return score[depth];
}

