#include <stdint.h>
#include "bitboard.h"
#include "board.h"
#include "game.h"
#include "pawns.h"
#include "config.h"

#undef PRINT_PAWN_EVALUATION_BITBOARDS

static void print_uint8_t(uint8_t b)
{
   int c;
   
   for (c=0; c<8; c++) {
      printf ("%d", (b>>c)&1);
   }
}

static inline size_t map_key_to_index(uint64_t key, size_t nelem)
{
   return key & (nelem - 1);
}

pawn_hash_table_t *create_pawn_hash_table(size_t nelem)
{
   pawn_hash_table_t *table = calloc(1, sizeof *table);

   table->number_of_elements = PAWN_TABLE_SIZE;
   table->data = calloc(PAWN_TABLE_SIZE, sizeof *table->data);
   return table; 
}

void destroy_pawn_hash_table(pawn_hash_table_t *table)
{
   if (table) {
      free(table->data);
      free(table);
   }
}

pawn_structure_t *query_pawn_table_entry(pawn_hash_table_t *table, uint64_t key)
{
   size_t index;

   if (!table)
      return NULL;

   /* Map the key onto the array index, check if entry is there */
   index = map_key_to_index(key, table->number_of_elements);

   if (table->data[index].key == key)
      return table->data + index;
   else
      return NULL;
}

#define store_pawn_structure()\
{                              \
   int n;   \
   int index = map_key_to_index(board->pawn_hash,\
                        game->pawn_structure->number_of_elements);\
   pawn_structure = game->pawn_structure->data+index;\
   \
   pawn_structure->key = board->pawn_hash;\
   pawn_structure->score = ev;\
   pawn_structure->open_files = open_files;\
   pawn_structure->strong_squares[0] = \
       (isolated_pawns[1] | backward_pawns[1]) >> 8;\
   pawn_structure->strong_squares[1] = \
       (isolated_pawns[0] | backward_pawns[0]) << 8;\
   for (n=0; n<2; n++) {\
      pawn_structure->half_open_files[n] = half_open_files[n];\
      pawn_structure->connected_free[n] = free_pawn_span[n]&free_pawns[n];\
      pawn_structure->free[n] = free_pawns[n];\
      pawn_structure->outposts[n] =\
               attack_span[n] & pawn_structure->strong_squares[n];\
   }\
}

/* Evaluate pawn structure, from the point-of-view of *white* */
pawn_structure_t *evaluate_pawn_structure(gamestate_t *game)
{
   pawn_structure_t *pawn_structure;
   bitboard_t pawns[2];
   bitboard_t duo[2];
   bitboard_t backward_pawns[2];
   bitboard_t isolated_pawns[2];
   bitboard_t connected_pawns[2];
   bitboard_t doubled_pawns[2];
   bitboard_t free_pawns[2];
   bitboard_t free_pawn_span[2];
   bitboard_t span[2];
   bitboard_t attack_span[2];
   bitboard_t back_attack_span[2];
   bitboard_t bb;
   board_t *board;
   uint8_t pawn_files[2];
   uint8_t rook_files[2];
   uint8_t open_files;
   uint8_t half_open_files[2];
   int n;
   int ev = 0;

   assert(game);
   board = game->board;

   pawn_structure =
      query_pawn_table_entry(game->pawn_structure, board->pawn_hash);

   if (!pawn_structure) {
      /* Construct white and black pawn boards */
      pawns[0] = board->bbp[0] & board->bbc[0];
      pawns[1] = board->bbp[0] & board->bbc[1];

      /* Collapse all rows into one row to get file masks */
      pawn_files[0] = pawn_files[1] = 0;
      for (n=0; n<8; n++) {
         pawn_files[0] |= pawns[0]>>(n*8);
         pawn_files[1] |= pawns[1]>>(n*8);
      }
      open_files = ~(pawn_files[0]|pawn_files[1]);
      half_open_files[0] = (~pawn_files[0]) & ~open_files;
      half_open_files[1] = (~pawn_files[1]) & ~open_files;

      /* Construct forward black/white spans and attack spans: these are
       * squares on a file containing a pawn, or that are adjacent to a file
       * containing a pawn (forward attack files).
       */
      span[0] = pawns[0];
      span[0] |= span[0]<<8;
      span[0] |= span[0]<<16;
      span[0] |= span[0]<<32;

      span[1] = pawns[1];
      span[1] |= span[1]>>8;
      span[1] |= span[1]>>16;
      span[1] |= span[1]>>32;

      attack_span[0] =
         ((span[0]&board_minus_h)<<9)|((span[0]&board_minus_a)<<7);
      attack_span[1] =
         ((span[1]&board_minus_h)>>7)|((span[1]&board_minus_a)>>9);

      /* Calculate backward extension of attack spans (needed to distinguish
       * between isolated and backward pawns.
       */
      back_attack_span[0] = attack_span[0]>>8;
      back_attack_span[0] |= back_attack_span[0]>>16;
      back_attack_span[0] |= back_attack_span[0]>>32;

      back_attack_span[1] = attack_span[1]<<8;
      back_attack_span[1] |= back_attack_span[1]<<16;
      back_attack_span[1] |= back_attack_span[1]<<32;

      /* Find doubled (or tripled) pawns */
      doubled_pawns[0] = pawns[0] & (span[0]<<8);
      doubled_pawns[1] = pawns[1] & (span[1]>>8);

      /* Find free pawns: any pawns that can never be captured or
       * stopped by an enemy pawn
       */
      free_pawns[0] = pawns[0] & ~(span[1] | attack_span[1]);
      free_pawns[1] = pawns[1] & ~(span[0] | attack_span[0]);

      /* Span of free pawns, useful to find connected free pawns */
      free_pawn_span[0] = free_pawns[0];
      free_pawn_span[0] |= free_pawn_span[0]>>8;
      free_pawn_span[0] |= free_pawn_span[0]>>16;
      free_pawn_span[0] |= free_pawn_span[0]>>32;
      free_pawn_span[0] |= free_pawn_span[0]<<8;
      free_pawn_span[0] |= free_pawn_span[0]<<16;
      free_pawn_span[0] |= free_pawn_span[0]<<32;
      free_pawn_span[0] = ((free_pawn_span[0]&board_minus_h)>>7) |
         ((free_pawn_span[0]&board_minus_a)>>9);

      free_pawn_span[1] = free_pawns[1];
      free_pawn_span[1] |= free_pawn_span[1]>>8;
      free_pawn_span[1] |= free_pawn_span[1]>>16;
      free_pawn_span[1] |= free_pawn_span[1]>>32;
      free_pawn_span[1] |= free_pawn_span[1]<<8;
      free_pawn_span[1] |= free_pawn_span[1]<<16;
      free_pawn_span[1] |= free_pawn_span[1]<<32;
      free_pawn_span[1] = ((free_pawn_span[1]&board_minus_h)>>7) |
         ((free_pawn_span[1]&board_minus_a)>>9);

      /* duos */
      duo[0] = (pawns[0]&((pawns[0]&board_minus_a)<<1)) |
         (pawns[0]&((pawns[0]&board_minus_h)>>1));
      duo[1] = (pawns[1]&((pawns[1]&board_minus_a)<<1)) |
         (pawns[1]&((pawns[1]&board_minus_h)>>1));

      /* Find isolated pawns: pawns that have no friendly pawns on
       * neighbouring files.
       * As calculated here, this includes backward pawns: pawns that have no
       * friendly pawns behind them, but have friendly pawns ahead.
       * We need to consider pawns next to eachother explicitly, otherwise a
       * duo (say) will be classified as a connected pair of isolated pawns
       * (!)
       */
      isolated_pawns[0] = pawns[0] & ~(attack_span[0]|attack_span[0]>>8);
      isolated_pawns[1] = pawns[1] & ~(attack_span[1]|attack_span[1]<<8);

      /* Backward pawns */
      backward_pawns[0] = isolated_pawns[0] & back_attack_span[0];
      backward_pawns[1] = isolated_pawns[1] & back_attack_span[1];

      /* Backward pawns are not isolated */
      isolated_pawns[0] ^= backward_pawns[0];
      isolated_pawns[1] ^= backward_pawns[1];

      /* Connected pawns */
      connected_pawns[0] = pawns[0] ^ isolated_pawns[0];
      connected_pawns[1] = pawns[1] ^ isolated_pawns[1];

      /* A "duo" is a special case of two connected pawns on the same row.
       * "Hanging pawns" is an isolated duo.
       */

#ifdef PRINT_PAWN_EVALUATION_BITBOARDS
      printf("White: "); print_uint8_t(pawn_files[0]); printf("\n");
      printf("Black: "); print_uint8_t(pawn_files[1]); printf("\n");
      printf("Open:  "); print_uint8_t(open_files); printf("\n");
      printf("Half/W:"); print_uint8_t(half_open_files[0]); printf("\n");
      printf("Half/B:"); print_uint8_t(half_open_files[1]); printf("\n");

      printf("Doubled pawns:\n");
      printf_bitboard(doubled_pawns[0]|doubled_pawns[1]);

      printf("Free pawns:\n");
      printf_bitboard(free_pawns[0]|free_pawns[1]);

      printf("Isolated pawns:\n");
      printf_bitboard(isolated_pawns[0]|isolated_pawns[1]);

      printf("Connected pawns:\n");
      printf_bitboard(connected_pawns[0]|connected_pawns[1]);

      printf("Backward pawns:\n");
      printf_bitboard(backward_pawns[0]|backward_pawns[1]);

      printf("Duos\n");
      printf_bitboard(duo[0]|duo[1]);

      printf("Connected free pawns:\n");
      printf_bitboard((free_pawn_span[1])&free_pawns[1]);
#endif

      /* Free pawns */
      /* FIXME: should penalise free pawns that are blocked; free pawns that
       * are close to promotion are more valuable!
       */
      ev += sparse_bits_set(free_pawns[0])*20;
      ev -= sparse_bits_set(free_pawns[1])*20;

      /* Free pawns that are close to promotion are worth more */
      bb = free_pawns[0];
      while (bb) {
         int p = sbr64(bb);
         int dist;

         unset_bitboard(&bb, p);
         dist = unpack_row(p);

         /* FIXME: should correct for the other king being in the "square"
          * (ie, having a shorter Manhatten-distance to the promotion square.
          */
         if (dist > 4)
            ev += (dist-4)*30;
      }

      /* Free pawns that are close to promotion are worth more */
      bb = free_pawns[1];
      while (bb) {
         int p = sbr64(bb);
         int dist;

         unset_bitboard(&bb, p);
         dist = 7 - unpack_row(p);

         /* FIXME: should correct for the other king being in the "square"
          * (ie, having a shorter Manhatten-distance to the promotion square.
          */
         if (dist > 4)
            ev -= (dist-4)*30;
      }

      if (pawns[1]&board_rank2) {
         //print_bitboards(board);
         //printf("Black pawns on the second rank!\n");
      }

      /* Extra bonus: connected free pawns */
      ev += sparse_bits_set(connected_pawns[0]&free_pawns[0])*12;
      ev -= sparse_bits_set(connected_pawns[1]&free_pawns[1])*12;

#if 0
      /* Connected free pawns on the 6th row beat a rook */
      /* FIXME: that doesn't mean that they're *worth* a rook, especially
       * if the opponent still has other pieces as well!
       * In that situation, the bonus should probably be reduced...
       */
      if ( !is_power_of_two(( (free_pawn_span[0])&free_pawns[0] ) &
               (board_rank6|board_rank7|board_rank8) ))
         ev += 500;

      /* Connected free pawns on the 6th row beat a rook */
      if ( !is_power_of_two(( (free_pawn_span[1])&free_pawns[1] ) &
               (board_rank1|board_rank2|board_rank3) ))
         ev -= 500;
#endif

      /* Weak pawns: doubled pawns */
      ev -= sparse_bits_set(doubled_pawns[0])*30;
      ev += sparse_bits_set(doubled_pawns[1])*30;

      /* Weak pawns: isolated pawns */
      ev -= sparse_bits_set(isolated_pawns[0])*12;
      ev += sparse_bits_set(isolated_pawns[1])*12;

      /* Weak pawns: backward pawns */
      ev -= sparse_bits_set(backward_pawns[0])*5;
      ev += sparse_bits_set(backward_pawns[1])*5;

      store_pawn_structure();
   }
   
   return pawn_structure;
}


