#include "bitboard.h"
#include "board.h"
#include "game.h"
#include "evaluate.h"
#include "pawns.h"
#include "sbr.h"

/* The evaluation function should consider the following:
 *
 * TODO: it is probably slightly better to give penalties for not being in
 * a favourable position than it is for giving bonuses for being there (eg,
 * penalise a rook on a closed file rather than award a rook on an open
 * file).
 *
 * Idea: Generate piece square tables based on pawn structure and the
 * location of the two kings. Before the search, initialise these tables
 * with values based on the current position. When doing move ordering,
 * use the difference between square values to sort regular moves
 * (reduction candidates). When it becomes time to do the evaluation,
 * we can recompute the tables (if needed, or look them up in a hash table)
 * and compute accurate scores for the pieces. For lazy evaluation, we
 * could use the score based on the old tables.
 *
 * Always:
 *  - material balance, this is a simple sum of piece values
 *  - slight (+5) bonus for side-to-move
 *  - piece-square tables
 *
 * Opening:
 *  - advance pawns
 *  - develop light pieces towards the centre
 *  - penalise moving a rook if not yet castled
 *  - penalise moving the king except when castling
 *  - penalise undeveloped light pieces
 *
 * Middle game:
 *  - penalise king-in-centre
 *  - penalise trapped pieces:
 *     * rook in the corner trapped by the king (encourage castling)
 *     * bishop trapped behind pawns of the same colour
 *     * bishop trapped in the corner behind enemy pawns
 *  - weakness of squares of a particular colour: majority of pawns on one
 *    colour and no bishop of that colour
 *  - award bonus for the bishop pair if neither bishop is bad (trapped
 *    behind own pawns)
 *  - give slight bonus for knight vs bishop if no bishop pair
 *  - evaluate king safety:
 *     * pawn structure
 *     * enemy pieces
 *  - pawn structure:
 *     * penalise doubled pawns
 *     * penalise backward pawns or isolated pawns
 *     * weak squares/bonus for knight/bishop on strong square
 *  - award rooks on open files/7th row
 *  - if material is equal, give a bonus to the side with the most pieces
 *    (ie, non-pawns). Rationale: two light pieces are better than
 *    rook+pawn, etc.
 *
 * End game:
 *  - test for simple endings/exact positions
 *  - push king to centre
 *  - give bonus to side with more pawns if there are fewer pieces
 *    (encourage exchanging pieces if winning)
 *  - conversely, give bonus to fewer pawns if a pawn down
 *  - slightly decrease effective value of knight, which is weaker in the
 *    end game than sliding pieces.
 *  - bonus for pushing a candidate
 *  - bonus for free pawns/advanced pawns
 *     * defended free pawns
 *     * connected free pawns
 *     * defending king outside square
 *     * "2 pawns on 6th row beat a rook"
 *     * key squares in front of free pawn should be occupied by king
 *  - rook should attack base of enemy pawn structure (generalise 7th row rule)
 *  - recognise draw-ish end games: rook vs rook+minor, opposing bishops,
 *    bishop vs knight with pawns on one wing, wrong pawn.
 *
 * Board control:
 *  award a bonus for the player who controls most squares in the extended
 *  centre and/or near the enemy king (although the latter should probably
 *  be done by king safety code).
 */

/* Kauffman says rooks should be more valuable in open positions - but this
 * whould only hold for rooks that actually take advantage of the openness
 * of the position - ie, for rooks on open files.
 */
#undef ROOK_BONUS_ONLY_ON_OPEN_FILE

/* Crude rule-of-thumb piece position tables, to encourage the program to
 * move pieces towards the centre of the board.
 */
static int knight_table[64] = {
   0, 2,  4,  6,  6,  4, 2, 0,
   2, 4,  6,  8,  8,  6, 4, 2,
   4, 6,  8, 10, 10,  8, 6, 4,
   6, 8, 10, 12, 12, 10, 8, 6,
   6, 8, 10, 12, 12, 10, 8, 6,
   4, 6,  8, 10, 10,  8, 6, 4,
   2, 4,  6,  8,  8,  6, 4, 2,
   0, 2,  4,  6,  6,  4, 2, 0
};

static int bishop_table[64] = {
   0, 2,  4,  6,  6,  4, 2, 0,
   2, 6,  6,  8,  8,  6, 6, 2,
   4, 6,  8,  8,  8,  8, 6, 4,
   6, 8, 10, 12, 12, 10, 8, 6,
   6, 8, 10, 12, 12, 10, 8, 6,
   4, 6,  8,  8,  8,  8, 6, 4,
   2, 6,  6,  8,  8,  6, 6, 2,
   0, 2,  4,  6,  6,  4, 2, 0
};

/* This piece of code and the comment below are copied from Crafty.
 *
 * This array is indexed by rook advantage and minor piece advantage.
 * rook advantage is 4 + white rook equivalents - black rook equivalents 
 * where a rook equivalent is number of rooks + 2 * number of queens.
 * minor piece advantage is 4 + white minors - black minors.  
 *
 * The classic bad trade case is two minors for a rook.  If white trades
 * two minors for a rook, rook advantage is +5 and minor advantage is +2.
 * imbalance[5][2] gives a penalty of -42 for this trade.
 */
static const int imbalance[9][9] = {
/* n=-4  n=-3  n=-2  n=-1   n=0  n=+1  n=+2  n=+3 +n=+4 */
  {-126, -126, -126, -126, -126, -126, -126, -126,  -42 }, /* R=-4 */
  {-126, -126, -126, -126, -126, -126, -126,  -42,   42 }, /* R=-3 */
  {-126, -126, -126, -126, -126, -126,  -42,   42,   84 }, /* R=-2 */
  {-126, -126, -126, -126, -104,  -42,   42,   84,  126 }, /* R=-1 */
  {-126, -126, -126,  -88,    0,   88,  126,  126,  126 }, /*  R=0 */
  {-126,  -84,  -42,   42,  104,  126,  126,  126,  126 }, /* R=+1 */
  { -84,  -42,   42,  126,  126,  126,  126,  126,  126 }, /* R=+2 */
  { -42,   42,  126,  126,  126,  126,  126,  126,  126 }, /* R=+3 */
  {  42,  126,  126,  126,  126,  126,  126,  126,  126 }  /* R=+4 */
};

/* All of these things could (more cheaply) be updated incrementally, with
 * the possible exception of pawn structure, which may need some pattern
 * matching/hash tables...
 * In other words: FIXME!
 */

static int max(int x, int y)
{
   return (x>y)?x:y;
}

static int min(int x, int y)
{
   return (x<y)?x:y;
}

static int king_distance(int square1, int square2)
{
   int row1, file1;
   int row2, file2;

   row1 = unpack_row(square1);
   file1 = unpack_column(square1);
   row2 = unpack_row(square2);
   file2 = unpack_column(square2);

   return max(abs(row1 - row2), abs(file1-file2));
}

/* MiniMax evaluation function during the opening
 * Adapted from 2003 chess program, which had it adapted from another
 * program that I found online, but have since lost (and I can't remember
 * where it came from).
 * In other words, the ideas here are not mine, but they'll be
 * updated/replaced eventually.
 */
static inline int opening_evaluation(board_t *board, int player, int opponent)
{
   bitboard_t bb;
   bitboard_t bk;
   int square;
   int ev = 0;

   /* Piece positions */

   /* Centralised pawns */
   bb = and_bitboards(and_bitboards(board->bbc[player], board->bbp[0]), board_centre);
   if (bb)
      ev+=15;
   bb = and_bitboards(and_bitboards(board->bbc[player], board->bbp[0]), board_xcentre);
   if (bb)
      ev+=10;

   bb = and_bitboards(and_bitboards(board->bbc[opponent], board->bbp[0]), board_centre);
   if (bb)
      ev-=15;
   bb = and_bitboards(and_bitboards(board->bbc[opponent], board->bbp[0]), board_xcentre);
   if (bb)
      ev-=10;

   /* Knights */
   bb = and_bitboards(board->bbc[player], board->bbp[1]);
   while(bb) {
      square = sbr64(bb);
      ev += knight_table[square];
      unset_bitboard(&bb, square);
   }   
   bb = and_bitboards(board->bbc[opponent], board->bbp[1]);
   while(bb) {
      square = sbr64(bb);
      ev -= knight_table[square];
      unset_bitboard(&bb, square);
   }

   /* Bishops */
   bb = and_bitboards(board->bbc[player], board->bbp[2]);
   while(bb) {
      square = sbr64(bb);
      ev += bishop_table[square];
      unset_bitboard(&bb, square);
   }   
   bb = and_bitboards(board->bbc[opponent], board->bbp[2]);
   while(bb) {
      square = sbr64(bb);
      ev -= bishop_table[square];
      unset_bitboard(&bb, square);
   }
   
   /* Discourage queen sorties */
   bb = and_bitboards(board->bbp[4], or_bitboards(board_rank4, board_rank5));
   if (and_bitboards(board->bbc[player], bb))
      ev-=12;
   if (and_bitboards(board->bbc[opponent], bb))
      ev+=12;

   bk = and_bitboards(board->bbp[5], board->init);
   /* Penalize rooks that have moved while the King has not moved */
   bb = invert_bitboard(and_bitboards(board->bbp[3], board->init));
   if (and_bitboards(bb, board->bbp[player]) && 
       and_bitboards(board->bbp[player], bk)) {
      ev-=15;
   }

   bb = invert_bitboard(and_bitboards(board->bbp[3], board->init));
   if (and_bitboards(bb, board->bbp[opponent]) && 
       and_bitboards(board->bbp[opponent], bk)) {
      ev+=15;
   }
   
   /* Penalize if opponent king has castled and own king has not */
   if (and_bitboards(board->bbp[player], bk)) {
      ev-=10;
   }
   if (and_bitboards(board->bbp[opponent], bk)) {
      ev+=10;
   }
   
   return ev;
}

static int evaluate_lone_king(board_t *board, int player)
{
   int attacker, defender;
   int attacker_king_square, defender_king_square, row, column, ev, opp;

   attacker = 0;
   if (is_power_of_two(board->bbc[0])) {
      attacker = 1;
   }
   defender = 1 - attacker;

   /* Single piece draw */
   if (board->material[attacker] == 300) {
      return 0;
   }

   /* Lone kings */
   if (is_power_of_two(board->bbc[attacker]))
      return 0;

   /* The attacker should try to force the defending king to the edge of
    * the board and bring his own king close.
    */
   
   /* How close is the defender to the edge? */
   defender_king_square = sbr64(board->bbc[defender]);
   row = unpack_row(defender_king_square);
   column = unpack_row(defender_king_square);
   ev = 6*(abs(2*row - 7) + abs(2*column - 7));

   /* KNBK: the lone king can only be mated in the corner of the bishop
    * colour
    */
   if (board->bbp[1] && board->bbp[2]) {
      int bishop_colour = sbr64(board->bbp[2]);
      if (bishop_colour & 1) {   /* White square */
         /* The defending king is safest along the A8-H1 diagonal */
         ev += 3*abs(7-calculate_a8h1_diagonal(defender_king_square));
      } else {
         /* The defending king is safest along the A1-H8 diagonal */
         ev += 3*abs(7-calculate_a1h8_diagonal(defender_king_square));
      }
   }

   /* Manhatten distance between the two kings */
   attacker_king_square = sbr64(board->bbc[attacker]);
   row -= unpack_row(attacker_king_square);
   column -= unpack_row(attacker_king_square);
   ev += 3*(7-abs(row) + 7-abs(column));

   if (player == defender)
      ev = -ev;

   /* Give bonus points for opposition */
   opp = abs(row) + abs(column);
   if ( (opp == 2) || (opp == 4) )
      ev+=4;

   return ev;
}

/* Evaluate the king safety, from white's point of view. */
int evaluate_king_safety(gamestate_t *game)
{
   board_t *board = game->board;
   int square;
   int ev = 0;
   /* Really simple: check whether the king has moved away from the centre,
    * but penalise trapped rooks. Deduct points for a weakened pawn shield.
    */

   if (1||board->bbp[4] & board->bbc[1]) {
      int row, file;
      bitboard_t pawn_shield;
      square = sbr64(board->bbp[5] & board->bbc[0]);
      row = unpack_row(square);
      file = unpack_column(square);
      ev -= 80*row;

      if (file==3 || file==4)
         ev -= 20;

      /* Check for trapped rooks */
      if (file>4 && (board->bbp[3]&board->bbc[0]&(F1_MASK|G1_MASK)))
         ev -= 50;

      /* Check for trapped rooks */
      if (file<3 && (board->bbp[3]&board->bbc[0]&(A1_MASK|B1_MASK)))
         ev -= 50;

      /* Damaged pawn shield */
      pawn_shield = make_bitboard_square(square)>>8;
      pawn_shield |= shift_bitboard_file_left(pawn_shield);
      pawn_shield |= shift_bitboard_file_right(pawn_shield);
      ev -= 8*(3-sparse_bits_set(pawn_shield & board->bbp[0] & board->bbc[0]));
   }


   if (1||board->bbp[4] & board->bbc[0]) {
      int row, file;
      bitboard_t pawn_shield;
      square = sbr64(board->bbp[5] & board->bbc[1]);
      row = 7-unpack_row(square);
      file = unpack_column(square);
      ev += 80*row;

      if (file==3 || file==4)
         ev += 20;

      /* Check for trapped rooks */
      if (file>4 && (board->bbp[3]&board->bbc[1]&(F8_MASK|G8_MASK)))
         ev += 50;

      /* Check for trapped rooks */
      if (file<3 && (board->bbp[3]&board->bbc[1]&(A8_MASK|B8_MASK)))
         ev += 50;

      /* Damaged pawn shield */
      pawn_shield = make_bitboard_square(square)>>8;
      pawn_shield |= shift_bitboard_file_left(pawn_shield);
      pawn_shield |= shift_bitboard_file_right(pawn_shield);
      ev += 8*(3-sparse_bits_set(pawn_shield & board->bbp[0] & board->bbc[1]));
   }

   return ev;
}

/* Lazy evaluation: only look at material */
int lazy_evaluation(gamestate_t *game, int player)
{
   player >>= 7;
   return game->board->material[player]-game->board->material[1-player];
}

/* Evaluate the current state of the game from the point of view of the
 * player to move.
 * Returns a positive score if the player to move is ahead or a negative
 * score if he is behind. Returns 0 iff the game is a draw.
 */
int static_evaluation(gamestate_t *game, int player)
{
   static const bitboard_t back_rank[2] = {
      board_rank6|board_rank7|board_rank8, 
      board_rank1|board_rank2|board_rank3
   };
   static const bitboard_t wrong_afile_bishop[2] = {
      0xAA55AA55AA55AA55ll, 0x55AA55AA55AA55AAll
   };
   static const bitboard_t wrong_hfile_bishop[2] = {
      0x55AA55AA55AA55AAll, 0xAA55AA55AA55AA55ll
   };
   static const int last_rank[2] = {7, 0};
   bitboard_t stop_squares[2];
   board_t *board;
   pawn_structure_t *pawn_structure;
   int majors[2], minors[2];
   int minor_idx, major_idx;
   int ev = 0;
   int opponent;
   int own_pawns, opponent_pawns;
   int own_pieces, opponent_pieces;
   uint8_t rook_files[2];
   uint8_t open_files;
   uint8_t half_open_files[2];
   int king_square[2];
   bitboard_t bb;
   int n;

   board = game->board;
   
   /* Convert to 1-based counting for black and white */
   player >>= 7;
   opponent = player^0x01;

   /* Assess material balance */
   ev = board->material[player]-board->material[opponent];

   /* Count material */
   own_pawns = sparse_bits_set(board->bbc[player]&board->bbp[0]);
   own_pieces = sparse_bits_set(board->bbc[player]) - own_pawns - 1;
   opponent_pawns = sparse_bits_set(board->bbc[opponent]&board->bbp[0]);
   opponent_pieces = sparse_bits_set(board->bbc[opponent]) - opponent_pawns - 1;

   bb = board->bbp[1] | board->bbp[2];
   minors[0] = sparse_bits_set(bb & board->bbc[0]);
   minors[1] = sparse_bits_set(bb & board->bbc[1]);
   majors[0] = (sparse_bits_set(board->bbp[3] & board->bbc[0]) + 
               2*sparse_bits_set(board->bbp[4] & board->bbc[0]));
   majors[1] = (sparse_bits_set(board->bbp[3] & board->bbc[1]) + 
               2*sparse_bits_set(board->bbp[4] & board->bbc[1]));

   king_square[0] = sbr64(board->bbp[5] & board->bbc[0]);
   king_square[1] = sbr64(board->bbp[5] & board->bbc[1]);

   /* Material imbalances: the number of majors and minors does not match
    * between sides. Pawns may make up the difference, but in general pawns
    * for pieces is not worth it. This code was adapted from Crafty
    */
   minor_idx = minors[player] - minors[opponent];
   major_idx = majors[player] - majors[opponent];

   ev += imbalance[max(min(4+major_idx, 8), 0)][max(min(4+minor_idx, 8), 0)];

   /* Give a minute bonus to the side to move. The idea behind this is to
    * break the degeneracy between positions that are exactly balanced, and
    * those that are legitimate draws.
    * This value should be small enough that it doesn't affect the
    * evaluation too much.
    */
   if (ev == 0) ev+=5;
   
   /* Balance from the piece-square tables */
   //printf ("%d %d %d\n", ev, board->psq[player], board->psq[opponent]);
   ev += (board->psq[player]-board->psq[opponent]);

   /* Award a bonus for having the bishop pair */
   if (!is_power_of_two(board->bbc[player] & board->bbp[2]))
      ev += EVAL_BISHOP_PAIR;
   if (!is_power_of_two(board->bbc[opponent] & board->bbp[2]))
      ev -= EVAL_BISHOP_PAIR;

   /* If ahead in material, award a penalty for having pieces on the
    * board, award a bonus for having pawns on the board.
    * This is to encourage piece exchanges when ahead and pawn exchanges
    * when behind.
    * Exception: if there are only rooks left, it's better to have two than
    * to have one.
    */
   if ( (board->bbp[1]|board->bbp[2]|board->bbp[4]) ) {
      if (board->material[player] > board->material[opponent]) {
         ev -= (own_pieces + opponent_pieces)*2;
         ev += (own_pawns + opponent_pawns)*4;
      } else if (board->material[player] < board->material[opponent]) {
         ev += (own_pieces + opponent_pieces)*2;
         ev -= (own_pawns + opponent_pawns)*4;
      }
   } else if ( board->bbp[3] ) {
      /* Draw score slightly closer to draw if there is only one rook on
       * each side.
       */
      if ( (majors[0]+majors[1]) == 2 ) {
         ev -= ev/8;
      }
   }

   /* Adjust the material score: rooks become stronger as pawns disappear,
    * knights become a bit weaker.
    * FIXME: rooks should probably only get the bonus if they are on open
    * files.
    */
#ifndef ROOK_BONUS_ONLY_ON_OPEN_FILE
   ev += bits_set(board->bbp[3]&board->bbc[player]) * (5-own_pawns)*3;
   ev -= bits_set(board->bbp[3]&board->bbc[opponent]) * (5-opponent_pawns)*3;
#endif
   //ev -= bits_set(board->bbp[1]&board->bbc[player]) * (5-own_pawns)*4;
   //ev += bits_set(board->bbp[1]&board->bbc[opponent]) * (5-opponent_pawns)*4;

   /* only apply the knight bonus for the first knight. Kauffman says
    * knight pairs should be penalised, which in itself seems odd to me. I
    * think it makes more sense to apply the bonus only to the first
    * knight.
    */
   ev -= ( (board->bbp[1]&board->bbc[player]) != 0) * (5-own_pawns)*4;
   ev += ( (board->bbp[1]&board->bbc[opponent]) != 0) * (5-opponent_pawns)*4;


   /* Check for lone king endings */
   if (is_power_of_two(board->bbc[0]) || is_power_of_two(board->bbc[1])) {
      return ev+evaluate_lone_king(board, player);
   }
   //return ev;

   /* If the material difference is large, there's no point in doing the
    * extra detailed work... it's never going to make up.
    */
   //if ( (ev > 300) || (ev < -300) )
   //   return ev;
   
   /* Fine tune */
   if (and_bitboards(board->bbp[5], board->init)) {
      /* The opening: this is before either king has castled */
      ev += opening_evaluation(board, player, opponent);
   } else if (board->material[player]+board->material[opponent] < ENDGAME_MATERIAL) {
      /* The end game, this is when the total material on the board drops
       * below some fixed number
       * *TODO*
       */
   } else {
      /* If it's neither opening nor end game, it's middle game */
      //ev += (1 - 2*player)*evaluate_king_safety(game);
   }

   /* Evaluate pawn structure and piece scores that depend on it */
   pawn_structure = evaluate_pawn_structure(game);
   assert(pawn_structure);

   ev += (1 - 2*player)*pawn_structure->score;
   open_files = pawn_structure->open_files;
   half_open_files[0] = pawn_structure->half_open_files[0];
   half_open_files[1] = pawn_structure->half_open_files[1];

   /* Evaluate some special pawn formations */

   /* Passed duo's
    * These will defeat a rook - but not necessarily if the rook aided by
    * other pieces!
    */
   if ( !is_power_of_two(
            pawn_structure->connected_free[player] & back_rank[player]) )
         ev += EVAL_ADVANCED_DUO;

   if ( !is_power_of_two(
            pawn_structure->connected_free[opponent] & back_rank[opponent]) )
         ev -= EVAL_ADVANCED_DUO;

   /* Free pawns: check whether the enemy king is outside "the square"
    * If it is, evauate this pawn as nearly a queen in pawn endings.
    * Crude and should be improved, but the search will make up for this,
    * probably...
    * This mainly misses cases where the defending king is inside the
    * square but blocked by other pawns or the attacking king. The search
    * should find these cases anyway.
    */
   if ( (board->bbp[0]|board->bbp[5]) == (board->bbc[0]|board->bbc[1]) ) {
      /* Only kings & pawns left */
      bb = pawn_structure->free[player];
      while (bb) {
         int pawn_square = sbr64(bb);
         int promotion_square;
         int pawn_distance;
         bb ^= make_bitboard_square(pawn_square);
         promotion_square = (pawn_square & 7) | (last_rank[player]<<3);
         pawn_distance = abs(unpack_row(pawn_square) - last_rank[player]);
         //pawn_distance = abs(pawn_square - promotion_square)>>3;

         /* Player has tempo */
         if (king_distance(king_square[opponent], promotion_square) > pawn_distance) {
            ev += 800;
         }
      }

      bb = pawn_structure->free[opponent];
      while (bb) {
         int pawn_square = sbr64(bb);
         int promotion_square;
         int pawn_distance;
         bb ^= make_bitboard_square(pawn_square);
         promotion_square = (pawn_square & 7) | (last_rank[opponent]<<3);
         pawn_distance = abs(unpack_row(pawn_square) - last_rank[opponent]);
         //pawn_distance = abs(pawn_square - promotion_square)>>3;

         /* Player has tempo */
         if ((king_distance(king_square[player], promotion_square)-1) > pawn_distance) {
            ev -= 800;
         }
      }
   }

   /* Evaluate stop squares */
   stop_squares[0] = pawn_structure->free[1] >> 8;
   stop_squares[1] = pawn_structure->free[0] << 8;

   /* Pieces on stop square: bonus for own piece on stop square, penalty
    * for own piece on enemy stop square (a piece blocking a free pawn of
    * the same colour)
    */
   ev += sparse_bits_set(stop_squares[player] & board->bbc[player]) * 50;
   ev -= sparse_bits_set(stop_squares[opponent] & board->bbc[player]) * 25;
   ev -= sparse_bits_set(stop_squares[opponent] & board->bbc[opponent]) * 50;
   ev += sparse_bits_set(stop_squares[player] & board->bbc[opponent]) * 25;

   /* Outposts and weak/strong squares.
    * For our purposes, we mean the following:
    * A weak square is a square in front of a backward pawn.
    * One player's weak square is the other player's strong square.
    * An outpost is a strong square defended by an own pawn
    * Note that an outpost is also a strong square.
    */

   ev += sparse_bits_set(
            pawn_structure->strong_squares[player]&board->bbc[player])*15;
   ev -= sparse_bits_set(
            pawn_structure->strong_squares[opponent]&board->bbc[opponent])*15;

   ev += sparse_bits_set(
            pawn_structure->outposts[player]&board->bbc[player]&
            (board->bbp[1]|board->bbp[2]))*25;
   ev -= sparse_bits_set(
            pawn_structure->outposts[opponent]&board->bbc[opponent]&
            (board->bbp[1]|board->bbp[2]))*25;

   /* TODO: award bonuses for major pieces on open files; this should be of
    * the order of 20-30 cp, and of-sets the penalty for a doubled pawn.
    */
#ifdef ROOK_BONUS_ONLY_ON_OPEN_FILE
   bb = board->bbp[3] & board->bbc[player];
   while (bb) {
      int square = sbr64(bb);
      int file = unpack_column(square);
      bb ^= make_bitboard_square(square);
      file = 1 << file;

      if (file & open_files) ev += 20;
      if (file & half_open_files[player]) ev += 10;

      /* Multiple (two) rooks on the same file get a bonus */
      if (bb & (board_afile*file)) {
         ev += 30;
         break;
      }
#if 0
      if (own_pawns > 5) {    /* Penalise rooks on closed files */
         if ( (1<<file) & ~(open_files|half_open_files[player]))
            ev += (5 - own_pawns)*3;
      } else {                /* Award bonus to rooks on open files */
         int m = 3;
         /* Reduce bonus for half open files */
         if ((1<<file) & half_open_files[player]) m = 2;
         if ( (1<<file) & (open_files|half_open_files[player])) {
            ev += (5 - own_pawns)*m;
         }
      }
#endif
   }

   bb = board->bbp[3] & board->bbc[opponent];
   while (bb) {
      int square = sbr64(bb);
      int file = unpack_column(square);
      bb ^= make_bitboard_square(square);

      if (file & open_files) ev -= 20;
      if (file & half_open_files[opponent]) ev -= 10;

      /* Multiple (two) rooks on the same file get a bonus */
      if (bb & (board_afile*file)) {
         ev -= 30;
         break;
      }
#if 0
      if (opponent_pawns > 5) {    /* Penalise rooks on closed files */
         if ( (1<<file) & ~(open_files|half_open_files[opponent]))
            ev -= (5 - opponent_pawns)*3;
      } else {                /* Award bonus to rooks on open files */
         int m = 3;
         /* Reduce bonus for half open files */
         if ((1<<file) & half_open_files[opponent]) m = 2;
         if ( (1<<file) & (open_files|half_open_files[opponent])) {
            ev -= (5 - opponent_pawns)*m;
         }
      }
#endif
   }
#else
   rook_files[0] = rook_files[1] = 0;
   for (n=0; n<8; n++) {
      rook_files[0] |= (board->bbp[3] & board->bbc[0])>>(n*8);
      rook_files[1] |= (board->bbp[3] & board->bbc[1])>>(n*8);
   }

   ev += sparse_bits_set(rook_files[player] & open_files)*25;
   ev -= sparse_bits_set(rook_files[opponent] & open_files)*25;
   ev += sparse_bits_set(rook_files[player] & half_open_files[player])*10;
   ev -= sparse_bits_set(rook_files[opponent] & half_open_files[opponent])*10;
#endif

   /* Simple end-game evaluation, using rules-of-thumb. These could
    * (should) be replaced by bitbases.
    */
   switch (own_pieces+opponent_pieces) {
      case 0:     /* Pawn ending */
         if (own_pawns && opponent_pawns) {  /* Both sides have pawns */
            /* We should be doing something a bit more sophisticated, but
             * for now we just bring the king out in the open and let the
             * main search handle the rest.
             */
            ev += piece_square_table[6][player][king_square[player]];
            ev -= piece_square_table[6][opponent][king_square[opponent]];
         } else { /* Only one side has pawns; let the search handle this */
         }
         break;

      case 1:     /* Lone piece or piece against or with pawn */
         /* Although not quite so simple, we let Q/R against pawn win, but
          * set minor against pawn or lone king to be at most a draw.
          * The latter is normally correct, the former may not be.
          */
         if (ev>0 && minors[player]) {
            ev = 0;
         }

         if (minors[player] && own_pawns) {
            /* Check for rook pawn + wrong bishop */
            bb = 0;
            if (board->bbc[player]&board->bbp[0]&board_afile)
               bb |= wrong_afile_bishop[player];
            if (board->bbc[player]&board->bbp[0]&board_hfile)
               bb |= wrong_hfile_bishop[player];
            if ((bb ^ board_all) && board->bbp[2] & bb && ev>0)
               ev /= 10;
         }
         break;

      case 2:     /* Two pieces and (possibly) pawns */
         /* Only handle the special case of pawns+minor vs (pawns)+minor */
         if (minors[0] && minors[1] && (own_pawns || opponent_pawns) ) {
            /* Check knight vs bishop and bishop vs bishop */
            if (board->bbp[1] && board->bbp[2]) {
               /* Knight vs. bishop: drawish if only pawns on one side, as
                * long as the knight is on the same side of the board to
                * stop the pawns.
                */
               if ( !(board->bbp[0] & board_qwing) ||
                    !(board->bbp[0] & board_kwing) ) {
                  if ( ((board->bbp[0] & board_qwing) &&
                        (board->bbp[1] & board_qwing)) ||
                       ((board->bbp[0] & board_qwing) &&
                        (board->bbp[1] & board_qwing)) )
                     ev /= 4;
                  
               }
            } else { /* Bishop vs. bishop, drawish if opposite coloured bishops */
               if ((board->bbp[2]&board_dark)&&(board->bbp[2]&board_light)) {
                  ev /= 4;
               }
            }
         } else if (majors[0] && majors[1] && (own_pawns||opponent_pawns) ) {
            /* Pawns+major vs (pawns)+major */
            if (board->bbp[3] == 0) {  /* Queen+pawn(s) vs queen+(pawns) */
               /* More complicated than this, but the position is
                * normally drawish. Leave it to the search to find the
                * exceptions.
                */
               ev /= 4;
            } else {
               /* Rook + pawn(s) vs. rook + (pawns) */
               if (!own_pawns || !opponent_pawns) {   /* Pawns on one side */
                  if (own_pawns == 1) {
                     /* Check for defending king in front of the pawn */
                     int pawn_square = sbr64(board->bbp[0]);
                     int promotion_square;
                     promotion_square = (pawn_square&7)|(last_rank[player]<<3);
                     if (king_distance(king_square[opponent],
                           promotion_square) < 2)  ev /= 2;
                  } else if (opponent_pawns == 1) {
                     /* Check for defending king in front of the pawn */
                     int pawn_square = sbr64(board->bbp[0]);
                     int promotion_square;
                     promotion_square=(pawn_square&7)|(last_rank[opponent]<<3);
                     if (king_distance(king_square[player],
                           promotion_square) < 2)  ev /= 2;
                  }
               }
            }
         }

      default:
         if (own_pawns || opponent_pawns) {
            /* Multiple pieces facing pawns. Pull the King to the centre */
            if ( (own_pieces+opponent_pieces)<4 || (majors[0]+majors[1])<3 ) {
               ev += piece_square_table[6][player][king_square[player]];
               ev -= piece_square_table[6][opponent][king_square[opponent]];
#if 0
            } else {
               /* Don't pull kings to the centre if there are still queens
                * on the board: penalise central positions for the king as
                * long as the other side still has a queen.
                */
               if ( (board->bbp[4] & board->bbc[opponent]) )
                  ev -= piece_square_table[6][player][king_square[player]];
               if ( (board->bbp[4] & board->bbc[player]) )
                  ev += piece_square_table[6][opponent][king_square[opponent]];
#endif
            }
         }
         break;
   }

   /* Mobility */
   //bb = get_attack_bitboard(board, player);
   //ev += bits_set(bb);
   //bb = get_attack_bitboard(board, opponent);
   //ev -= bits_set(bb);

   /* Tapered evaluation: if we approach the 50-move limit, then let the
    * score go to draw-ish smoothly.
    * Idea taken from Crafty
    */
   if (game->moves_played && game->fifty_counter[game->moves_played-1]>80)
      ev *= (100 - game->fifty_counter[game->moves_played-1])/20;


   return ev;
}
