#include <sys/time.h>
#include "computer.h"
#include "sbr.h"
#include "alphabeta.h"
#include "genrand.h"
#include "board.h"
#include "movegen.h"
#include "game.h"
#include "inline/game.h"

#ifdef DEBUG
#define abtrace printf
#else
#define abtrace(s,...)
#endif

#define PRINT_MOVE_EVALUATIONS
#undef CLEAR_KILLER_TABLE
#undef CLEAR_TRANSPOSITION_TABLE

uint64_t get_timer(void)
{
   struct timeval t;
   gettimeofday(&t, NULL);
   
   return t.tv_sec*1000000 + t.tv_usec; 
}

/* Retrieve principle variation from the hash table */
void retrieve_principle_variation(gamestate_t *game, move_t move)
{
   static int n = 30;
   hash_table_entry_t *hash;

   if (n == 0) return;
   n--;
   
   printf("%s ", move_string(move, NULL));
   playmove(game, move);

   /* Look up th resulting position in the hash table */
   hash = query_table_entry(game->transposition_table, game->board->hash);

   /* Recursive call to get next move from the table */
   if (hash)
      retrieve_principle_variation(game, hash->best_move);

   takeback(game);
}

static int movelist_sorter(void *score_ptr, const void *mi1, const void *mi2)
{
   int *score = (int*)score_ptr;
   int i1 = *(int*)mi1;
   int i2 = *(int*)mi2;

   return score[i2] - score[i1];
}

void computer_play(gamestate_t *game, int max_depth)
{
   uint64_t time, prev_time, prev_prev_time;
   alphabeta_state_t principle_variation;
   movelist_t movelist;
   int move_perm[MAX_MOVES];
   int score[MAX_MOVES];
   int alpha, beta;
   int try_fail_alpha, try_fail_beta;
   int previous_positions_evaluated;
   int best_score;
   int best_move;
   int legal_moves;
   int depth;
   int me;
   int n;

   /* Setup data structures */
   /* FIXME: unless these are cleared to 0, the search sometimes returns
    * *incorrect* evaluations of the moves!
    */
#ifdef CLEAR_KILLER_TABLE
   memset(game->killer, 0, sizeof game->killer);
#endif

#ifdef CLEAR_TRANSPOSITION_TABLE
   if (game->transposition_table) {
      memset(game->transposition_table->data, 0, game->transposition_table->number_of_elements*sizeof(hash_table_entry_t));
      memset(game->transposition_table->depth_data, 0, game->transposition_table->number_of_elements*sizeof(hash_table_entry_t));
   }
#endif

   memset(game->history, 0, sizeof game->history);

   generate_moves(&movelist, game->board, game->player);

   if (max_depth>MAX_SEARCH_DEPTH) max_depth = MAX_SEARCH_DEPTH;
   best_move = 0;
   best_score = -3*CHECKMATE;
   alpha = -3*CHECKMATE;
   beta = 3*CHECKMATE;
   principle_variation.size_of_variation = 0;

   me = game->player;
   legal_moves = movelist.num_moves;

   abort_search = false;
   previous_positions_evaluated = 0;
   positions_evaluated = 0;
   moves_searched = 0;
   positions_in_hashtable = 0;
   branches_pruned = 0;
   time = prev_time = prev_prev_time = 0;

   /* First step: de a 2-ply initial full-breath search (to establish move
    * ordering)
    */
   prev_time = time = get_timer();
   prev_prev_time = prev_time;
   for (n=0; n<movelist.num_moves; n++) {
      move_perm[n] = n;

      playmove(game, movelist.move[n]);
      if (!player_in_check(game, me)) {   /* We found a legal move! */
         alphabeta_state_t state =
            alphabeta(game, 1, 1, alpha, beta);

         score[n] = -state.score;
         if (score[n] > best_score) {
            best_score = score[n];
            best_move = move_perm[n];
            memcpy(&principle_variation, &state, sizeof state);
         }
      } else {
         legal_moves--;
         score[n] = -3*CHECKMATE;
      }
      takeback(game);
   }
   prev_prev_time = prev_time;
   prev_time = time;
   time = get_timer();

   /* Hmm... no legal moves? */
   if (legal_moves == 0)
      return;

   /* Sort the move list based on the score */
   qsort_r(move_perm, movelist.num_moves, sizeof *move_perm, score,
                                                            movelist_sorter);
   best_move = move_perm[0];
   printf("[%d] --> %s %g\n", 2,
         move_string(movelist.move[best_move], NULL), score[best_move]/100.0);

   /* Uh oh... */
   if (legal_moves == 1) {
      playmove(game, movelist.move[best_move]);
      return;
   }

   for (depth=2; depth<max_depth; depth++) {
      int best_score = score[best_move];
      int new_best_move = best_move;

      previous_positions_evaluated = positions_evaluated;
      //positions_evaluated = 0;

      for (n=0; n<legal_moves; n++) {
         alphabeta_state_t state;
         int reduction = 0;

         /* If we've found a forced check mate, there's no point in trying to
          * improve our score here...
          */
         if ((abs(best_score)+max_depth) >= CHECKMATE) break;

         /* Aspiration search */
         if (move_perm[n] == new_best_move) {
            alpha = best_score - 50;
            beta = best_score + 50;
         } else {
            alpha = best_score;
            beta = best_score + 1;
         }
         try_fail_alpha = alpha - 150;
         try_fail_beta = beta + 150;

#ifdef PRINT_MOVE_EVALUATIONS
       printf("[% 2d]     % 2d/% 2d %s (%.2f) %.2f", depth+1,
             n, legal_moves,
             move_string(movelist.move[move_perm[n]], NULL),
                best_score/100.0, score[move_perm[n]]/100.0);
#endif

         //reduction = n>1;
         playmove(game, movelist.move[move_perm[n]]);
#ifdef PRINT_MOVE_EVALUATIONS
         hash_table_entry_t *hash;
         hash = query_table_entry(game->transposition_table, game->board->hash);
#endif
         while (true) {
            //state = alphabeta(game, depth-reduction, depth-reduction, -beta, -alpha);
            state = alphabeta(game, depth-reduction, depth-reduction, -alpha, -beta);
            state.score = -state.score;
            abtrace("alphabeta [%d %d] returned %d\n",alpha,beta,state.score);

            /* Fail high/low, adjust window.
             * NB: this assumes fail soft alpha-beta, which may return a
             * score outside of [alpha, beta]!
             */
            if (state.score <= alpha) {
               alpha = state.score - 1;
//             alpha = try_fail_alpha;
               try_fail_alpha = -3*CHECKMATE;
               abtrace("Fail low %d, set alpha to %d\n", state.score, alpha);
               continue;
            }
            if (state.score >= beta) {
               beta = state.score + 1;
//             beta = try_fail_beta;
               try_fail_beta = 3*CHECKMATE;
               abtrace("Fail high %d, set beta to %d\n", state.score, beta);
               continue;
            }
            break;
         } 
         takeback(game);

#ifdef PRINT_MOVE_EVALUATIONS
         if (abort_search) {
            printf("(interrupted)\n");
            break;
         }
         printf("-> %.2f\n", state.score/100.0);
         /*
         if (state.size_of_variation) {
            int n;
            printf("         ");
            for (n=0; n<state.size_of_variation; n++) {
               printf("%s ",
                     move_string(state.mainline[n], NULL));
            }
            printf("\n");
         }
         */
         if (hash) {
            printf("         Transposition table: %.2f %02x %d ... %s\n",
                  -hash->score/100.0, hash->flags, hash->depth+1,
                  move_string(hash->best_move, NULL));
            printf("         <");

            if (state.size_of_variation) {
               int n;
               for (n=0; n<state.size_of_variation; n++) {
                  printf("%s ", move_string(state.mainline[n], NULL));
               }
            }

            //retrieve_principle_variation(game, movelist.move[move_perm[n]]);
            printf(">\n");
         } else {
            printf("         (no hash)\n");
         }
#endif
         if (abort_search) break;

         /* store principle variation */
         if (n == 0) 
            memcpy(&principle_variation, &state, sizeof state);

         /* New best move? */
         score[move_perm[n]] = state.score;
         if ( (state.score > best_score) || (new_best_move == move_perm[n])) {
            if (new_best_move != move_perm[n]) {
               printf("[% 2d] !!! %s %d %.2f -> %.2f\n", depth+1,
                  move_string(movelist.move[move_perm[n]], NULL),
                  new_best_move, best_score/100.0, state.score/100.0);
            }
            best_score = state.score;
            memcpy(&principle_variation, &state, sizeof state);
            new_best_move = move_perm[n];
            if (principle_variation.size_of_variation) {
               int n;
               printf("         ");
               for (n=0; n<principle_variation.size_of_variation; n++) {
                  printf("%s ",
                     move_string(principle_variation.mainline[n], NULL));
               }
               printf("\n");
            }
            store_table_entry(game->transposition_table, game->board->hash, 
                  depth, best_score, HASH_TYPE_EXACT,
                  movelist.move[new_best_move]);
         }
      }

      if (abort_search) break;

      /* Sort the move list based on the score */
      qsort_r(move_perm, movelist.num_moves, sizeof *move_perm, score,
            movelist_sorter);
      //new_best_move = move_perm[0];
      best_move = new_best_move;

      printf("[% 2d] --> %s %d %.2f (<BR> = %.2f)\n", depth+1,
                  move_string(movelist.move[best_move], NULL), best_move,
                     score[best_move]/100.0,
                     (float)moves_searched/positions_evaluated);
//                     (float)positions_evaluated/previous_positions_evaluated);
      if (principle_variation.size_of_variation) {
         printf("         ");
         for (n=0; n<principle_variation.size_of_variation; n++) {
            printf("%s ", move_string(principle_variation.mainline[n], NULL));
         }
         printf("\n");
      }

      prev_prev_time = prev_time;
      prev_time = time;
      time = get_timer();
      printf("EBR = %.2f\n", (float)(time - prev_time)/(prev_time - prev_prev_time));

      /* Break out early if we've found a checkmate */
      //if ((abs(best_score)+1000) >= CHECKMATE) {
      if ((abs(best_score)+depth) >= CHECKMATE) {
         int ply = CHECKMATE - abs(best_score);
         printf("Mate in %d (%d ply)\n", ply/2 + 1, ply);
         break;
      }
   }

   retrieve_principle_variation(game, movelist.move[best_move]);
   printf("\n");

   printf("%d nodes visited (%d [%.2f%%] in transposition table) %d branches pruned, %d moves searched (average branching ratio %.2f)\n",
         positions_evaluated, positions_in_hashtable,
         100.0*(float)positions_in_hashtable/(positions_in_hashtable +
         positions_evaluated), branches_pruned, moves_searched,
         (float)moves_searched/positions_evaluated);

   playmove(game, movelist.move[best_move]);

   return;
}
