#ifndef SQUARE_H
#define SQUARE_H

/* Map squares from the rotated bitboard back to the normal board. */
extern int diagonal_a8h1_map[64];
extern int diagonal_a1h8_map[64];
extern int diagonal_a8h1_square[64];
extern int diagonal_a1h8_square[64];
extern int diagonal_a8h1_offset[64];
extern int diagonal_a1h8_offset[64];
extern int diagonal_a8h1_ord[64];
extern int diagonal_a1h8_ord[64];
extern int diagonal_a8h1_diagonal[64];
extern int diagonal_a1h8_diagonal[64];
extern int diagonal_a8h1_inverse_map[64];
extern int diagonal_a1h8_inverse_map[64];
void construct_inverse_diagonal_maps(void);

/********************************************************************
 * Functions for dealing with packing/unpacking square rows/columns *
 ********************************************************************/
static inline int pack_row_column(int row, int column)
{
   return column + (row<<3);
}

static inline int unpack_row(int packed)
{
   return packed>>3;
}

static inline int unpack_column(int packed)
{
   return packed&7;
}

static inline int xchg_row_column(int packed)
{
   return (packed>>3) | ((packed & 7)<<3);
}

/********************************************************************
 * Functions for mapping squares onto a rotated bitboard (a8-h1)    * 
 ********************************************************************/
/* returns the number of the diagonal parallel to a8-h1 that the square is
 * on. Used for rotated bitboards.
 */
static inline int calculate_a8h1_diagonal(int packed)
{
   return unpack_row(packed)+unpack_column(packed);
}

/* Get the number of the square along the length of the diagonal, starting
 * from the lower right hand side of the board.
 * Also used for rotated bitboards.
 */
static inline int calculate_a8h1_diagonal_ord(int packed)
{
   int row = unpack_row(packed);
   int diag_nr = calculate_a8h1_diagonal(packed);

   /* We need to switch the way we count above the main diagonal, k will be
    * the switch to decide this: it is 0 below the diagonal and 1 above it.
    */
   int k = (diag_nr&8) >> 3;
   int n = k * (1 + (diag_nr&7) );
   return row - n;
}

/* Calculate the offset of the start of the diagonal in the rotated
 * bitboard. Again used for rotated bitboards.
 */
static inline int calculate_a8h1_diagonal_offset(int packed)
{
   int diag_nr = calculate_a8h1_diagonal(packed);

   /* We need to switch the way we count above the main diagonal, k will be
    * the switch to decide this: it is 0 below the diagonal and 1 above it.
    */
   int k = (diag_nr&8) >> 3;

   /* The total area under the diagonal, if we pretend that the board
    * continues indefinitely; really this is just Gauss' expression for the
    * sum of the integers from 1 to diag_nr, which is the number of squares
    * below this diagonal.
    * As long as we are below the main diagonal, we can use this number
    * directly, but above it we need to correct for the fact that the board
    * ends at the h-file.
    */
   int area = diag_nr*(diag_nr+1)/2;

   return area - k*((diag_nr&7) + 1)*(diag_nr&7);
}

/* Map a packed square to the equivalent square on the bitboard mapping the
 * main diagonal.
 * Again used for rotated bitboards.
 */
static inline int calculate_a8h1_diagonal_square(int packed)
{
   int diag_nr = calculate_a8h1_diagonal(packed);
   int row = unpack_row(packed);

   /* We need to switch the way we count above the main diagonal, k will be
    * the switch to decide this: it is 0 below the diagonal and 1 above it.
    */
   int k = (diag_nr&8) >> 3;

   /* The total area under the diagonal, if we pretend that the board
    * continues indefinitely; really this is just Gauss' expression for the
    * sum of the integers from 1 to diag_nr, which is the number of squares
    * below this diagonal.
    * As long as we are below the main diagonal, we can use this number
    * directly, but above it we need to correct for the fact that the board
    * ends at the h-file.
    */
   int area = diag_nr*(diag_nr+1)/2;

   return row + area - k*((diag_nr&7) + 1)*((diag_nr&7) + 1);
}

/*
 * Wrapper functions
 */
static inline int get_a8h1_diagonal(int packed)
{
   return calculate_a8h1_diagonal(packed);
}


/* Get the number of the square along the length of the diagonal, starting
 * from the lower right hand side of the board.
 * Also used for rotated bitboards.
 */
static inline int get_a8h1_diagonal_ord(int packed)
{
   return calculate_a8h1_diagonal_ord(packed);
}


/* Calculate the offset of the start of the diagonal in the rotated
 * bitboard. Again used for rotated bitboards.
 */
static inline int get_a8h1_diagonal_offset(int packed)
{
   return calculate_a8h1_diagonal_offset(packed);
}


/* Map a packed square to the equivalent square on the bitboard mapping the
 * main diagonal.
 * Again used for rotated bitboards.
 */
static inline int get_a8h1_diagonal_square(int packed)
{
   return calculate_a8h1_diagonal_square(packed);
}



/********************************************************************
 * Functions for mapping squares onto a rotated bitboard (a1-h8)    * 
 ********************************************************************/
/* returns the number of the diagonal parallel to a1-h8 that the square is
 * on. Used for rotated bitboards.
 */
static inline int calculate_a1h8_diagonal(int packed)
{
   return unpack_row(packed)+7-unpack_column(packed);
}

/* Get the number of the square along the length of the diagonal, starting
 * from the lower right hand side of the board.
 * Also used for rotated bitboards.
 */
static inline int calculate_a1h8_diagonal_ord(int packed)
{
   int row = unpack_row(packed);
   int diag_nr = calculate_a1h8_diagonal(packed);

   /* We need to switch the way we count above the main diagonal, k will be
    * the switch to decide this: it is 0 below the diagonal and 1 above it.
    */
   int k = (diag_nr&8) >> 3;
   int n = k * (1 + (diag_nr&7) );
   return row - n;
}

/* Calculate the offset of the start of the diagonal in the rotated
 * bitboard. Again used for rotated bitboards.
 */
static inline int calculate_a1h8_diagonal_offset(int packed)
{
   int diag_nr = calculate_a1h8_diagonal(packed);

   /* We need to switch the way we count above the main diagonal, k will be
    * the switch to decide this: it is 0 below the diagonal and 1 above it.
    */
   int k = (diag_nr&8) >> 3;

   /* The total area under the diagonal, if we pretend that the board
    * continues indefinitely; really this is just Gauss' expression for the
    * sum of the integers from 1 to diag_nr, which is the number of squares
    * below this diagonal.
    * As long as we are below the main diagonal, we can use this number
    * directly, but above it we need to correct for the fact that the board
    * ends at the h-file.
    */
   int area = diag_nr*(diag_nr+1)/2;

   return area - k*((diag_nr&7) + 1)*(diag_nr&7);
}

/* Map a packed square to the equivalent square on the bitboard mapping the
 * main diagonal.
 * Again used for rotated bitboards.
 */
static inline int calculate_a1h8_diagonal_square(int packed)
{
   int diag_nr = calculate_a1h8_diagonal(packed);
   int row = unpack_row(packed);

   /* We need to switch the way we count above the main diagonal, k will be
    * the switch to decide this: it is 0 below the diagonal and 1 above it.
    */
   int k = (diag_nr&8) >> 3;

   /* The total area under the diagonal, if we pretend that the board
    * continues indefinitely; really this is just Gauss' expression for the
    * sum of the integers from 1 to diag_nr, which is the number of squares
    * below this diagonal.
    * As long as we are below the main diagonal, we can use this number
    * directly, but above it we need to correct for the fact that the board
    * ends at the h-file.
    */
   int area = diag_nr*(diag_nr+1)/2;

   return row + area - k*((diag_nr&7) + 1)*((diag_nr&7) + 1);
}

/* returns the number of the diagonal parallel to a1-h8 that the square is
 * on. Used for rotated bitboards.
 */
static inline int get_a1h8_diagonal(int packed)
{
   return calculate_a1h8_diagonal(packed);
}

/* Get the number of the square along the length of the diagonal, starting
 * from the lower right hand side of the board.
 * Also used for rotated bitboards.
 */
static inline int get_a1h8_diagonal_ord(int packed)
{
   return calculate_a1h8_diagonal_ord(packed);
}

/* Calculate the offset of the start of the diagonal in the rotated
 * bitboard. Again used for rotated bitboards.
 */
static inline int get_a1h8_diagonal_offset(int packed)
{
   return calculate_a1h8_diagonal_offset(packed);
}

/* Map a packed square to the equivalent square on the bitboard mapping the
 * main diagonal.
 * Again used for rotated bitboards.
 */
static inline int get_a1h8_diagonal_square(int packed)
{
   return calculate_a1h8_diagonal_square(packed);
}

#endif
