#ifndef sbr_H
#define sbr_H

#include <stdint.h>

#if defined __GNUC__ && (defined __i386__ || defined __LP64__)
static inline int sbr32(uint32_t n)
{
   asm ("bsr %%eax, %%eax": "=a" (n) : "a" (n) );
   return n;
}
#else
static inline int sbr32(uint32_t n)
{
   int i = 0;
   for (n=n>>1; n; n=n>>1)
      i++;
   return i;
}
#endif

#if defined(__GNUC__) && defined(__LP64__)
static inline int sbr64(const uint64_t bb)
{
   uint64_t ret;
   asm (
         "bsrq %[bb], %[ret]"
         :[ret] "=r" (ret)
         :[bb] "mr" (bb)
       );
   return ret;
}
#else
static inline int sbr64(uint64_t n)
{
   uint32_t i = (uint32_t)(n>>32);
   return i ? 32 + sbr32(i) : sbr32(n);
}
#endif

#if defined __GNUC__ && (defined __i386__ || defined __LP64__)
static inline int sbf32(uint32_t n)
{
   asm ("bsf %%eax, %%eax": "=a" (n) : "a" (n) );
   return n;
}
#else
static inline int sbf32(uint32_t n)
{
   int i = 0;
   for (n=n<<1; n; n=n<<1)
      i++;
   return i;
}
#endif

#if defined(__GNUC__) && defined(__LP64__)
static inline int sbf64(const uint64_t bb)
{
   uint64_t ret;
   asm (
         "bsfq %[bb], %[ret]"
         :[ret] "=r" (ret)
         :[bb] "mr" (bb)
       );
   return ret;
}
#else
static inline int sbf64(uint64_t n)
{
   uint32_t i = (uint32_t)(n&0xFFFFFFFF);
   return i ? sbf32(i) : 32 + sbf32(n);
}
#endif

/* Returns true if n is a power of 2 */
/* Works for any iteger type */
#define is_power_of_two(n)    (!((n) & ((n)-1)))

/* Return the number of bits set on a bitboard
 * From http://chessprogramming.wikispaces.com/Population+Count
 */
static inline int bits_set(uint64_t x)
{
    const uint64_t k1 = 0x5555555555555555ll;
    const uint64_t k2 = 0x3333333333333333ll;
    const uint64_t k4 = 0x0f0f0f0f0f0f0f0fll;
    const uint64_t kf = 0x0101010101010101ll;
    x =  x       - ((x >> 1)  & k1);
    x = (x & k2) + ((x >> 2)  & k2);
    x = (x       +  (x >> 4)) & k4;
    x = (x * kf) >> 56;
    return (int) x;
}

/* Alternative to the above, for bitboards that are likely to be sparse
 * this is reportedly faster.
 * From http://chessprogramming.wikispaces.com/Population+Count
 * Due to Brian Kernighan.
 */
static inline int sparse_bits_set(uint64_t x) {
   int count = 0;
   while (x) {
       count++;
       x &= x - 1;
   }
   return count;
}


#endif
