#ifndef PIECES_H
#define PIECES_H

#include <stdint.h>

typedef uint8_t piece_t;

#define NUM_PIECE_TYPES  6

/* Pieces and their symbolic names/bitfield identifiers */
#define NONE     0
#define PAWN     (1<<0)
#define KNIGHT   (1<<1)
#define BISHOP   (1<<2)
#define ROOK     (1<<3)
#define QUEEN    (1<<4)
#define KING     (1<<5)

static const int piece_index[48] = {
   0, 
   0, 1, 0, 2,  0, 0, 0, 3,   0, 0, 0, 0,  0, 0, 0, 4,
   0, 0, 0, 0,  0, 0, 0, 0,   0, 0, 0, 0,  0, 0, 0, 5,
};

static const int piece_value_index[48] = {
   0, 
   1, 2, 0, 3,  0, 0, 0, 4,   0, 0, 0, 0,  0, 0, 0, 5,
   0, 0, 0, 0,  0, 0, 0, 0,   0, 0, 0, 0,  0, 0, 0, 0,
};

/* Bitfield colours */
#define WHITE    0
#define BLACK    128

/* Piece classes */
#define MAJOR     (ROOK|QUEEN)
#define MINOR     (KNIGHT|BISHOP)
#define OFFICER   (MAJOR|MINOR)
#define WOOD      (OFFICER|PAWN)    /* Van Wijgerden & Brunia: "hout" */
#define PIECE     (WOOD|KING)

/* Special move flags */
#define CASTLE_MOVE     (KING|ROOK|INIBIT)
#define ENPASSANT_MOVE  (PAWN|KING)
#define PROMOTION_MOVE  (PAWN|OFFICER)

/* Special evaluations */
#define CHECKMATE 100000
#define STALEMATE      0

/* Piece values                      Pawn Knight Bishop Rook Queen King */
//static const int piece_values[6] = { 100, 300,   300,   500, 900,  100000 };
extern int _chess_piece_values[NUM_PIECE_TYPES+1];

/* Getting information about pieces */
static inline uint8_t piece_colour(uint8_t piece)
{
   return piece&BLACK;
}

static inline int piece_value(uint8_t piece)
{
   return _chess_piece_values[piece_value_index[piece&PIECE]];
}

#endif
