#ifndef MOVE_H
#define MOVE_H

#include <stdbool.h>
#include <stdint.h>
#include "pieces.h"
#include "assert.h"

/* Define structure to hold a chess move */
/* FIXME: this should be generalised: we don't need to store the value of
 * the piece that was taken (because we revert to the previous board
 * position when we take back a move), but we could store the destination
 * square and the square that should be cleared during this move.
 * Normally these would be the same, but not for an en-passant capture. By
 * setting the data structure up like that we don't need special code to
 * treat en-passant captures.
 */
typedef union move_t {
   uint64_t m;             /* use only to compare/clear struct */
   struct {
      uint8_t piece;       /* Piece on source square (redundant?) */
      uint8_t from;        /* Origin square */
      uint8_t to;          /* Destination square */
      uint8_t capture;     /* Capture square; distinct because of en-passant */
      uint8_t tpiece;      /* Piece on destination square (for promotions) */
      uint8_t castle;      /* Castle move; actually xor mask for rook */
      uint8_t score;       /* score modifier, for move ordering */
      uint8_t ep:6;        /* En-passant square */
      uint8_t is_capture:1;/* Bit flag: this is a capture move */
   };
} move_t;

/* datastructure to hold the number of legal moves in a position */
#define MAX_MOVES    218   /* According to Wikipedia; seems safe... */
typedef struct movelist_t {
   move_t move[MAX_MOVES];
   int num_moves;
} movelist_t;

extern move_t irreversible_move;
char *short_move_string(const move_t move, movelist_t *movelist, char *buffer);
extern char *move_string(const move_t move, char *buffer);

/**********************************************************************
 *                      static inline functions                       *
 **********************************************************************/
/* Inlinable functions */
/* Get the player that makes the move */
static inline uint8_t get_move_player(const move_t move)
{
   return move.piece&BLACK;
}

/* Get the piece that moves */
static inline uint8_t get_move_piece(const move_t move)
{
   return move.piece;
}

/* Get the origin square */
static inline uint8_t get_origin(const move_t move)
{
   return move.from;
}

/* Get the destination square */
static inline uint8_t get_destination(const move_t move)
{
   return move.to;
}

/* Get the captured piece (if any) */
//static inline uint8_t get_captured_piece(const move_t move)
//{
//   return move.taken;
//}

/* Encode a standard move. Standard moves are everything apart from:
 *  1. Castling
 *  2. En-passant capture
 *  3. Promotion
 */
static inline move_t encode_normal_move(const uint8_t piece, const uint8_t from, const uint8_t to)
{
   move_t move;

   /* Initialise all elements to 0 */
   move.m = 0;

   move.piece = piece;
   move.from = from;
   move.to = to;
   move.capture = to;

   return move;
}

/* Encode a double pawn move; normal apart from that it stores the
 * en-passant square.
 */
static inline move_t encode_doublep_move(const uint8_t piece, const uint8_t from, const uint8_t to, const uint8_t ep)
{
   move_t move;

   /* Initialise all elements to 0 */
   move.m = 0;

   move.piece = piece;
   move.from = from;
   move.to = to;
   move.capture = to;
   move.ep = ep;

   return move;
}

/* Encode a promotion move. */
static inline move_t encode_promotion_move(const uint8_t tpiece, const uint8_t from, const uint8_t to)
{
   move_t move;

   /* Initialise all elements to 0 */
   move.m = 0;

   move.piece = PAWN | (tpiece & BLACK);
   move.tpiece = tpiece;
   move.from = from;
   move.to = to;
   move.capture = to;

   return move;
}

/* Encode en-passant capture */
static inline move_t encode_enpassant_capture(const uint8_t piece, const uint8_t from, const uint8_t to, const uint8_t ep)
{
   move_t move;

   /* Initialise all elements to 0 */
   move.m = 0;

   move.piece = piece;
   move.from = from;
   move.to = to;
   move.capture = ep;
   move.is_capture = 1;

   return move;
}

/* Encode a castling move */
static inline move_t encode_castling(const uint8_t piece, const uint8_t from, uint8_t to, uint8_t rook_bits)
{
   move_t move;
   move = encode_normal_move(piece, from, to);
   move.castle = rook_bits;

   return move;
}

/* Compare two moves */
static inline int moves_are_equal(const move_t m1, const move_t m2)
{
   return m1.m == m2.m;
}

/****************/
/*  Move types  */
/****************/
/* Returns true if a move is a promotion move */
#ifdef PROMOTION_MOVE               /* Defined for chess, but not Xiangqi */
static inline int is_promotion(const move_t move)
{
   return move.tpiece;
}
#endif

/* Returns TRUE if the move is an en-passant move */
#ifdef ENPASSANT_MOVE               /* Defined for chess, but not Xiangqi */
static inline int is_enpassant(const move_t move)
{
   return move.to != move.capture;
}
#endif

/* Returns TRUE if the move is a double pawn move */
#ifdef ENPASSANT_MOVE               /* Defined for chess, but not Xiangqi */
static inline int is_doublepawn(const move_t move)
{
   return (move.piece & PAWN) && (abs(move.from-move.to)==16);
}
#endif

/* returns TRUE is a move is a castling move */
#ifdef CASTLE_MOVE                  /* Defined for chess, but not Xiangqi */
static inline int is_castling(const move_t move)
{
   return move.castle;
}
#endif

/* returns TRUE is a move is a capture move */
static inline int is_capture(const move_t move)
{
   return move.is_capture;
}

/* Get promotion piece */
#ifdef PROMOTION_MOVE               /* Defined for chess, but not Xiangqi */
static inline int get_promotion_piece(const move_t move)
{
   return move.tpiece;
}
#endif

/* Get the square the captured piece was on.
 * This is the destination square for ordinary moves but not for en-passant
 * captures.
 */
static inline int get_captured_square(const move_t move)
{
   return move.capture;
}
#endif

