#ifndef INLINE_GAME_H
#define INLINE_GAME_H

/* These functions are static inline in header files so the compiler might
 * be able to inline them.
 */

/* We can check whether the move was a checking move without needing to do
 * the (slightly expensive) attack lookup: if neither the target nor the
 * destination square are in the 'superpiece' attack sphere
 */
static inline int move_checked_player(gamestate_t *game, int side, move_t move)
{
   uint8_t king_square = sbr64(and_bitboards(game->board->bbc[side>>7],
                                     game->board->bbp[5]));
   if (square_is_attacked(game->board, king_square, side^BLACK))
      return true;
   return false;
}

static inline bool player_in_check(gamestate_t *game, int colour)
{
   uint8_t king_square = sbr64(and_bitboards(game->board->bbc[colour>>7],
                                     game->board->bbp[5]));
   if (square_is_attacked(game->board, king_square, colour^BLACK))
      return true;
   return false;
}

/* Calculate the new hash key that would result if the move were played on
 * a board with hashkey "key"
 * Does not handle promotions or castling moves yet, which may cause
 * spurious table misses. Not desirable, but not fatal either.
 */
static inline uint64_t update_hashkey(const board_t *board, move_t move)
{
   int piece, colour;
   int key = board->hash;
   key ^= en_passant_key[move.ep];
   key ^= en_passant_key[board->ep_square];
   key ^= move_colour_key[1];
   piece = get_move_piece(move); 
   colour = (piece>>7);
   key ^= piece_key[colour][piece_index[piece&PIECE]][move.from];
   key ^= piece_key[colour][piece_index[piece&PIECE]][move.to];
   piece = get_piece(board, move.capture);
   if (piece) {
      colour = piece_colour(piece)>>7;
      key ^= piece_key[colour][piece_index[piece&PIECE]][move.capture];
   }
   return key;
}

/* Play a null-move; we don't simply flip the current player because that
 * will mess up the killer tables (and possibly the transposition table,
 * since the hash keys are not properly updated).
 */
static inline void play_null_move(gamestate_t *game)
{

   game->move_list[game->moves_played].m = 0;
   game->last_move = game->moves_played;
   game->moves_played++;
   game->board++;
   memcpy(game->board, game->board-1, sizeof *game->board);

   game->player ^= BLACK;
   game->board->hash ^= move_colour_key[1];
   //game->repetition_hash_table[game->board->hash&0xFFFF]++;
}

static inline void playmove(gamestate_t *game, move_t move)
{
   assert(game);

   /* The following assertion should be prevented from a higher level: we
    * don't want to check whether we have the space to store the next move
    * in a low level function like this. The search algorithm should make
    * sure that there is enough space to store to the end of the current
    * search depth before trying to play any moves. In other words,
    * max_moves-moves_played needs to be larger than the horizon depth,
    * allowing for quiescence search. We could take the horizon depth + 100
    * and always be safe.
    */
   assert(game->moves_played < game->max_moves);

   game->move_list[game->moves_played].m = move.m;
   game->moves_played++;
   game->last_move = game->moves_played;
   game->board++;
   memcpy(game->board, game->board-1, sizeof *game->board);

   /* Advance the 50-move counter if this move was reversible, reset if it
    * wasn't.
    */
   game->fifty_counter[game->moves_played] =
      (move.m & irreversible_move.m)?0:
      (game->fifty_counter[game->moves_played-1] + 1);

   /* Remove any piece that was captured at the destination */
   /* Redundant, it should be empty */
   //remove_piece(game->board, get_destination(move));

   /* Remove piece from the capture square (normally the same as the
    * destination, except on e-p moves)
    */
   remove_piece(game->board, move.capture);

   /* Move the piece to its new location */
   move_piece(game->board, get_move_piece(move), get_origin(move), get_destination(move));

   /* Store en-passant square
    * The hash key for this needs some clarification. If there is no EP
    * square, then move.ep = board->ep_square = 0 and this is a NOP. If
    * there is, then this places an extra white and black pawn on the board
    * on this square, which should be a fairly unique combination.
    * Note that the en-passant square is not set if no enemy pawn is
    * pressent to exact the capture, which is the common case. This
    * prevents pollution of the hash table where identical positions would
    * otherwise be stored differently.
    */
   //game->board->hash ^= piece_key[0][0][game->board->ep_square];
   //game->board->hash ^= piece_key[0][0][move.ep];
   //game->board->hash ^= piece_key[1][0][game->board->ep_square];
   //game->board->hash ^= piece_key[1][0][move.ep];
   game->board->hash ^= en_passant_key[move.ep];
   game->board->hash ^= en_passant_key[game->board->ep_square];
   game->board->ep_square = move.ep;

   /* Check special moves: pawn promotions and castling */
   if (is_promotion(move)) {
      remove_piece(game->board, get_destination(move));
      place_piece(game->board, get_destination(move), is_promotion(move));
   }

   /* Castling; the King has already been moved, now just need to move the
    * rook.
    */
   if (is_castling(move)) {
      castle_rook(game->board, (move.piece&BLACK)>>7, move.castle);
   }

   game->player ^= BLACK;
   //game->board->hash ^= move_colour_key[0];
   game->board->hash ^= move_colour_key[1];
   game->repetition_hash_table[game->board->hash&0xFFFF]++;

   /* Update check state */
   /* FIXME: do we really need this here? It slows things down quite a
    * bit... there might be a more clever way to do this.
    * Not that the perft test is misleading here - the actual search will
    * also do the test that we do here.
    */
   game->board->in_check = move_checked_player(game, game->player, move);
}

/* To take back a move, simply go back one spot in the array. We could save
 * memory by actually reversing the move, but what's the point? Memory is
 * cheap these days!
 */
static inline void takeback(gamestate_t *game)
{
   assert(game);
   if (game->moves_played) {
      game->repetition_hash_table[game->board->hash&0xFFFF]--;
      game->moves_played--;
      game->board--;
      game->player ^= BLACK;
   }
}

/* Like the regular takeback function, but doesn't check if any moves have
 * been played. Can be used instead of takeback in the search function to
 * avoid an unnecessary branch there.
 */
static inline void takeback_no_check(gamestate_t *game)
{
   game->moves_played--;
   game->board--;
   game->player ^= BLACK;
}

/* Redo the last move: if a move has been reverted, but no new move has
 * been played yet.
 * Mainly for player interaction.
 */
static inline void redo_last_move(gamestate_t *game)
{
   if (game->moves_played < game->last_move-1) {
      game->moves_played++;
      game->board++;
      game->player ^= BLACK;
   }
}

/* Again mainly for the benefit of user interaction: check if the move was
 * valid or left the king in check. Don't destroy the move history of the
 * game if it does.
 */
static inline bool playmove_was_ok(gamestate_t *game, move_t move)
{
   move_t old_move = game->move_list[game->moves_played+1];
   board_t old_board = game->board_list[game->moves_played+1];
   int old_last_move = game->last_move;

   memcpy(&old_board, game->board+1, sizeof *game->board);

   playmove(game, move);

   if (player_in_check(game, game->player^BLACK)) {
      takeback(game);
      game->move_list[game->moves_played+1] = old_move;
      memcpy(game->board+1, &old_board, sizeof *game->board);
      game->last_move = old_last_move;
      return false;
   }
   return true;
}

/* Again mainly for the benefit of user interaction: check if the current
 * position is a mate (check or stale). The search function will detect
 * this in a different way.
 */
static inline bool player_is_mate(gamestate_t *game)
{
   /* By making a move, we affect the *current* move and the *next* board. */
   move_t old_move = game->move_list[game->moves_played];
   board_t old_board = game->board_list[game->moves_played+1];
   int old_last_move = game->last_move;
   movelist_t movelist;
   int n;

   memcpy(&old_board, game->board+1, sizeof *game->board);

   generate_moves(&movelist, game->board, game->player);
   for (n=0; n<movelist.num_moves; n++) {
      int check;
      playmove(game, movelist.move[n]);
      check = player_in_check(game, game->player^BLACK);
      takeback(game);

      game->move_list[game->moves_played] = old_move;
      memcpy(game->board+1, &old_board, sizeof *game->board);
      game->last_move = old_last_move;
      if (!check)
         return false;
   }
   return true;
}
#endif
