#ifndef GAME_H
#define GAME_H

#include <string.h>
#include <stdbool.h>
#include "assert.h"
#include "config.h"

#include "move.h"
#include "board.h"
#include "hashtable.h"
#include "pawn_table.h"

typedef struct {
   uint16_t length_of_variation;
   move_t move[MAX_TOTAL_DEPTH];
} variation_t;

typedef struct gamestate_t {
   board_t *board;      /* pointer to the current board position */

   uint8_t player;      /* Who's turn it is to move */
   size_t moves_played; /* Number of moves played to current position */
   size_t last_move;    /* Number of the last move played in the game; useful when we take back a move */

   size_t max_moves;    /* Maximum number of moves that can be stored */
   board_t *board_list; /* Past board positions, so moves can be taken back */
   move_t *move_list;   /* list of moves played in the game */
   int16_t *score;          /* Evaluation score associated with each position */
   int8_t *fifty_counter;  /* Counter for the 50-moves rule */ 

   /* Killer moves, storage space requirements must come from the search
    * function.
    */
   move_t killer[3][MAX_TOTAL_DEPTH];
#ifdef USE_MATE_KILLER
   move_t mate_killer[MAX_TOTAL_DEPTH];
#endif

   /* Data structure for retrieving the principle variation. At each depth,
    * there are two branches for the tree: the principle variation and the
    * "current" branch. If the current branch turns out to be better than
    * the PV, it becomes the new PV.
    * A binary tree would be enough to store this information (and be easy
    * to manipulate), but for now we use a plain NxN array (of which half
    * the space is wasted, obviously)
    */
#ifdef PRINCIPLE_VARIATION_IN_GAME_STRUCT
   move_t principle_variation[MAX_TOTAL_DEPTH][MAX_TOTAL_DEPTH];
   int length_of_variation[MAX_TOTAL_DEPTH];
#endif

   /* Counter moves, for use with the counter move heuristic.
    * Should perhaps have an extra inex for the side to move?
    */
   move_t counter[64][64];

   /* History table: each move that generates a cut-off gets recorded in
    * this table.
    */
   int history[2][64][64];
   int max_history;

   /* Transposition (hash) table, with a separate one for QS */
   hash_table_t *transposition_table;
   hash_table_t *qs_table;

   /* Pawn structure hash table */
   pawn_hash_table_t *pawn_structure;

   /* Hash table for repetition detection */
   int8_t repetition_hash_table[0xFFFF+1];

   /* Principle variation and data structure for holding temporary
    * principle variation subtrees
    */
   //variation_t principle_variation;

   /* various things having to do with time management */
   int time_left[2];       /* Total time left on the clock, in msec */
   int time_inc[2];        /* Time increment per move, in msec */
   int time_per_move;      /* Time per move, in msec */
   uint64_t start_time;    /* Timer value when the search started */
   int movestogo;
   int extra_time;
   int running_clock;
   int root_moves_played;

   /* Various function pointers, so we can easily customise things and
    * adjust to different UIs.
    */
   void (*output_iteration)(const char *, ...) __attribute__((format(printf, 1, 2)));
   void (*uci_output)(const char *, ...) __attribute__((format(printf, 1, 2)));

   bool (*check_clock)(const struct gamestate_t *game);
} gamestate_t;

extern void playgame_init(void);
extern void playgame_shutdown(void);

/* Setters for virtual methods, so we can customise the engine output
 * easily
 */
extern void set_default_output_function(void (*func)(const char *, ...));
extern void set_uci_output_function(void (*func)(const char *, ...));

extern gamestate_t *create_game(void);
extern void start_new_game(gamestate_t *game);
extern void end_game(gamestate_t *game);

#endif
