#ifndef board_t_H
#define board_t_H

#include <stdbool.h>
#include "assert.h"
#include "sbr.h"
#include "square.h"
#include "bitboard.h"
#include "squares.h"
#include "pieces.h"
#include "inline/hashkey.h"
#include "piece_tables.h"

typedef struct board_t {
   /* Bitboard representation of the chess board.
    *  Index 0 is for pawns
    *  Index 1 is for knights
    *  Index 2 is for bishops
    *  Index 3 is for rooks
    *  Index 4 is for queens
    *  Index 5 is for kings
    */
   bitboard_t bbp[6];

   /* Bitboard encoding initial positions, used in move generation for
    * castling moves.
    */
   bitboard_t init;

   /* Board occupancy bitboards
    *  Index 0 is for white pieces
    *  Index 1 is for black pieces 
    */
   bitboard_t bbc[2];

   /* Flipped occupancy bitboard (rows & files interchanged), used to find
    * file occupation numbers, which are used in move generation. We make
    * no distinction between black and white here (it's not needed)
    */
   bitboard_t bbf;

   /* Rotated occupancy bitboards (+/- 45 degrees), used to find diagonal
    * occupancy numbers, which are used in move generation. Again, we make
    * no distinction between black and white here (not needed)
    */
   bitboard_t bbr[2];

   /* Hash key for this board */
   uint64_t hash, pawn_hash;
   
   /* En-passant square */
   int8_t ep_square;

   /* Material balance for black and white. It's a bit odd to put this here
    * (it more properly belongs in the game strunct than the board struct),
    * but it does save us some bookkeeping.
    */
   int16_t material[2];
   int16_t psq[2];

   /* Whether the current player is in check or not.
    * CAUTION: this is detected by the move generator, which cannot (should
    * not) modify the state of the board. That means this value need to be
    * set manually immediately AFTER the move generator is called if it is
    * to be used.
    */
   bool in_check;
} board_t;

extern void clear_board(board_t *board);
extern void begin_board(board_t *board);

extern void count_material(board_t *board);

extern void print_bitboards(board_t *board);

static inline uint8_t get_row_occupancy_number(const board_t *board, const int row)
{
   bitboard_t bb;
   assert(board);
   assert(row<8);

   bb = or_bitboards(board->bbc[0], board->bbc[1]);
   return get_bitboard_row(bb, row);
}

static inline uint8_t get_file_occupancy_number(const board_t *board, const int file)
{
   assert(board);
   assert(row<8);

   return get_bitboard_row(board->bbf, file);
}

/* Get the occupancy number for the diagonal parallel to h1-a8 that passes
 * through the indicated square.
 * The number of bits used by the occupancy number are given by the length
 * of the diagonal, which is 1+(diag_nr&7) (diag_nr starting from 0).
 * Any extra high bits are *not* cleared, however. If this is considered
 * necessary, the caller should do this explicitly.
 */
static inline uint8_t get_diagonal_a8h1_occupancy_number(const board_t *board, const int square)
{
   assert(board);
   assert(row<8);

   int shift = get_a8h1_diagonal_offset(square);
   return (board->bbr[0] >> shift) & 0xFF;
}

/* Get the occupancy number for the diagonal parallel to a1-a8 that passes
 * through the indicated square.
 * The number of bits used by the occupancy number are given by the length
 * of the diagonal, which is 1+(diag_nr&7) (diag_nr starting from 0).
 * Any extra high bits are *not* cleared, however. If this is considered
 * necessary, the caller should do this explicitly.
 */
static inline uint8_t get_diagonal_a1h8_occupancy_number(const board_t *board, const int square)
{
   assert(board);
   assert(row<8);

   int shift = get_a1h8_diagonal_offset(square);
   return (board->bbr[1] >> shift) & 0xFF;
}

static inline void get_occupancy_numbers(const board_t *board, int *row, int *file)
{
   assert(board);
   assert(row);
   assert(file);

   *row = get_row_occupancy_number(board, *row);
   *file = get_file_occupancy_number(board, *file);
}

/* Functions used to manipulate a chess board - written as inlineble functions */
static inline uint8_t get_piece(const board_t *board, const int where)
{
   assert(where<=H8);
   assert(board);
   bitboard_t bb_where = ((bitboard_t)1)<<where;
   int piece = ((board->bbc[1]>>where)&1)<<7;
   if (board->bbp[0] & bb_where)
      return piece | PAWN;
   if (board->bbp[1] & bb_where)
      return piece | KNIGHT;
   if (board->bbp[2] & bb_where)
      return piece | BISHOP;
   if (board->bbp[3] & bb_where)
      return piece | ROOK;
   if (board->bbp[4] & bb_where)
      return piece | QUEEN;
   if (board->bbp[5] & bb_where)
      return piece | KING;

   return piece;
}

/* Get the colour flag of the piece on this board.
 * CAUTION: this only answers the question "is this a black piece?"
 */
static inline uint8_t get_colour_flag(const board_t *board, const int where)
{
   assert(where<=H8);
   assert(board);

   return ((board->bbc[1]>>where)&1)<<7;
}

static inline void remove_piece(board_t *board, const int where)
{
   int colour;
   int piece;
   
   assert(where<=H8);
   assert(board);

   piece = get_piece(board, where);
   if (piece) {
      const int piece_idx = piece_index[piece&PIECE];

      colour = piece_colour(piece)>>7;
      board->material[colour] -= piece_value(piece);
      unset_bitboard(&(board->bbc[colour]), where);
      unset_bitboard(&(board->init), where);
      unset_bitboard(&(board->bbp[piece_index[piece&PIECE]]), where);
      unset_bitboard(&board->bbf, xchg_row_column(where));
      unset_bitboard(&board->bbr[0], get_a8h1_diagonal_square(where));
      unset_bitboard(&board->bbr[1], get_a1h8_diagonal_square(where));
      /* Update hash key */
      board->hash ^= piece_key[colour][piece_idx][where];
      if (piece & PAWN) {
         board->pawn_hash ^= piece_key[colour][piece_idx][where];
      }
      /* Update piece-square table evaluation */
      board->psq[colour] -= piece_square_table[piece_idx][colour][where];
   }
}

static inline void place_piece(board_t *board, const int where, const int piece)
{
   int colour;
   const int piece_idx = piece_index[piece&PIECE];
   assert(where<=H8);

   colour = piece_colour(piece)>>7;

   board->material[colour] += piece_value(piece);
   set_bitboard(&(board->bbc[colour]), where);
   set_bitboard(&(board->bbp[piece_index[piece&PIECE]]), where);
   set_bitboard(&board->bbf, xchg_row_column(where));
   set_bitboard(&board->bbr[0], get_a8h1_diagonal_square(where));
   set_bitboard(&board->bbr[1], get_a1h8_diagonal_square(where));
   
   /* Update hash key */
   board->hash ^= piece_key[colour][piece_idx][where];
   if (piece & PAWN) {
      board->pawn_hash ^= piece_key[colour][piece_idx][where];
   }
   /* Update piece-square table evaluation */
   board->psq[colour] -= piece_square_table[piece_idx][colour][where];
}

/* Combine some of the operations in the above functions for better use of
 * the computer power that comes from using bitboards.
 * NB: the destination square must be *empty* for this to work correctly,
 * in other words, this does NOT handle captures.
 */
static inline void move_piece(board_t *board, int piece, const int from, const int to)
{
   bitboard_t *bbc, *bbp, mask;
   const int piece_idx = piece_index[piece&PIECE];
   int colour;
   assert(from<=H8);
   assert(to<=H8);

   colour = piece_colour(piece)>>7;

   bbc = &board->bbc[colour];
   bbp = &board->bbp[piece_index[piece&PIECE]];
   mask = or_bitboards(make_bitboard_square(from), make_bitboard_square(to));

   *bbc = xor_bitboards(*bbc, mask);
   *bbp = xor_bitboards(*bbp, mask);
   board->init = mask_bitboards(board->init, mask);

   /* Flipped bitboard, rows and files interchanged */
   mask = or_bitboards(make_bitboard_square(xchg_row_column(from)),
                       make_bitboard_square(xchg_row_column(to)));
   board->bbf = xor_bitboards(board->bbf, mask);

   /* Rotated bitboards */
   mask = or_bitboards(make_bitboard_square(get_a8h1_diagonal_square(from)),
                       make_bitboard_square(get_a8h1_diagonal_square(to)));
   board->bbr[0] = xor_bitboards(board->bbr[0], mask);

   mask = or_bitboards(make_bitboard_square(get_a1h8_diagonal_square(from)),
                       make_bitboard_square(get_a1h8_diagonal_square(to)));
   board->bbr[1] = xor_bitboards(board->bbr[1], mask);

   /* Update hash key */
   board->hash ^= piece_key[colour][piece_idx][from];
   board->hash ^= piece_key[colour][piece_idx][to];
   if (piece & PAWN) {
      board->pawn_hash ^= piece_key[colour][piece_idx][from];
      board->pawn_hash ^= piece_key[colour][piece_idx][to];
   }
   /* Update piece-square table evaluation */
   board->psq[colour] += piece_square_table[piece_idx][colour][to];
   board->psq[colour] -= piece_square_table[piece_idx][colour][from];
}

/* cidx should be 0 or 1, for white and black respectively */
static inline void castle_rook(board_t *board, int cidx, uint8_t rook_bits)
{
   bitboard_t bb = make_bitboard_row(rook_bits, cidx*7);
   bitboard_t bbf = 0ll, bbr1 = 0ll, bbr2 = 0ll;

   /* Move rook by flipping bits on the relevant bitboards. This flips both
    * bits simultaneously.
    */
   board->bbp[3] = xor_bitboards(board->bbp[3], bb);
   board->bbc[cidx] = xor_bitboards(board->bbc[cidx], bb);
   board->init = mask_bitboards(board->init, bb);


   /* Now for the tricky part: the rotated bitboards. What bits to flip
    * there we have to look up from a table containing the 4 possible
    * combinations (black/white, king/queen).
    * FIXME: for now, we calculate the bitboards here as they are needed.
    * Probably doesn't hurt so much as castling doesn't happen that often.
    */
   if (cidx == 0) {     /* White */
      if ( (rook_bits & 1) ) {      /* Queen side */
         bbf = or_bitboards(make_bitboard_square(xchg_row_column(A1)),
                            make_bitboard_square(xchg_row_column(D1)));
         bbr1 = or_bitboards( 
                        make_bitboard_square(get_a8h1_diagonal_square(A1)),
                        make_bitboard_square(get_a8h1_diagonal_square(D1)));
         bbr2 = or_bitboards( 
                        make_bitboard_square(get_a1h8_diagonal_square(A1)),
                        make_bitboard_square(get_a1h8_diagonal_square(D1)));
      } else {                      /* King side */
         bbf = or_bitboards(make_bitboard_square(xchg_row_column(H1)),
                            make_bitboard_square(xchg_row_column(F1)));
         bbr1 = or_bitboards( 
                        make_bitboard_square(get_a8h1_diagonal_square(H1)),
                        make_bitboard_square(get_a8h1_diagonal_square(F1)));
         bbr2 = or_bitboards( 
                        make_bitboard_square(get_a1h8_diagonal_square(H1)),
                        make_bitboard_square(get_a1h8_diagonal_square(F1)));
      }
   } else {          /* Black */
      if ( (rook_bits & 1) ) {      /* Queen side */
         bbf = or_bitboards(make_bitboard_square(xchg_row_column(A8)),
                            make_bitboard_square(xchg_row_column(D8)));
         bbr1 = or_bitboards( 
                        make_bitboard_square(get_a8h1_diagonal_square(A8)),
                        make_bitboard_square(get_a8h1_diagonal_square(D8)));
         bbr2 = or_bitboards( 
                        make_bitboard_square(get_a1h8_diagonal_square(A8)),
                        make_bitboard_square(get_a1h8_diagonal_square(D8)));
      } else {                      /* King side */
         bbf = or_bitboards(make_bitboard_square(xchg_row_column(H8)),
                            make_bitboard_square(xchg_row_column(F8)));
         bbr1 = or_bitboards( 
                        make_bitboard_square(get_a8h1_diagonal_square(H8)),
                        make_bitboard_square(get_a8h1_diagonal_square(F8)));
         bbr2 = or_bitboards( 
                        make_bitboard_square(get_a1h8_diagonal_square(H8)),
                        make_bitboard_square(get_a1h8_diagonal_square(F8)));
      }
   }

   /* Flip bits on rotated bitboards */
   board->bbf = xor_bitboards(board->bbf, bbf);
   board->bbr[0] = xor_bitboards(board->bbr[0], bbr1);
   board->bbr[1] = xor_bitboards(board->bbr[1], bbr2);

   /* Update hash key */
   board->hash ^= castle_rook_key[cidx][rook_bits&1];
}
#endif
