#ifndef bitboard_t_H
#define bitboard_t_H

#include <stdint.h>
#include "assert.h"
#include "square.h"
#include "sbr.h"

typedef uint64_t bitboard_t;

/* Special bitboards, used for masking and move generation:
 * The "board_empty" and "board_all" bitboards speak for themselves.
 * The "board_minus_(a|h)" boards are used in move generation when
 * shifting pawn bit boards.
 * The centre, xcentre, kwing and qwing boards can be used in tactical
 * situations to specifically look for moves in that part of the board.
 * Rank encodings are mainly used in move generation, but could also be
 * used in function evaluations (think rooks on the 7th row).
 * The "board_ep" bitboard is used in move generation: valid en-passant
 * squares occur along these rows.
 * The "castlek" and "castleq" bitboards are bitmasks that have to fit the
 * king, rook and inibit bitboards; used in move generation. Their "i"
 * counter parts mark the squares that have to be empty.
 */
static const bitboard_t board_empty   = 0x0000000000000000ll;
static const bitboard_t board_all     = 0xFFFFFFFFFFFFFFFFll;
static const bitboard_t board_minus_a = 0xFEFEFEFEFEFEFEFEll;
static const bitboard_t board_minus_h = 0x7F7F7F7F7F7F7F7Fll;
static const bitboard_t board_centre  = 0x0000001818000000ll;
static const bitboard_t board_xcentre = 0x00003C24243C0000ll;
static const bitboard_t board_kwing   = 0xF0F0F0F0F0F0F0F0ll;
static const bitboard_t board_qwing   = 0x0F0F0F0F0F0F0F0Fll;
static const bitboard_t board_dark    = 0xAA55AA55AA55AA55ll;
static const bitboard_t board_light   = 0x55AA55AA55AA55AAll;
static const bitboard_t board_rank1   = 0x00000000000000FFll;
static const bitboard_t board_rank2   = 0x000000000000FF00ll;
static const bitboard_t board_rank3   = 0x0000000000FF0000ll;
static const bitboard_t board_rank4   = 0x00000000FF000000ll;
static const bitboard_t board_rank5   = 0x000000FF00000000ll;
static const bitboard_t board_rank6   = 0x0000FF0000000000ll;
static const bitboard_t board_rank7   = 0x00FF000000000000ll;
static const bitboard_t board_rank8   = 0xFF00000000000000ll;
static const bitboard_t board_afile   = 0x0101010101010101ll;
static const bitboard_t board_bfile   = 0x0202020202020202ll;
static const bitboard_t board_cfile   = 0x0303030303030303ll;
static const bitboard_t board_dfile   = 0x0404040404040404ll;
static const bitboard_t board_efile   = 0x1010101010101010ll;
static const bitboard_t board_ffile   = 0x2020202020202020ll;
static const bitboard_t board_gfile   = 0x4040404040404040ll;
static const bitboard_t board_hfile   = 0x8080808080808080ll;
static const bitboard_t board_ep      = 0x0000FF0000FF0000ll;
static const bitboard_t board_castlek = 0x9000000000000090ll;
static const bitboard_t board_castleki= 0x6000000000000060ll;
static const bitboard_t board_castleq = 0x1100000000000011ll;
static const bitboard_t board_castleqi= 0x0E0000000000000Ell;

/* Rook castling masks: these are to be xor'ed with the rook's position
 * bitboards to move the rook to its proper square.
 * For kingside castling, this flips bits 5 & 7 (h->f), so 0xA0
 * For queenside castling, this flips bits 0 & 3 (a->d), so 0x09
 */
static const uint8_t castlek_rook_bits = 0xA0;
static const uint8_t castleq_rook_bits = 0x09;

extern void printf_bitboard(bitboard_t b);

/*******************************************/
/* Bitboard manipulation routines (inline) */
/*******************************************/

/* Set a bit on the bitboard */
static inline void set_bitboard(bitboard_t *board, const int square)
{
   assert(board);
   (*board) |= ((uint64_t)1)<<square;
}

/* Toggles a square on a bitboard */
static inline void toggle_bitboard(bitboard_t *board, const int square)
{
   assert(board);
   (*board) ^= ((uint64_t)1)<<square;
}

/* Clears a square on the bitboard */
static inline void unset_bitboard(bitboard_t *board, const int square)
{
   assert(board);
   bitboard_t mask = ~((uint64_t)1<<square);
   (*board) &= mask;
}

/* Returns true if a square on the bitboard is set */
static inline int get_bitboard(bitboard_t *board, const int square)
{
   assert(board);
   return ((*board) & ((uint64_t)1)<<square)!=0;
}

/* ANDs two bitboards */
static inline bitboard_t and_bitboards(const bitboard_t b1, const bitboard_t b2)
{
   return b1&b2;
}

/* MASKs two bitboards: set bits in first bitboard to 0 if set in second */
static inline bitboard_t mask_bitboards(const bitboard_t b1, const bitboard_t b2)
{
   return b1&~b2;
}

/* XORs two bitboards */
static inline bitboard_t xor_bitboards(const bitboard_t b1, const bitboard_t b2)
{
   return b1^b2;
}

/* ORs two bitboards */
static inline bitboard_t or_bitboards(const bitboard_t b1, const bitboard_t b2)
{
   return b1|b2;
}

/* Inverts a bitboard */
static inline bitboard_t invert_bitboard(const bitboard_t b)
{
   return ~b;
}

/* Retrieve one row from a bitboard */
static inline uint8_t get_bitboard_row(const bitboard_t b, const int row)
{
   return b >> (row*8);
}

/* Return a bitboard representation of a given square */
static inline bitboard_t make_bitboard_square(const int square)
{
   return ((uint64_t)1)<<square;
}

/* Expand one row into a full bitboard */
static inline bitboard_t make_bitboard_row(const uint8_t data, const int row_nr)
{
   return ((uint64_t)data) << (row_nr*8);
}

/* Expand one packed file into a full bitboard */
static inline bitboard_t make_bitboard_file(const uint8_t data, const int file_nr)
{
   bitboard_t b;
   int n;
   b = 0;
   for(n=0;n<8;n++) {
      if (data & (1<<n)) {
         set_bitboard(&b, pack_row_column(n, file_nr));
      }
   }
   return b;
}

static inline bitboard_t make_bitboard_a8h1_diagonal(uint8_t data, int square)
{
   bitboard_t bb = 0;
   int diag_nr = get_a8h1_diagonal(square);
   int offset = get_a8h1_diagonal_offset(square);
   /* Find the length of the diagonal: this is the number of bits in
    * the occupancy number that we actually care about.
    */
   int diag_length = 8 - abs(7 - diag_nr);
   int n;
   for (n=0; n<diag_length; n++)
      if (data & (1<<n))
         set_bitboard(&bb, diagonal_a8h1_inverse_map[offset+n]);

   return bb;
}

static inline bitboard_t make_bitboard_a1h8_diagonal(uint8_t data, int square)
{
   bitboard_t bb = 0;
   int diag_nr = get_a1h8_diagonal(square);
   int offset = get_a1h8_diagonal_offset(square);
   /* Find the length of the diagonal: this is the number of bits in
    * the occupancy number that we actually care about.
    */
   int diag_length = 8 - abs(7 - diag_nr);
   int n;
   for (n=0; n<diag_length; n++)
      if (data & (1<<n))
         set_bitboard(&bb, diagonal_a1h8_inverse_map[offset+n]);

   return bb;
}

/* Shift a bitboard to the right (that is, a->b, b->c etc.) by 1 file */
/*  and discards what gets shifted in on the left */
/* Note that shifting the board to the right implies shifting the binary */
/*  values to the left, since files to the right have a higher index */
static inline bitboard_t shift_bitboard_file_right(const bitboard_t b)
{
   return and_bitboards(board_minus_a, (b<<1));
}

/* Shift a bitboard to the left (that is, b->a, c->b etc.) by 1 file */
/*  and discards what gets shifted in on the right */
/* Note that shifting the board to the left implies shifting the binary */
/*  values to the right, since files to the left have a lower index */
static inline bitboard_t shift_bitboard_file_left(const bitboard_t b)
{
   return and_bitboards(board_minus_h, (b>>1));
}

/* Shift a bitboard up one row (that is, 1->2, 2->3 etc */
static inline bitboard_t shift_bitboard_row_up(const bitboard_t b)
{
   return b<<8;
}

/* Shift a bitboard down one row (that is, 1->2, 2->3 etc */
static inline bitboard_t shift_bitboard_row_down(const bitboard_t b)
{
   return b>>8;
}

/* Combinations of the above */
static inline bitboard_t shift_bitboard_right_up(const bitboard_t b)
{
   return shift_bitboard_row_up(shift_bitboard_file_right(b));
}

static inline bitboard_t shift_bitboard_right_down(const bitboard_t b)
{
   return shift_bitboard_row_down(shift_bitboard_file_right(b));
}

static inline bitboard_t shift_bitboard_left_up(const bitboard_t b)
{
   return shift_bitboard_row_up(shift_bitboard_file_left(b));
}

static inline bitboard_t shift_bitboard_left_down(const bitboard_t b)
{
   return shift_bitboard_row_down(shift_bitboard_file_left(b));
}
#endif
