
#include "HueWheelRR.hpp"



void DrawArrow(BITMAP* bmp , int sx , int sy , int dx , int dy , int color) {
  if (!((sx == dx) && (sy == dy))) {
    double arad = atan2(dy - sy , dx - sx) + M_PI;
    double aleft = arad + M_PI/6.0;
    double aright = arad - M_PI/6.0;
    double length = 10.0;
    int d2x = int(dx + length*cos(aleft));
    int d2y = int(dy + length*sin(aleft));
    int d3x = int(dx + length*cos(aright));
    int d3y = int(dy + length*sin(aright));
    line(bmp , sx  , sy  , dx , dy , color);
    line(bmp , d2x , d2y , dx , dy , color);
    line(bmp , d3x , d3y , dx , dy , color);
  }
}



bool HueWheelRR::ReAllocateBitmaps() {
  FreeBitmaps();
  huewheel    = create_bitmap(width , width);
  hwbuffer    = create_bitmap(width , width);
  svtriangle  = create_bitmap(width , width);
  svindicator = create_bitmap(indwidth,indwidth);
  buffer      = create_bitmap(width , width);
  
  if (!huewheel || !hwbuffer || !svtriangle || !svindicator || !buffer) {
    FreeBitmaps();
    return false;
  }
  return true;
}



void HueWheelRR::FreeBitmaps() {
  if (huewheel)   {destroy_bitmap(huewheel);}
  if (hwbuffer)   {destroy_bitmap(hwbuffer);}
  if (svtriangle) {destroy_bitmap(svtriangle);}
  if (svindicator) {destroy_bitmap(svindicator);}
  if (buffer)     {destroy_bitmap(buffer);}
  huewheel = hwbuffer = svtriangle = svindicator = buffer = 0;
}



void HueWheelRR::InitializeVertices() {
  V3D_f zero = {0.0,0.0,0.0,0,0,0};
  vH = vW = vB = vM =     zero;
  vecHB = vecBW = vecWH = zero;
  vH_ol = vW_ol = vB_ol = zero;
}



void HueWheelRR::ResetVerticePositions() {
  // Fixed angle equilateral triangle pointing right, with full hue on the right,
  //   saturation zero on the upper left, and value zero on the lower left
  static const double sine2pi_over_3 = sin((2.0*M_PI)/3.0);
  static const double cos2pi_over_3 = cos((2.0*M_PI)/3.0);
  static const double circle = 2.0*M_PI;
  
  double radius = irad - 10;
  if (radius < 1.0) {radius = 1.0;}
  double xoffset = radius*cos2pi_over_3;
  double yoffset = radius*sine2pi_over_3;
  
  vH.x = midx + radius;
  vH.y = midy;
  vW.x = midx + xoffset;
  vW.y = midy - yoffset;
  vB.x = midx + xoffset;
  vB.y = midy + yoffset;
  vM.x = (vH.x + vW.x)/2.0f;
  vM.y = (vH.y + vW.y)/2.0f;
  
  vecHB.x = vB.x - vH.x;
  vecHB.y = vB.y - vH.y;
  vecBW.x = vW.x - vB.x;
  vecBW.y = vW.y - vB.y;
  vecWH.x = vH.x - vW.x;
  vecWH.y = vH.y - vW.y;
  
  radius += 5;
  xoffset = radius*cos2pi_over_3;
  yoffset = radius*sine2pi_over_3;
  
  vH_ol.x = midx + radius;
  vH_ol.y = midy;
  vW_ol.x = midx + xoffset;
  vW_ol.y = midy - yoffset;
  vB_ol.x = midx + xoffset;
  vB_ol.y = midy + yoffset;
  
  distHB = sqrt(vecHB.x*vecHB.x + vecHB.y*vecHB.y);
  distBW = sqrt(vecBW.x*vecBW.x + vecBW.y*vecBW.y);
  distWH = sqrt(vecWH.x*vecWH.x + vecWH.y*vecWH.y);
  distMW = distWH/2.0;
  
  double BMxdiff = vM.x - vB.x;
  double BMydiff = vM.y - vB.y;
  distBM = sqrt(BMxdiff*BMxdiff + BMydiff*BMydiff);
  
  angle_HB_rad = atan2(vB.y - vH.y , vB.x - vH.x);
  if (angle_HB_rad < 0.0) {angle_HB_rad += circle;}
  
  angle_BW_rad = atan2(vW.y - vB.y , vW.x - vB.x);
  if (angle_BW_rad < 0.0) {angle_BW_rad += circle;}
  
  angle_WH_rad = atan2(vH.y - vW.y , vH.x - vW.x);
  if (angle_WH_rad < 0.0) {angle_WH_rad += circle;}
  
  angle_BM_rad = atan2(vM.y - vB.y , vM.x - vB.x);
  if (angle_BM_rad < 0.0) {angle_BM_rad += circle;}
  
  angle_MW_rad = atan2(vW.y - vH.y , vW.x - vH.x);// angle MW is the same as HW
  if (angle_MW_rad < 0.0) {angle_MW_rad += circle;}
  
  angle_MB_rad = angle_BM_rad - M_PI;
  if (angle_MB_rad < 0.0) {angle_MB_rad += circle;}

}



void HueWheelRR::ResetVerticeColors() {
  int red(0),green(0),blue(0);
  hsv_to_rgb(h,1.0,1.0,&red,&green,&blue);
  /// Fudge for making draw_sprite work when the hue produces magic pink
  if ((red == 255) && (green == 0) && (blue == 255)) {
    red = 254;
    blue = 254;
  }
  vH.c = makecol(red,green,blue);
  vW.c = makecol(255,255,255);
  vB.c = makecol(0,0,0);
  int outline_color = makecol(127,127,127);
  vH_ol.c = outline_color;
  vW_ol.c = outline_color;
  vB_ol.c = outline_color;
}



void HueWheelRR::ResetVertices() {
  ResetVerticePositions();
  ResetVerticeColors();
}



void HueWheelRR::ResetIndicatorPos() {
  svix = int(vB.x + v*(vecBW.x + s*vecWH.x));
  sviy = int(vB.y + v*(vecBW.y + s*vecWH.y));
}



bool HueWheelRR::DrawStartingHueWheel() {
  clear_to_color(huewheel , bitmap_mask_color(huewheel));
  clear_to_color(hwbuffer , bitmap_mask_color(hwbuffer));
  BITMAP* mask = create_bitmap(huewheel->w , huewheel->h);
  if (mask) {
    const int maskcolor = makecol(0,255,0);
    clear_to_color(mask , maskcolor);
    circlefill(mask , midx , midy , orad + 1, bitmap_mask_color(mask));
    circlefill(mask , midx , midy , irad - 2 , maskcolor);
    
    int rx = 0, ry = 0;
    
//    double angle_rad = 0.0;
    const double rad_to_deg = 180.0/M_PI;
    
    double hue = 0.0 , sat = 1.0 , val = 1.0;
    int red = 0 , green = 0 , blue = 0;
    int color = 0;
    
    for (int ypos = 0 ; ypos < mask->h ; ++ypos) {
      for (int xpos = 0 ; xpos < mask->w ; ++xpos) {
        if (getpixel(mask , xpos , ypos) != maskcolor) {
          rx = xpos - midx;
          ry = ypos - midy;
          hue = atan2(ry,rx)*rad_to_deg;
          if (hue < 0.0) (hue += 360.0);
          hsv_to_rgb(hue,sat,val,&red,&green,&blue);
          /// Fudge the colors so this bitmap will work with draw_sprite
          if ((red == 255) && (green == 0) && (blue == 255)) {
            red = 254;
            blue = 254;
          }
          color = makecol(red,green,blue);
          putpixel(huewheel , xpos , ypos , color);
        }
      }
    }
    
    /// Create a list of pixels on the outer and inner edges of the hue wheel so when
    /// it is rotated into the hwbuffer the edges can be cleaned consistently - this 
    /// way the shape should never change when it is rotated
    int outline_pixel_count = 0;
    clear_to_color(mask , bitmap_mask_color(mask));
    circlefill(mask , midx , midy , orad + 2 , maskcolor);
    circlefill(mask , midx , midy , orad , bitmap_mask_color(mask));
    circlefill(mask , midx , midy , irad - 1 , maskcolor);
    circlefill(mask , midx , midy , irad - 3 , maskcolor);
    for (int ypos = 0 ; ypos < mask->h ; ++ypos) {
      for (int xpos = 0 ; xpos < mask->w ; ++xpos) {
        if (getpixel(mask , xpos , ypos) == maskcolor) {
          outline_pixel_count += 1;
        }
      }
    }
    num_outline_pixels = outline_pixel_count;
    if (hw_outline) {delete [] hw_outline;}
    hw_outline = new Pos2d[outline_pixel_count];
    outline_pixel_count = 0;
    for (int ypos = 0 ; ypos < mask->h ; ++ypos) {
      for (int xpos = 0 ; xpos < mask->w ; ++xpos) {
        if (getpixel(mask , xpos , ypos) == maskcolor) {
          hw_outline[outline_pixel_count] = Pos2d(xpos,ypos);
          ++outline_pixel_count;
        }
      }
    }
    
    destroy_bitmap(mask);
    return true;
  }
  return false;
}



void HueWheelRR::DrawStartingSatValTriangle() {
  clear_to_color(svtriangle , bitmap_mask_color(svtriangle));
  triangle3d_f(svtriangle , POLYTYPE_GCOL , NULL , &vH_ol , &vW_ol , &vB_ol);
}



void HueWheelRR::DrawSVIndicator() {
  int indmidx = indrad + 1;
  int indmidy = indrad + 1;
  clear_to_color(svindicator , bitmap_mask_color(svindicator));
  circlefill(svindicator , indmidx , indmidy , indrad     , makecol(255,255,255));
  circlefill(svindicator , indmidx , indmidy , indrad - 1 , makecol(0,0,0));
  circlefill(svindicator , indmidx , indmidy , indrad - 2 , bitmap_mask_color(svindicator));
}



void HueWheelRR::DrawHueWheel() {
  draw_sprite(buffer , hwbuffer , 0 , 0);
}



void HueWheelRR::DrawSatValTriangle() {
  triangle3d_f(svtriangle , POLYTYPE_GCOL , NULL , &vH , &vW , &vB);
}



void HueWheelRR::DrawHueWheelBuffer() {
  static const double deg_to_afix = 256.0/360.0;
  rotate_sprite(hwbuffer , huewheel , 0 , 0 , ftofix(hue_angle_deg*deg_to_afix));
  const int mask_color = bitmap_mask_color(hwbuffer);
  for (int i = 0 ; i < num_outline_pixels ; ++i) {
    putpixel(hwbuffer , hw_outline[i].x , hw_outline[i].y , mask_color);
  }
}



void HueWheelRR::UpdateBuffer() {
  if (hue_changed || sv_changed) {
    if (hue_changed) {
      draw_sprite(buffer , hwbuffer , 0 , 0);
    }
    if (hue_changed || sv_changed) {
      draw_sprite(buffer , svtriangle , 0 , 0);
    }
  }
}



HueWheelRR::HueWheelRR(int xpos , int ypos , int inner_radius , int outer_radius , int indicator_radius) :/// = 5) :
          tlx(xpos) , tly(ypos) , cx(xpos + outer_radius + 1) , cy(ypos + outer_radius + 1) ,
          midx(outer_radius + 1) , midy(outer_radius + 1) , 
          width(2*outer_radius + 3) , irad(inner_radius) , orad(outer_radius) ,
          indrad(indicator_radius) , indwidth(2*indicator_radius + 3) ,
          // V3D_f objects initialized in the InitializeVertices() function
          svix(0) , sviy(0) ,
          distWH(0.0) , distBM(0.0) , distMW(0.0) ,
          angle_HB_rad(0.0) , angle_BW_rad(0.0) , angle_WH_rad(0.0) ,
          angle_BM_rad(0.0) , angle_MW_rad(0.0) , angle_MB_rad(0.0) ,
          h(0.0f) , s(1.0f) , v(1.0f) , 
          r(255) , g(0) , b(0) , rgbcolor(0) ,
          hue_angle_deg(0.0) , old_hue_angle_deg(0.0) , click_angle_deg(0.0) ,
          old_mbtn(0) ,
          drag_tick_count(0) , drag_min_ticks(1) ,
          hue_click(false) , hue_drag(false) , sv_click(false) , sv_drag(false) ,
          hue_changed(false) , sv_changed(false) ,
          NewHue(0.0) , NewSaturation(1.0) , NewValue(1.0) ,
          huewheel(0) , hwbuffer(0) , svtriangle(0) , svindicator(0) , buffer(0) ,
          hw_outline(0) ,
          num_outline_pixels(0) {
  rgbcolor = makecol(r,g,b);
  InitializeVertices();
  ResetVertices();
  ResetIndicatorPos();
  if (ReAllocateBitmaps()) {
    DrawStartingHueWheel();
    DrawStartingSatValTriangle();
    DrawHueWheelBuffer();
    DrawHueWheel();
    DrawSatValTriangle();
    DrawSVIndicator();
    clear_to_color(buffer , bitmap_mask_color(buffer));
    hue_changed = true;
    sv_changed = true;
    UpdateBuffer();
  }
}



HueWheelRR::~HueWheelRR() {
  FreeBitmaps();
  if (hw_outline) {delete [] hw_outline;}
}



void HueWheelRR::Draw(BITMAP* dest) {
  draw_sprite(dest , buffer , tlx , tly);
  draw_sprite(dest , svindicator , tlx + svix - indwidth/2 , tly + sviy - indwidth/2);
}



//bool HueWheelRR::HandleMouseInput(int mpos , int mbtn) {
bool HueWheelRR::HandleMouseInput(const int mx , const int my , int mbtn) {
  static const double rad_to_deg = 180.0/M_PI;
/*
  int mx = mpos >> 16;
  int my = mpos & 0x0000ffff;
  mx -= tlx;
  my -= tly;
*/
  int ox = 0;
  int oy = 0;
  hue_changed = false;
  sv_changed = false;
  
  if (hue_click) {
    if (mbtn & 1) {hue_drag = true;drag_tick_count = 0;}
  }
  if (sv_click) {
    if (mbtn & 1) {sv_drag = true;drag_tick_count = 0;}
  }
  
  if (hue_drag) {
    if (mbtn & 1) {
      // still held down
      ++drag_tick_count;
    } else {
      // released
      hue_drag = false;
      if (drag_tick_count <= drag_min_ticks) {
        NewHue = click_angle_deg - hue_angle_deg;
        if (NewHue < 0.0) {NewHue += 360.0;}
///         360 - (hue_angle_deg - click_angle_deg)
//        hue_angle_deg -= click_angle_deg;
//        if (hue_angle_deg < 0.0) {hue_angle_deg += 360.0;}
//        h = 360.0 - hue_angle_deg;
        hue_changed = true;
      }
      drag_tick_count = 0;
    }
    if (drag_tick_count > drag_min_ticks) {
      // find the relative amount that the rotation has changed by
      ox = mx - midx;
      oy = my - midy;
      double drag_angle_deg = atan2(oy,ox)*rad_to_deg;
      if (drag_angle_deg < 0.0) {drag_angle_deg += 360.0;}
      double delta_angle_deg = drag_angle_deg - click_angle_deg;
      if (delta_angle_deg < 0.0) {delta_angle_deg += 360.0;}
      NewHue = 360.0 - (old_hue_angle_deg + delta_angle_deg);
      if (NewHue < 0.0) {NewHue += 360.0;}
      hue_changed = true;
    }
  }
  if (sv_drag) {
    if (mbtn & 1) {
      // still held down
      ++drag_tick_count;
    } else {
      // just released
      sv_drag = false;
      if (drag_tick_count < drag_min_ticks) {
        sv_changed = true;
      }
      drag_tick_count = 0;
    }
    if (drag_tick_count >= drag_min_ticks) {
      bool outside_triangle = FindClosestSVfromMousePos(mx , my , NewSaturation , NewValue);
//      Log() << "    FindClosestSVfromMousePos(" << mx << " , " << my << " ,,) returned NewSaturation = ";
//      Log() << NewSaturation << " , NewValue = " << NewValue << "( " << (outside_triangle?"Outside":"Inside");
//      Log() << " triangle )" << endl;
      if (!outside_triangle) {
        FindSVfromMousePos(mx , my , NewSaturation , NewValue);
      }
      /// No out of bounds values should be returned from the FindClosestSVfromMousePos function, but
      ///   log any that happen just in case
      if (((NewSaturation < 0.0) || (NewSaturation > 1.0)) || ((NewValue < 0.0) || (NewValue > 1.0))) {
        NewSaturation = s;
        NewValue = v;
      } else {
        sv_changed = true;
      }
    }
  }
  hue_click = false;
  sv_click = false;
  if ((mbtn & 1) && !(old_mbtn & 1)) {/// LMB click
    /// Check for new click within hue wheel
    ox = mx - midx;
    oy = my - midy;
    double dsq = ox*ox + oy*oy;
    if ((dsq >= irad*irad) && (dsq <= orad*orad)) {
      hue_click = true;
      click_angle_deg = atan2(oy,ox)*rad_to_deg;
      if (click_angle_deg < 0.0) {click_angle_deg += 360.0;}
      old_hue_angle_deg = hue_angle_deg;
    }
    
    FindSVfromMousePos(mx , my , NewSaturation , NewValue);
    /// Handle out of bounds values for Value and Saturation here...(They indicate a position outside the triangle)
    if (((NewSaturation < 0.0) || (NewSaturation > 1.0)) || ((NewValue < 0.0) || (NewValue > 1.0))) {
      NewSaturation = s;
      NewValue = v;
    } else {
      sv_click = true;
    }
  }
  
  if (hue_changed || sv_changed) {
    SetHSV(NewHue , NewSaturation , NewValue);
    NewHue = h;
    NewSaturation = s;
    NewValue = v;
  }
  old_mbtn = mbtn;
  return (hue_changed || sv_changed);
}



bool HueWheelRR::FindClosestSVfromMousePos(const double mx , const double my , double& saturation , double& value) {
  const double msx = mx;
  const double msy = my;
  const double angle[3] = {angle_HB_rad , angle_BW_rad , angle_WH_rad};
  const double vx[3] = {vH.x , vB.x , vW.x};
  const double vy[3] = {vH.y , vB.y , vW.y};
  const double dist[3] = {distHB , distBW , distWH};
  const double sat[3] = {1.0 , 0.0 , 0.0};
  const double val[3] = {1.0 , 0.0 , 1.0};
//  struct V3D_f* const vecp[3] = {&vecHB , &vecBW , &vecWH};
  bool inside_triangle = true;
  double angle_rad = 0.0;
  double vecVPx = 0.0;
  double vecVPy = 0.0;
  double distVP = 0.0;
  double SP = 0.0;
  double ratio = 0.0;
  int sector = 0;
  for (int i = 0 ; i < 3 ; ++i) {
    vecVPx = msx - vx[i];
    vecVPy = msy - vy[i];
    /// Find angle from the current vector to the vector from the current vertice to the mouse position
    if ((vecVPx == 0.0) && (vecVPy == 0.0)) {// Angle is undefined since they are the same point but 0.0 will work
      angle_rad = 0.0;
    } else {
      /// Angle from V to P
      angle_rad = atan2(vecVPy , vecVPx);
      if (angle_rad < 0.0) {angle_rad += 2.0*M_PI;}
      /// Angle between the vector from V to P and the vector from V[i] to V[i+1]
      angle_rad -= angle[i];
      if (angle_rad < -M_PI) {angle_rad += 2.0*M_PI;}
      if (angle_rad > M_PI) {angle_rad -= 2.0*M_PI;}
    }
    /// Find scalar projection of the vector to the mouse onto the current vector (how far along the edge it is)
    distVP = sqrt(vecVPx*vecVPx + vecVPy*vecVPy);
    SP = distVP*cos(angle_rad);
    /// If the angle is greater than 0 (now from -PI to PI) it points outside or along the edge of the triangle
    if (angle_rad <= 0.0) {
      inside_triangle = false;
      if (SP <= 0.0) {// Closest is the current vertice
//          mx = vx[i];
//          my = vy[i];
        //sector += 0;
        value = val[i];
        if (i != 1) {// saturation is undefined at B
          saturation = sat[i];
        }
      } else if (SP < dist[i]) {// Closest is along the edge
        ratio = SP/dist[i];
//          mx = vx[i] + ratio*vecp[i]->x;
//          my = vy[i] + ratio*vecp[i]->y;
        sector += 1;
        switch (i) {
          case 0 : // Edge HB
            value = 1.0 - ratio;
            saturation = 1.0;
            break;
          case 1 : // Edge BW
            value = ratio;
            saturation = 0.0;
            break;
          case 2 : // Edge WH
            value = 1.0;
            saturation = ratio;
            break;
          default : {} // Error
        }
      } else {// Closest is the next vertice
//          mx = vx[(i+1)%3];
//          my = vy[(i+1)%3];
        sector += 2;
        value = val[(i+1)%3];
        if ((i+1)%3 != 1) {// saturation is undefined at B
          saturation = sat[(i+1)%3];
        }
      }
      break;
    }
    sector += 3;
  }
/*
  Log() << "  FindClosestSVfromMousePos (sector " << sector << " : ";
  Log() << (((sector + 2)%3)?"edge":"corner") << " ) is sending ( " << mx << " , my = " << my << " , saturation = ";
  Log() << saturation << " , value = " << value << " ) to FindSVfromMousePos :" << endl;
*/
  return !inside_triangle;
//  if (inside_triangle) {
//    FindSVfromMousePos(mx , my , saturation , value);
//  }
}



bool HueWheelRR::FindSVfromMousePos(double mx , double my , double& saturation , double& value) {
  bool sv_different = false;
  double BPx = mx - double(vB.x);
  double BPy = my - double(vB.y);
  double WPx = mx - double(vW.x);
  double WPy = my - double(vW.y);
  
  double distBP = sqrt(BPy*BPy + BPx*BPx);
  double distWP = sqrt(WPy*WPy + WPx*WPx);
  
  if (distBP == 0.0) {
    /// P is at B
    value = 0.0;
    return ((value != v) || (saturation != s));
  }
  if (distWP == 0.0) {
    /// P is at W
    value = 1.0;
    saturation = 0.0;
    return ((value != v) || (saturation != s));
  }
  
  double angle_BP_rad = atan2(BPy,BPx);
  if (angle_BP_rad < 0.0) {angle_BP_rad += 2.0*M_PI;}
  double angle_WP_rad = atan2(WPy,WPx);
  if (angle_WP_rad < 0.0) {angle_WP_rad += 2.0*M_PI;}
  
  double angle_BM_to_BP = angle_BP_rad - angle_BM_rad;
  double angle_WH_to_WP = angle_WP_rad - angle_WH_rad;
  // Scalar projections of WP onto WH and BP onto BM
  double SPBPBM = distBP*cos(angle_BM_to_BP);
  double SPWPWH = distWP*cos(angle_WH_to_WP);
  value = SPBPBM/distBM;
///  saturation = SPWPWH*value/distWH;
  if (value) {
    saturation = (SPWPWH + (value - 1.0)*distMW)/(value*distWH);
  }
  if ((value != v) || (saturation != s)) {
    sv_different = true;
  }
  return sv_different;
}



void HueWheelRR::VisuallyFindClosestSVfromMousePos(BITMAP* dest , double mx , double my , double& saturation , double& value) {
  int hcolor = vH.c;
  int bcolor = vB.c;
  int wcolor = vW.c;
  int color[3] = {hcolor , bcolor , wcolor};
  const double rad_to_deg = 180.0/M_PI;
  const char* anglename[3] = {"Angle from HB to HP" , "Angle from BW to BP" , "Angle from WH to WP"};
  const double msx = mx;
  const double msy = my;
  const double angle[3] = {angle_HB_rad , angle_BW_rad , angle_WH_rad};
  const double vx[3] = {vH.x , vB.x , vW.x};
  const double vy[3] = {vH.y , vB.y , vW.y};
  const double dist[3] = {distHB , distBW , distWH};
  const double sat[3] = {1.0 , 0.0 , 0.0};
  const double val[3] = {1.0 , 0.0 , 1.0};
  struct V3D_f* const vecp[3] = {&vecHB , &vecBW , &vecWH};
  bool inside_triangle = true;
  double angle_rad = 0.0;
  double vecVPx = 0.0;
  double vecVPy = 0.0;
  double distVP = 0.0;
  double SP = 0.0;
  double ratio = 0.0;
  int sector = 0;
//  for (int i = 0 ; i < 3 ; ++i) {
//    circle(dest , tlx + int(vx[i]) , tly + int(vy[i]) , 10 , color[i]);
//    DrawArrow(dest , tlx + int(vx[i]) , tly + int(vy[i]) , tlx + int(vx[i] + vecp[i]->x), tly + int(vy[i] + vecp[i]->y) , color[i]);
//  }
  for (int i = 0 ; i < 3 ; ++i) {
    vecVPx = msx - vx[i];
    vecVPy = msy - vy[i];
    DrawArrow(dest , tlx + vx[i] , tly + vy[i] , tlx + msx , tly + msy , color[i]);
    /// Find angle from the current vector to the vector from the current vertice to the mouse position
    if ((vecVPx == 0.0) && (vecVPy == 0.0)) {// Angle is undefined since they are the same point, leave mx&my unaltered
      angle_rad = 0.0;
//      break;
    } else {
      /// Angle from V to P
      angle_rad = atan2(vecVPy , vecVPx);
      if (angle_rad < 0.0) {angle_rad += 2.0*M_PI;}
      /// Angle between the vector from V to P and the vector from V[i] to V[i+1]
      angle_rad -= angle[i];
      if (angle_rad < -M_PI) {angle_rad += 2.0*M_PI;}
      if (angle_rad > M_PI) {angle_rad -= 2.0*M_PI;}
    }
    /// Find scalar projection of the vector to the mouse onto the current vector (how far along the edge it is)
    distVP = sqrt(vecVPx*vecVPx + vecVPy*vecVPy);
    SP = distVP*cos(angle_rad);
    DrawArrow(dest , tlx + vx[i] , tly + vy[i] , tlx + vx[i] + (SP/dist[i])*vecp[i]->x , tly + vy[i] + (SP/dist[i])*vecp[i]->y , color[i]);
    textprintf_ex(dest , font , 12 , dest->h - 48 + i*12 , makecol(255,255,255) , -1 ,
                  "%s = %f" , anglename[i], angle_rad*rad_to_deg);
    /// If the angle is greater than 0 (now from -PI to PI) it points outside or along the edge of the triangle
    if (angle_rad <= 0.0) {
      inside_triangle = false;
      if (SP <= 0.0) {// Closest is the current vertice
        mx = vx[i];
        my = vy[i];
        //sector += 0;
        value = val[i];
        if (i != 1) {// saturation is undefined at B
          saturation = sat[i];
        }
      } else if (SP < dist[i]) {// Closest is along the edge
        ratio = SP/dist[i];
        mx = vx[i] + ratio*vecp[i]->x;
        my = vy[i] + ratio*vecp[i]->y;
        sector += 1;
        switch (i) {
          case 0 : // Edge HB
            value = 1.0 - ratio;
            saturation = 1.0;
            break;
          case 1 : // Edge BW
            value = ratio;
            saturation = 0.0;
            break;
          case 2 : // Edge WH
            value = 1.0;
            saturation = ratio;
            break;
          default : {} // Error
        }
      } else {// Closest is the next vertice
        mx = vx[(i+1)%3];
        my = vy[(i+1)%3];
        sector += 2;
        value = val[(i+1)%3];
        if ((i+1)%3 != 1) {// saturation is undefined at B
          saturation = sat[(i+1)%3];
        }
      }
      break;
    }
    sector += 3;
  }
  for (int i = 0 ; i < 3 ; ++i) {
    circle(dest , tlx + int(mx) , tly + int(my) , 10 + i , color[i]);
  }
  /// Now that the mouse position has been adjusted to the closest point on the triangle, pass it 
  ///   to FindSVfromMousePos
/*
  Log() << "  FindClosestSVfromMousePos (sector " << sector << " : ";
  Log() << (((sector + 2)%3)?"edge":"corner") << " ) is sending ( " << mx << " , my = " << my << " , saturation = ";
  Log() << saturation << " , value = " << value << " ) to FindSVfromMousePos :" << endl;
*/
  textprintf_ex(dest , font , 12 , dest->h - 60 , makecol(255,255,255) , -1 ,
                "%s" , (inside_triangle?"Inside Triangle":"Outside Triangle"));
  if (inside_triangle) {
    FindSVfromMousePos(int(mx) , int(my) , saturation , value);
//    VisuallyFindSVfromMousePos(dest , int(mx) , int(my) , saturation , value);
  } else {
//  void VisualizeIndicatorPos(BITMAP* dest , double sat , double val , int color);
//    VisualizeIndicatorPos(dest , saturation , value , makecol(255,127,0));
  }
}




bool HueWheelRR::VisuallyFindSVfromMousePos(BITMAP* dest , int mx , int my , double& saturation , double& value) {
  bool sv_different = false;
  int tomousecolor = makecol(255,255,255);
//  int distcolor = makecol(0,255,0);
  int scalarcolor = makecol(0,255,255);
//  int tospcolor = makecol(0,0,255);
  
  double BPx = double(mx) - double(vB.x);
  double BPy = double(my) - double(vB.y);
  double WPx = double(mx) - double(vW.x);
  double WPy = double(my) - double(vW.y);
  
  double distBP = sqrt(BPy*BPy + BPx*BPx);
  double distWP = sqrt(WPy*WPy + WPx*WPx);
  
//  DrawArrow(dest , int(tlx + ) , int(tly + ) , int(tlx + ) , int(tly + ) , );
  DrawArrow(dest , int(tlx + vB.x) , int(tly + vB.y) , int(tlx + mx) , int(tly + my) , tomousecolor);
  DrawArrow(dest , int(tlx + vW.x) , int(tly + vW.y) , int(tlx + mx) , int(tly + my) , tomousecolor);
  
  if (distBP == 0.0) {
    /// P is at B
    value = 0.0;
    return ((value != v) || (saturation != s));
  }
  if (distWP == 0.0) {
    /// P is at W
    value = 1.0;
    saturation = 0.0;
    return ((value != v) || (saturation != s));
  }
  
  
  
  double angle_BP_rad = atan2(BPy,BPx);
  if (angle_BP_rad < 0.0) {angle_BP_rad += 2.0*M_PI;}
  double angle_WP_rad = atan2(WPy,WPx);
  if (angle_WP_rad < 0.0) {angle_WP_rad += 2.0*M_PI;}
  
  double angle_BM_to_BP = angle_BP_rad - angle_BM_rad;
  double angle_WH_to_WP = angle_WP_rad - angle_WH_rad;
  // Scalar projections of WP onto WH and BP onto BM
  double SPBPBM = distBP*cos(angle_BM_to_BP);
  double SPWPWH = distWP*cos(angle_WH_to_WP);
  DrawArrow(dest , int(tlx + vB.x) , int(tly + vB.y) , int(tlx + vB.x + SPBPBM*cos(angle_BM_rad)) , int(tly + vB.y + SPBPBM*sin(angle_BM_rad)) , scalarcolor);
  DrawArrow(dest , int(tlx + vW.x) , int(tly + vW.y) , int(tlx + vW.x + SPWPWH*cos(angle_WH_rad)) , int(tly + vW.y + SPWPWH*sin(angle_WH_rad)) , scalarcolor);
  value = SPBPBM/distBM;
  if (value) {
    saturation = (SPWPWH + (value - 1.0)*distMW)/(value*distWH);
  }
  if ((value != v) || (saturation != s)) {
    sv_different = true;
  }
  return sv_different;
}



void HueWheelRR::VisualizeIndicatorPos(BITMAP* dest , int color) {
//  int color = makecol(0,255,255);
  double BPonBWx = v*vecBW.x;
  double BPonBWy = v*vecBW.y;
  double PonBWx = vB.x + BPonBWx;
  double PonBWy = vB.y + BPonBWy;
  double PonBWPx = v*s*double(vecWH.x);
  double PonBWPy = v*s*double(vecWH.y);
  double Px = PonBWx + PonBWPx;
  double Py = PonBWy + PonBWPy;
  DrawArrow(dest , tlx + int(vB.x) , tly + int(vB.y) , tlx + int(PonBWx) , tly + int(PonBWy) , color);
  DrawArrow(dest , tlx + int(PonBWx) , tly + int(PonBWy) , tlx + int(Px) , tly + int(Py) , color);
//    svix = int(vB.x + v*(vecBW.x + s*vecWH.x));
//    sviy = int(vB.y + v*(vecBW.y + s*vecWH.y));
}



void HueWheelRR::VisualizeIndicatorPos(BITMAP* dest , double sat , double val , int color) {
//  int color = makecol(0,255,255);
  double BPonBWx = val*vecBW.x;
  double BPonBWy = val*vecBW.y;
  double PonBWx = vB.x + BPonBWx;
  double PonBWy = vB.y + BPonBWy;
  double PonBWPx = val*sat*double(vecWH.x);
  double PonBWPy = val*sat*double(vecWH.y);
  double Px = PonBWx + PonBWPx;
  double Py = PonBWy + PonBWPy;
  DrawArrow(dest , tlx + int(vB.x) , tly + int(vB.y) , tlx + int(PonBWx) , tly + int(PonBWy) , color);
  DrawArrow(dest , tlx + int(PonBWx) , tly + int(PonBWy) , tlx + int(Px) , tly + int(Py) , color);
//    svix = int(vB.x + v*(vecBW.x + s*vecWH.x));
//    sviy = int(vB.y + v*(vecBW.y + s*vecWH.y));
}



void HueWheelRR::SetHSV(double hue , double sat , double val) {
//  Log() << "SetHSV called with the values H = " << hue << " , S = " << sat << " , V = " << val << endl;
  hue_changed = false;
  sv_changed = false;
  if (h != hue) {
    hue_changed = true;
    h = hue;
    hue_angle_deg = 360.0 - h;
    DrawHueWheelBuffer();
    DrawHueWheel();
    NewHue = h;
  }
  if ((s != sat) || (v != val)) {
    sv_changed = true;
    s = sat;
    v = val;
    ResetIndicatorPos();
    NewSaturation = s;
    NewValue = v;
  }
  
  if (hue_changed || sv_changed) {
    hsv_to_rgb(h,s,v,&r,&g,&b);
    rgbcolor = makecol(r,g,b);
    ResetVerticeColors();
    DrawSatValTriangle();
    UpdateBuffer();
  }
//  Log() << "  At the end of SetHSV , h = " << h << " , s = " << s << " , v = " << v << endl;
}



void HueWheelRR::SetRGB(int red , int green , int blue) {
  r = red;
  g = green;
  b = blue;
  rgbcolor = makecol(r,g,b);
  rgb_to_hsv(r,g,b,&h,&s,&v);
  NewHue = h;
  NewSaturation = s;
  NewValue = v;
  hue_angle_deg = 360.0 - h;
  DrawHueWheelBuffer();
  DrawHueWheel();
  ResetIndicatorPos();
  ResetVerticeColors();
  DrawSatValTriangle();
  hue_changed = true;
  sv_changed = true;
  UpdateBuffer();
}



void HueWheelRR::SetDragDelay(int ticks) {
  drag_min_ticks = ticks;
}



void HueWheelRR::StoreHSV(double* hue , double* sat , double* val) {
  if (hue) {*hue = h;}
  if (sat) {*sat = s;}
  if (val) {*val = v;}
}



void HueWheelRR::StoreRGB(int* red , int* green , int* blue) {
  if (red) {*red = r;}
  if (green) {*green = g;}
  if (blue) {*blue = b;}
}



int HueWheelRR::GetWidth() {
  if (!buffer) {return 0;}
  return buffer->w;
}



int HueWheelRR::GetHeight() {
  if (!buffer) {return 0;}
  return buffer->w;
}






