#ifndef TOKEN_H
#define TOKEN_H

#include <string>
using std::string;

#include "stringutils.h"
#include "charutils.h"
#include "exception.h"

class Token {
	public:
		string Value;
	
		typedef enum Type {
			WHITESPACE,
			CPP,
			IDENT,
			NUMCONST,
			CHARCONST,
			STRCONST,
			OPERATOR,
			LINECOMMENT,
			BLOCKCOMMENT,
			
			EMPTY
		};
		
		Token(): Value() {}
		Token(const Token& rhs): Value(rhs.Value) {}
		Token& operator=(const Token& rhs) { Value = rhs.Value; return *this; }

		Token& Clear() { Value = ""; return *this; }
		
		Type GetType() const {
			if (Value.length() == 0)
				return EMPTY;
			if (CharIsWhitespace(Value[0]))
				return WHITESPACE;
			if (CharIsAlpha(Value[0]))
				return IDENT;
			if (CharIsNumeric(Value[0]) || (Value[0] == '-'))
				return NUMCONST;
			if (Value[0] == '#')
				return CPP;
			if (Value[0] == '\'')
				return CHARCONST;
			if (Value[0] == '"')
				return STRCONST;
			if (StringIsOperator(Value))
				return OPERATOR;
			if (StartsWith(Value, "//"))
				return LINECOMMENT;
			if (StartsWith(Value, "/*"))
				return BLOCKCOMMENT;
			throw Exception("Error: The current value (" + Value + ") does not represent a valid C++ token.");
		}
		
		bool WantsChar(char c)
		{
			switch (GetType())
			{
				case EMPTY: return true; // no token type yet, so always accept.
				case WHITESPACE: return CharIsWhitespace(c); // whitespace always accepts more whitespace
				case IDENT: return CharIsAlpha(c) || CharIsNumeric(c);
					
				case NUMCONST:
					if (EndsWith(Value, "f") || EndsWith(Value, "l") || EndsWith(Value, "u"))
						return false;
					if (CharIsNumeric(c))
						return true;
					switch (c) {
						case 'x':
						case 'X':
							return (Value == "0") || (Value == "-0"); // hex indicator
						case '.':
						case 'l':
						case 'u':
							return !Contains(Value, '.'); // floats and doubles may contain 1 decimal point
						case 'f':
							return true;
					}
					return false;
				
				case CPP:
					return (!EndsWith(Value, "\n") || EndsWith(Value, "\\\n"));
				
				case CHARCONST: 
					return ((Value == "'") || !EndsWith(Value, "'") || EndsWith(Value, "\\'"));
				
				case STRCONST: 
					return ((Value == "\"") ||!EndsWith(Value, "\"") || EndsWith(Value, "\\\""));
				
				case LINECOMMENT: return !EndsWith(Value, "\n");
				case BLOCKCOMMENT: return !EndsWith(Value, "*/");
				case OPERATOR: 
					return StringIsOperator(Value + c) || StringStartsComment(Value + c);
			}
			return false;
		}
		
		string GetCondensedValue() const {
			switch (GetType()) {
				case WHITESPACE: return " ";
				case CHARCONST:
					if (Value.size() == 3) {
						char buf[4];
						sprintf(buf, "%i", Value[1]);
						return string(buf);
					}
					if (Value == "'\''")
						return string("39");
					return Value;
				default: return Value;
			}
		}
};


#endif
