#include "parser.h"
#include "globals.h"
#include <iostream>
#include <map>
using namespace std;

void Parser::Tokenize(string str) {
	Token CurrentToken;
	
	for (unsigned i = 0; i < str.length(); ++i)	{
		if (!CurrentToken.WantsChar(str[i])) {
			tokens.push_back(CurrentToken);
			CurrentToken.Clear();
		}
		CurrentToken.Value += str[i];
	}
	if (CurrentToken.GetType() != Token::EMPTY)
		tokens.push_back(CurrentToken);
}

bool Parser::IsUnneccessaryToken(unsigned pos) const {
	Token::Type t = tokens[pos].GetType();
	
	switch (t) {
		case Token::WHITESPACE: break; // investigate further
		case Token::LINECOMMENT: return true; // strip comments
		case Token::BLOCKCOMMENT: return true; // strip comments
		case Token::EMPTY: return true; // strip empty tokens
		default: return false; // keep the rest intact
	}
	
	if (tokens[pos].GetType() != Token::WHITESPACE)
		return false;
	
	// first token never needs to be whitespace
	if (pos == 0)
		return true;
	
	// last token never needs to be whitespace
	if (pos == tokens.size()-1)
		return true;
	
	Token::Type leftType = tokens[pos-1].GetType();
	Token::Type rightType = tokens[pos+1].GetType();
	
	// no whitespace needed between operators
	if ((leftType == Token::OPERATOR) || (rightType == Token::OPERATOR))
		return true;
	
	// condense consecutive whitespace
	if ((leftType == Token::WHITESPACE) || (rightType == Token::WHITESPACE))
		return true;
	
	// no whitespace directly after a pp instruction
	if (leftType == Token::CPP)
		return true;
	
	return false;
}

void Parser::RemoveUnnecessaryTokens() {
	unsigned removals;
	
	do {
		removals = 0;
		for (unsigned i = 0; i < tokens.size(); ++i) {
			while ((i < tokens.size()) && IsUnneccessaryToken(i)) {
				tokens.erase(tokens.begin() + i);
				++removals;
			}
		}
	} while (removals);
}

template <class T> bool SecondMore(const T& a, const T& b) {
	return a.second > b.second;
}

void Parser::CollectIdentifiers() {
	map<string, unsigned> buf;
	for (unsigned i = 0; i < tokens.size(); ++i) {
		if (tokens[i].GetType() == Token::IDENT) {
			buf[tokens[i].Value] += 1;
		}
	}
	
	vector<pair<string, unsigned> > icount;
	
	for (map<string, unsigned>::const_iterator i = buf.begin(); i != buf.end(); ++i)
		icount.push_back(*i);
	
	sort(icount.begin(), icount.end(), SecondMore<pair<string, unsigned> >);
	
	for (vector<pair<string, unsigned> >::const_iterator i = icount.begin(); i != icount.end(); ++i)
		identifiers.push_back(i->first);
}

static char NthIdentifierChar(unsigned d) {
	if (d == 0)
		return '_';
	if (d < 27)
		return 'a' + d - 1;
	if (d < 53)
		return 'A' + d - 27;
	if (d < 63)
		return '0' + d - 53;
	throw Exception("Invalid digit.");
}

static string GetReplacedIdentifierName(unsigned index) {
	string result;
	
	unsigned firstDigit = index % 53;
	unsigned rest = index / 53;
	
	result += NthIdentifierChar(firstDigit);
	
	while (rest > 0) {
		result += NthIdentifierChar(rest % 63);
		rest /= 63;
	}
	return result;
}

string Parser::GetNextUsableIdentifier() {
	string result = GetReplacedIdentifierName(NextIdentIndex++);
	
	// while the result is either an identifier or a C++ keyword...
	while ((find(identifiers.begin(), identifiers.end(), result) != identifiers.end()) || StringIsInArray(result, CppKeywords))
		result = GetReplacedIdentifierName(NextIdentIndex++);
	
	return result;
}

void Parser::ReplaceUserIdentifiers() {
	map<string, string> reptbl;
	
	CollectIdentifiers();
	
	for (unsigned i = 0; i < identifiers.size(); ++i)
		if (StartsWith(identifiers[i], "$"))
			reptbl[identifiers[i]] = GetNextUsableIdentifier();
		
	for (unsigned i = 0; i < tokens.size(); ++i) {
		if ((tokens[i].GetType() == Token::IDENT) && StartsWith(tokens[i].Value, "$"))
			tokens[i].Value = reptbl[tokens[i].Value];
	}
}

void Parser::Read(string filename) {
	string data = ReadFile(filename.c_str());
	Tokenize(data);
	RemoveUnnecessaryTokens();
	ReplaceUserIdentifiers();
}

void Parser::WriteCondensed() const {
	unsigned LineLen = 0;
	for (unsigned i = 0; i < tokens.size(); ++i) {
		string str = tokens[i].GetCondensedValue();
		if (LineLen + str.length() > 80) {
			cout << "\n";
			LineLen = 0;
		}
		cout << str;
		if (EndsWith(str, "\n"))
			LineLen = 0;
		else
			LineLen += str.length();
	}
}
