import sys
import os
import os.path
import wave

import smpconvert

####################################

def float2bytes(f):
	if (isinstance(f, str)):
		try:
			f = float(f)
		except ValueError:
			raise ProcessError("Invalid number format.")
	
	first = int(f)
	if (first < 0) or (first > 31):
		raise ProcessError("Floating point numbers should be between 0 and 32.")
	second = int((f - first) * 1024.0)
	
	first_char  = chr((first << 2) + ((second & 0x300) >> 8))
	second_char = chr(second & 0xFF)
	
	return first_char + second_char

####################################

class Wave:
	def __init__(self, letter):
		self.letter = letter
		
	def __call__(self, opt, out):
		out.write(self.letter)
		if (len(opt) >= 1):
			vol_mod = opt[0]
		else:
			vol_mod = 1.0
		if (len(opt) >= 2):
			freq_mod = opt[1]
		else:
			freq_mod = 1.0
			
		out.write(float2bytes(vol_mod))
		out.write(float2bytes(freq_mod))
		
####################################

def envelope(opt, out):
	if (len(opt) == 0):
		raise ProcessError("ENV requires parameters.")
	if (len(opt) % 2):
		raise ProcessError("Odd number of parameters for ENV.")
	
	out.write('E')
	for op in opt:
		out.write(float2bytes(op))
	out.write(chr(255) + chr(255))

####################################

def digital(opt, out):	
	global SAMPLE_PATH
	
	if (len(opt) != 2):
		print opt
		raise ProcessError("DIGI requires 2 parameters (filename and base note).")
	
	out.write('D')
	filename, note = opt
	
	# base note
	note = note.replace("-", "")
	note_octave = note[-1]
	note_name = note[:-1]	
	note_num = 12 + int(note_octave) * 12 + NOTES[note_name]	
	out.write(chr(note_num))
	
	
	try:
		wave_in = wave.open(os.path.join(SAMPLE_PATH, filename), "rb")
	except wave.Error:
		raise ProcessError("Incorrect wave data in file %s." % filename)
	except IOError:
		raise ProcessError("Unable to read wave file %s." % filename)
	
	try:
		smpconvert.convert_sample(wave_in, out)
	except smpconvert.Error, e:
		raise ProcessError("Error while converting %s: %s" % (filename, str(e)))

####################################

def volume(opt, out):
	if (len(opt) != 1):
		raise ProcessError("VOL requires one parameter.")
	out.write('V')
	out.write(float2bytes(opt[0]))

####################################

SOUND_BLOCKS = {
	'sine':		Wave('S'),
	'pulse':	Wave('P'),
	'triangle':	Wave('T'),
	'noise':	Wave('N'),
	'env':		envelope,
	'vol':		volume,
	'digi':		digital
}

####################################

NOTES = {
	'c': 0,
	'c#': 1,
	'd': 2,
	'd#': 3,
	'e': 4,
	'f': 5,
	'f#': 6,
	'g': 7,
	'g#': 8,
	'a': 9,
	'a#': 10,
	'b': 11,
	'h': 11
}

def note2number(text):
	text = text.replace('-', '')
	try:
		note_name = text[:-1]
		note_octave = int(text[-1])
		
		note_number = 12 + note_octave * 12 + NOTES[note_name]
		return note_number
		
	except ValueError:
		raise ProcessError("Wrong value for octave: %s." % text[-1])
	except KeyError:
		raise ProcessError("Unknown note name: %s." % text[:-1])
		
####################################

num_to_sample = ["__prev"]
sample_to_num = {"__prev": 0}
num_to_set = [None]
set_to_num = {}

####################################

class ProcessError(StandardError):
	def __init__(self, msg):
		StandardError.__init__(self, msg)

####################################

def sblock_split(line):
	parts = line[0].split(None, 1)
	if (len(parts) == 1):
		return (parts[0], [])
	else:
		opt = [x.strip() for x in parts[1].split(",")]
		return (parts[0], opt)

####################################

def set_split(line):
	global sample_to_num
	
	channels = [x.strip() for x in line[0].split("\t")]
	if (len(channels) > 3):
		raise ProcessError("More than three tabs detected in a line = more than three channels.")
	while (len(channels) < 3):
		channels += ["-"]
	
	ret = []
	for chan in channels:
		if (chan == "-") or (chan == ""):
			ret += [(None, 0, None, [])]
			continue
		
		parts = chan.split(None, 3)
		if (parts[0][0] == '@'):
			note = 254
			sam_num = 0
			parts = ['', '', parts[0], " ".join(parts[1:])]
		else:
			if (len(parts) < 2):
				parts += ['__prev']
			note = note2number(parts[0])
			
			try:
				sam_num = sample_to_num[parts[1]]
			except KeyError:
				raise ProcessError("Unknown sample: '%s'." % parts[1])
			
		if (len(parts) >= 4):
			opt = [x.strip() for x in parts[3].split(",")]
			ret += [(note, sam_num, parts[2][1:], opt)]
		elif (len(parts) >= 3):
			ret += [(note, sam_num, parts[2][1:], [])]
		else:
			ret += [(note, sam_num, None, [])]
		
	return ret

####################################

def command_split(line):
	parts = line[0].split(None, 2)
	if (len(parts) == 1):
		return (parts[0], None, [])
	elif (len(parts) == 2):
		return (parts[0], parts[1], [])
	else:
		opt = [x.strip() for x in parts[2].split(",")]
		return (parts[0], parts[1], opt)

####################################

def process_sample(name, lines, out):
	global current_line, current_element
	global num_to_sample, sample_to_num
	
	print "Converting sample '%s'..." % name
	current_element = "SAMPLE " + name
	sample_num = len(num_to_sample)
	
	num_to_sample += [name]
	sample_to_num[name] = sample_num
	
	out.write(chr(sample_num))
	
	for line in lines:
		current_line = line
		block, opt = sblock_split(line)
		try:
			func = SOUND_BLOCKS[block]
		except KeyError:
			raise ProcessError("Unknown soundblock in a sample: '%s'." % block)
		
		func(opt, out)
		
	out.write('F')

####################################

def process_set(name, tempo, lines, out):
	global current_line, current_element
	global num_to_set, set_to_num
	
	print "Converting set '%s'..." % name
	
	current_element = "SET " + name
	set_num = len(num_to_set)
	
	num_to_set += [name]
	set_to_num[name] = set_num
	
	out.write(chr(set_num))
	try:
		tempo = int(tempo)
	except ValueError:
		raise ProcessError("Number expected for tempo, got: '%s'.", tempo)
	out.write(chr(tempo / 256))
	out.write(chr(tempo % 256))
	
	for line in lines:
		current_line = line
		channels = set_split(line)
		
		for chan in channels:
			note, sample, effect, opt = chan
			if (note is None):
				out.write(chr(255))
				continue
			
			out.write(chr(note))
			out.write(chr(sample))
			if (effect is None):
				out.write('F')
			else:
				try:
					func = SOUND_BLOCKS[effect]
					func(opt, out)
				except KeyError:
					raise ProcessError("Unknown effect: '%s'." % effect)
	
	out.write(chr(253))

####################################

def process_play(line, out):
	global current_element, current_line
	global set_to_num, num_to_set
	
	current_element = "PLAY line"
	current_line = line
	
	print "Converting PLAY list..."
	
	sets = [x.strip() for x in line[0].split(None)[1:]]
	for set in sets:
		try:
			set_num = set_to_num[set]
		except KeyError:
			raise ProcessError("Unknown set name: '%s'." % set)
		out.write(chr(set_num))
		
	out.write(chr(0))

####################################

def purge_file(name, into):
	f = open(name, "rb")
	fragment = f.read(4096)
	while (fragment != ""):
		into.write(fragment)
		fragment = f.read(4096)
	
	f.close()

####################################

def process_file(f, outname):
	global current_line, current_element
	
	samfile = open(outname + ".sam", "wb")
	setfile = open(outname + ".set", "wb")
	playfile = open(outname + ".ply", "wb")
	
	raw_lines = f.readlines()
	lines = []
	for no in xrange(1, len(raw_lines) + 1):
		x = raw_lines[no-1]
		if (x != "\n" and x[0] != '#'):
			lines += [(x[:-1].lower().strip(), no)]
	
	ptr = 0
	
	current_element = '<root>'
	
	while (ptr < len(lines)):
		current_line = lines[ptr]
		
		command, name, opt = command_split(current_line)
		
		if (command not in ["sample", "set", "play"]):
			raise ProcessError("SAMPLE, SET or PLAY expected at this point, '%s' found." % command)
			
		if (command == 'sample'):
			if (name is None):
				raise ProcessError("Unnamed SAMPLE")
			sample_beg = ptr
			while (command != 'end'):
				ptr += 1
				if (ptr >= len(lines)):
					raise ProcessError("Unterminated SAMPLE - matching END not found.")
				command, _, _ = command_split(lines[ptr])
			process_sample(name, lines[sample_beg+1:ptr], samfile)
			ptr += 1
		
		if (command == 'set'):
			if (name is None) or (opt == []):
				raise ProcessError("A SET needs a name and a tempo")
			set_beg = ptr
			while (command != 'end'):
				ptr += 1
				if (ptr >= len(lines)):
					raise ProcessError("Unterminated SET - matching END not found.")
				command, _, _ = command_split(lines[ptr])
			process_set(name, opt[0], lines[set_beg+1:ptr], setfile)
			ptr += 1
		
		if (command == 'play'):
			process_play(current_line, playfile)
			ptr += 1
		
		current_element = "<root>"
			
	samfile.close()
	setfile.close()
	playfile.close()
	
	current_line = ('', 'n/a')
	
	outfile = open(outname, "wb")
	outfile.write('SMoD')
	purge_file(outname + ".sam", outfile)
	outfile.write(chr(0))
	purge_file(outname + ".set", outfile)
	outfile.write(chr(0))
	purge_file(outname + ".ply", outfile)
	outfile.close()
	
	os.unlink(outname + ".sam")
	os.unlink(outname + ".set")
	os.unlink(outname + ".ply")
	
####################################

if (len(sys.argv) < 3):
	print "Usage: modconvert.py <zrodlowy> <docelowy>"
inname = sys.argv[1]
outname = sys.argv[2]

SAMPLE_PATH = os.path.dirname(inname)

try:
	infile = open(inname, "r")
	process_file(infile, outname)
	
	infile.close()
	
except ProcessError, e:
	try:
		os.unlink(outname + ".sam")
		os.unlink(outname + ".set")
		os.unlink(outname + ".ply")
	except StandardError:
		pass
	
	print "\nError while processing:"
	print "Element - ", current_element
	print "Line    - ", current_line[1]
	print "Source  - ", current_line[0]
	print "Error   - ", e
	infile.close()
	
	sys.exit(1)

#except StandardError, e:
#	print "\nUnexpected error while processing:"
#	print "Element - ", current_element
#	print "Line    - ", current_line[1]
#	print "Source  - ", current_line[0]
#	print "Error   - ", e
###	infile.close()
#	sys.exit(1)

