#include <cmath>
#include <iostream>

#include "SoundBlocks.h"

/**********************************************/

const double TWO_PI = 2 * M_PI;

/**********************************************/
/**********************************************/

Wave::Wave(std::vector<double> &param)
{
	amplification = param[0];
	freqMultiplier = param[1];
}

/**********************************************/
/**********************************************/

double Sine::level(double freq, double time)
{
	freq *= freqMultiplier;
	return amplification * std::sin(time * TWO_PI * freq);
}

/**********************************************/
/**********************************************/

double Pulse::level(double freq, double time)
{
	double iFreq = 1.0 / (freq * freqMultiplier);
	return amplification * ((2.0 * fmod(time, iFreq) < iFreq) ? 1.0 : -1.0);
}

/**********************************************/
/**********************************************/

Noise::Noise(std::vector<double> &param)
	: Wave(param)
{
	state = 1;
	lastTime = 0.0;
}

/**********************************************/

double Noise::nextRandom()
{
	state = (state * 22717 + 7537) % 65537;
	return ((state / 65537.0) * 2.0 - 1.0);
}

/**********************************************/

double Noise::level(double freq, double time)
{
	if (time < lastTime)
		lastTime = 0.0;
	
	double period = 1.0 / (freq * freqMultiplier);
	/*if (time - lastTime > period)
	{
		state = 1;
		lastTime = time;
	}*/
		
	return amplification * nextRandom();
}

/**********************************************/
/**********************************************/

Envelope::Envelope()
{
	totalTime = 0.0;
}

/**********************************************/

Envelope::Envelope(std::vector<double> &params)
{
	totalTime = 0.0;
	
	for (int i = 0; i < params.size(); i += 2)
		addTransition(params[i], params[i+1]);
}

/**********************************************/

void Envelope::addTransition(double duration, double target)
{
	Envelope::Transition trans = {duration, target};
	transitions.push_back(trans);
	
	totalTime += duration;
}

/**********************************************/

bool Envelope::soundDead(double time)
{
	return (time >= totalTime);
}

/**********************************************/

double Envelope::actualLevel(double inputLevel, double time)
{
	if ((time >= totalTime) || (time < 0.0))
		return 0.0;
		
	double total = transitions[0].duration, prevTotal = 0.0;
	int currentTransition = 0;
	
	while (total < time)
	{
		prevTotal += transitions[currentTransition].duration;
		currentTransition++;
		total += transitions[currentTransition].duration;
	}
	double coeff = (time - prevTotal) / (total - prevTotal);
	double from = (currentTransition > 0) ? transitions[currentTransition - 1].target : 0.0;
	double to   = transitions[currentTransition].target;
		
	return (coeff * to + (1.0 - coeff) * from) * inputLevel;
}
