/**********************************************************
*                                                         *
* INSERT PROJECT NAME AND VERSION                         *
*                                                         *    
* Map generator functions                                 *
* Tuesday the 29th, March 2005                            *
*                                                         *
* Copyright (c) by Marco Radaelli                         *
*                                                         *
******                                              *******
*                                                         *
* This module generates a complete map for a level, using *
* the passed screen width and height.                     *
*                                                         *
* It initializes a predefined map and then randomly       *
* chooses walls to remove                                 *
*                                                         *
* Next, it checks if all the rooms are connected togheter *
* and if necessary  it removes additional walls           *
*                                                         *
* Then it distributes pills of both types on the map      *
* and provides the total number for each type             *
*                                                         *
* As last, there's a method to free the memory            *
* occupied by the two maps                                *
*                                                         *
**********************************************************/

#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "mt.h"
#include "map.h"

#include "tests/dump.h"



/* Possible map types */
#define MAP_SMALL 0x01
#define MAP_BIG   0x02

/* Each tile can be in three states
 * - corridor
 *     the players can walk on it, it appears on screen
 * - non-destroyable wall
 *     the players can't walk on it, it is represented by an image
 * - destroyable wall
 *     is a wall that the map generator can destroy to make a corridor
 *
 * These macros help set and clear the relative bits
*/
#define TILE_WALL             0x01
#define TILE_DESTROYABLE_WALL 0x02
#define TILE_REMOVE_WALL      0xFE

/* In each level there are two types of pills
 * - the normal pill
 *     the pacman has to collect them all to complete the level
 * - the special pill
 *     gives to the pacman the ability to eat the ghosts
 *
 * These macros help set and clear the relative bits
*/
#define TILE_NORMAL_PILL  0x80
#define TILE_SPECIAL_PILL 0x40
#define TILE_REMOVE_NORMAL_PILL  0x7F
#define TILE_REMOVE_SPECIAL_PILL 0xBF

/* The map generator analyses the current map in order to
 * ensure that each walkable tile can be reached
 * from any other.
 *
 * To do so it keeps track of the tiles it has
 * already examined
*/
#define TILE_MARK        0x04
#define TILE_REMOVE_MARK 0xFB

/**/
#define SET_BIT(var, bit)         var |= 0x01 << bit
#define BIT_ALREADY_SET(var, bit) (var & (0x01 << bit))
#define ALL_BITS_ARE_SET(var)     (var == 0x0F)

/* When expanding the small map in the big map
 * this macros avoids that the pills are
 * propagated on each walkable tile.
 *
 * Instead, they are placed only in the centre
 * of each corridor
*/
#define ZERO_PILLS_BITS 0x3F

#ifndef ABS
#define ABS(x) ((x >= 0) ? (x) : (-x))
#endif



static MAP *create_map(int screen_w, int screen_h, char type);
static void destroy_map(MAP *map);
static void initialize_small_map(MAP *map);
static void destroy_walls(MAP *small, float probability);
static unsigned char *search_non_marked_tiles(MAP *map);
static void check_corridors(unsigned char *tile, int step, int h_step, int v_step);
static void clear_marks(MAP *small);
static void validate_small_map(MAP *map);
static int special_pills_in_the_area(MAP *small, unsigned char *tile);
static void distribute_pills(MAP *small, float percentage,
                             unsigned int *normal_pills_num, unsigned int *special_pills_num);
static void compose_big_map(MAP *small, MAP *big);



/* Allocates two pieces of memory:
 *   - one for the MAP structure, which holds informations
 *     about the map size and a pointer to the first tile
 *   - one for the actual tiles, which number depends
 *     on the screen size
*/
static MAP *create_map(int screen_w, int screen_h, char type)
{
 MAP *temp;

 temp = (MAP *) malloc(sizeof(MAP));

 if(!temp)
   return NULL;

 switch(type)
   {
    case MAP_SMALL:
      {
       temp->w = (screen_w / 8 - (screen_w / 8 - 1) % 4) / 2 + 1;
       temp->h = ((screen_h / 8 - (screen_h / 8 - 1) % 4) + 1) / 2;
      } break;

    case MAP_BIG:
      {
       temp->w = screen_w / 8 - (screen_w / 8 - 1) % 4;
       temp->h = screen_h / 8 - (screen_h / 8 - 1) % 4;
      } break;

    default:
      return NULL;
   }

 temp->tile = (unsigned char *) malloc(sizeof(unsigned char) * temp->w * temp->h);

 if(!temp->tile)
   {
    free(temp);
    return NULL;
   }
 
 memset(temp->tile, 0, sizeof(unsigned char) * temp->w * temp->h);

 return temp;
}



/* Frees the memory occupied by the tiles
 * and the map structure
*/
static void destroy_map(MAP *map)
{
 free(map->tile);
 free(map);
}



/* Initializes the map to a predefined state,
 * in which there is a wall, a corridor,
 * a wall, a corridor and so forth, ending with
 * something like a chessboard
*/
static void initialize_small_map(MAP *small)
{
 unsigned short int i, j, w, h;
 unsigned char current;

 w = small->w;
 h = small->h;

 /* Puts the border walls on left and right sides */
 for(i = 0; i < w; i++)
   {
    *(small->tile + i) |= TILE_WALL;
    *(small->tile + w * (h - 1) + i) |= TILE_WALL;
   }

 /* and on top and bottom sides */
 for(i = 0; i < h; i++)
   {
    *(small->tile + w * i) |= TILE_WALL;
    *(small->tile + w * i + (w - 1)) |= TILE_WALL;
   }

 w--;
 h--;

 /* Puts the inner walls in a
  * chessboard-like manner
 */
 for(i = 1; i < h; i++)
   for(j = 1; j < w; j++)
     {
      current = 0;

      switch(i % 2)
        {
         case 0:
           {
            current |= TILE_WALL;

            if(j % 2)
              current |= TILE_DESTROYABLE_WALL;

           } break;

         case 1:
           {
            if(!(j % 2))
              current |= TILE_WALL | TILE_DESTROYABLE_WALL;
           } break;
        }

      *(small->tile + (i * (w + 1)) + j) = current;
     }
}



static void destroy_walls(MAP *small, float probability)
{
 unsigned short int i, j, w, h;

 w = small->w;
 h = small->h;

 for(i = 0; i < h; i++)
   for(j = 0; j < w; j++)
     if(small->tile[i * w + j] & TILE_DESTROYABLE_WALL)
       if(genrand_real1() < probability) //if(rand()%101 < probability)
         small->tile[i * w + j] &= TILE_REMOVE_WALL;
}



static unsigned char *search_non_marked_tiles(MAP *small)
{
 unsigned char current, *current_ptr;
 unsigned short int i, j, w, h, extracted;
 int offset;

 static char already_run = 0;

 w = small->w;
 h = small->h;

 for(i = 0; i < h; i++)
   for(j = 0; j < w; j++)
     {
      current = small->tile[i * w + j];
      current_ptr = small->tile + i * w + j;

      if(!(current & (TILE_WALL | TILE_MARK)))
        {
         if(!already_run)
           {
            already_run++;
            return current_ptr;
           }

         extracted = 0;

         do
           {
            do
              offset = genrand_int32() % 4; //rand()%4;
            while(BIT_ALREADY_SET(extracted, offset));

            SET_BIT(extracted, offset);

            switch(offset)
              {
               case 0: offset = 1;  break;
               case 1: offset = -1; break;
               case 2: offset = w;  break;
               case 3: offset = -w; break;
              }

            current = small->tile[i * w + j + offset];
            current_ptr = small->tile + i * w + j + offset;
           }
         while(!(current & TILE_WALL) || !(current & TILE_DESTROYABLE_WALL) || !(*(current_ptr + offset) & TILE_MARK));

         *current_ptr &= TILE_REMOVE_WALL;

         extracted = 0;

         do
           {
            do
              offset = genrand_int32() % 4; //rand()%4;
            while(BIT_ALREADY_SET(extracted, offset) && !ALL_BITS_ARE_SET(extracted));

            if(ALL_BITS_ARE_SET(extracted))
              return current_ptr;

            SET_BIT(extracted, offset);

            switch(offset)
              {
               case 0: offset = 1;  break;
               case 1: offset = -1; break;
               case 2: offset = w;  break;
               case 3: offset = -w; break;
              }

            current = small->tile[i * w + j + offset];
           }
         while(!(current & TILE_WALL) || !(current & TILE_DESTROYABLE_WALL));

         *(small->tile + i * w + j + offset) &= TILE_REMOVE_WALL;

         return current_ptr;
        }
     }

 return NULL;
}



static void check_corridors(unsigned char *tile, int step, int h_step, int v_step)
{
 if(!step)
   {
    *tile |= TILE_MARK;

    if(!(*(tile - h_step) & TILE_MARK) &&
       !(*(tile - h_step) & TILE_WALL))
      check_corridors(tile - h_step, -h_step, h_step, v_step);

    if(!(*(tile + h_step) & TILE_MARK) &&
       !(*(tile + h_step) & TILE_WALL))
      check_corridors(tile + h_step, h_step, h_step, v_step);

    if(!(*(tile - v_step) & TILE_MARK) &&
       !(*(tile - v_step) & TILE_WALL))
      check_corridors(tile - v_step, -v_step, h_step, v_step);

    if(!(*(tile + v_step) & TILE_MARK) &&
       !(*(tile + v_step) & TILE_WALL))
      check_corridors(tile + v_step, v_step, h_step, v_step);

   }
 else
   do
     {
      *tile |= TILE_MARK;

      if(ABS(step) == h_step)
        {
         if(!(*(tile - v_step) & TILE_MARK) &&
            !(*(tile - v_step) & TILE_WALL))
           check_corridors(tile - v_step, -v_step, h_step, v_step);

         if(!(*(tile + v_step) & TILE_MARK) &&
            !(*(tile + v_step) & TILE_WALL))
           check_corridors(tile + v_step, v_step, h_step, v_step);
        }
      else
        {
         if(!(*(tile - h_step) & TILE_MARK) &&
            !(*(tile - h_step) & TILE_WALL))
           check_corridors(tile - h_step, -h_step, h_step, v_step);

         if(!(*(tile + h_step) & TILE_MARK) &&
            !(*(tile + h_step) & TILE_WALL))
           check_corridors(tile + h_step, h_step, h_step, v_step);
        }

      tile += step;
     }
   while(!(*tile & TILE_MARK) && !(*tile & TILE_WALL));
}



static void clear_marks(MAP *small)
{
 unsigned int i, j, w, h;

 w = small->w;
 h = small->h;

 for(i = 0; i < h; i++)
   for(j = 0; j < w; j++)
     small->tile[i * w + j] &= TILE_REMOVE_MARK;
}



static void validate_small_map(MAP *small)
{
 unsigned char *tile;

 while((tile = search_non_marked_tiles(small)))
   check_corridors(tile, 0, 1, small->w);
   
 clear_marks(small);
}



/* Checks if there is already a special pill in
 * the area (made of 81 tiles) which has the
 * passed tile as centre
 *
 * It helps getting a better ditribution of the
 * special pills on the map
 *
 * Returns 1 if finds a special pill, 0 otherwise
*/
static int special_pills_in_the_area(MAP *small, unsigned char *tile)
{
 int i, j;

 for(i = -4; i <= 4; i++)
   for(j = -4; j <= 4; j++)
     if((tile + i * small->w + j > small->tile) &&
        (*(tile + i * small->w + j) & TILE_SPECIAL_PILL))
       return 1;

 return 0;
}



/* Fills the map with the pills
 *
 * It first puts normal pills on every corridor, calculates
 * how many special pills it has to put using the
 * percentage parameter and then randomly replaces
 * some normal pills with the special ones, relying
 * on special_pills_in_the_area() to ensure that the
 * special pills aren't too close to each other
*/
static void distribute_pills(MAP *small, float percentage,
                             unsigned int *normal_pills_num, unsigned int *special_pills_num)
{
 unsigned int i, j, w, h, norm_pills, spec_pills, pills_to_change;

 w = small->w;
 h = small->h;

 norm_pills = spec_pills = 0;

 for(i = 0; i < h; i++)
   for(j = 0; j < w; j++)
     if(!(small->tile[i * w + j] & TILE_WALL))
       {
       	small->tile[i * w + j] |= TILE_NORMAL_PILL;
       	norm_pills++;
       }

 pills_to_change = norm_pills * percentage;

 while(pills_to_change)
 {
  do
    {
     i = genrand_int32() % h;
     j = genrand_int32() % w;
    }
  while(!(small->tile[i * w + j] & TILE_NORMAL_PILL) || 
        special_pills_in_the_area(small, (small->tile + i * w + j)));

  small->tile[i * w + j] |= TILE_SPECIAL_PILL;
  
  spec_pills++;
  norm_pills--;
  pills_to_change--;
 }
 
 *normal_pills_num  = norm_pills;
 *special_pills_num = spec_pills;
}

/* Translates the small map in a 1:1 representation
 * which is mainly used for the screen drawing
*/
static void compose_big_map(MAP *small, MAP *big)
{
 unsigned short int i, j;
 unsigned char current;

 for(i = 0; i < small->h; i++)
   for(j = 0; j < small->w; j++)
     {
      current = small->tile[i * small->w + j];

      switch(i % 2)
        {
         case 0:
           {
            if(!(j % 2))
              big->tile[i * 2 * big->w + j * 2] = current;
            else
              {
               big->tile[i * 2 * big->w + j * 2 - 1] =
               big->tile[i * 2 * big->w + j * 2 + 1] = current & ZERO_PILLS_BITS;

               big->tile[i * 2 * big->w + j * 2]     = current;
              }
           } break;

         case 1:
           {
            if(!(j % 2))
              {
               big->tile[(i * 2 - 1) * big->w + j * 2] =
               big->tile[(i * 2 + 1) * big->w + j * 2] = current & ZERO_PILLS_BITS;

               big->tile[(i * 2) * big->w + j * 2]     = current;
              }
            else
              {
               big->tile[(i * 2 - 1) * big->w + j * 2 - 1] = 
               big->tile[(i * 2 - 1) * big->w + j * 2]     = 
               big->tile[(i * 2 - 1) * big->w + j * 2 + 1] = 

               big->tile[(i * 2) * big->w + j * 2 - 1]     =
               big->tile[(i * 2) * big->w + j * 2 + 1]     =

               big->tile[(i * 2 + 1) * big->w + j * 2 - 1] =
               big->tile[(i * 2 + 1) * big->w + j * 2]     =
               big->tile[(i * 2 + 1) * big->w + j * 2 + 1] = current & ZERO_PILLS_BITS;
               
               big->tile[(i * 2) * big->w + j * 2]         = current;
              }
           } break;
        }
     }
}


/* Helper function, takes care of calling the above functions
 * in the right order and with the right parameters
 *
 * Returns a value different from 0 if there was a memory error
*/
int create_maps(int screen_w, int screen_h, MAP **small_map, MAP **big_map,
                unsigned int *normal_pills_num, unsigned int *special_pills_num)
{
 MAP *small, *big;
 unsigned int norm_pills, spec_pills;
 float probability;

 small = create_map(screen_w, screen_h, MAP_SMALL);

 if(!small)
   return 1;

 big = create_map(screen_w, screen_h, MAP_BIG);
 
 if(!big)
   {
    destroy_map(small);
    return 1;
   }

 initialize_small_map(small);

 /* Generates a slightly random probability
  * about the number of walls that will be
  * destroyed
  *
  * Just adds some randomness on randomness :)
 */
 do
   {
    probability = genrand_real1();
   }
 while(probability <= 0.10f || probability >= 0.30f);

 destroy_walls(small, probability);
 validate_small_map(small);
 distribute_pills(small, 0.025f, &norm_pills, &spec_pills);
 compose_big_map(small, big);

 *small_map = small;
 *big_map   = big;

 *normal_pills_num  = norm_pills;
 *special_pills_num = spec_pills;

 return 0;
}


/* Helper function, calls destroy_map() to free the
 * memory occupied by each map
 *
 * Provided mainly for simmetry with create_maps()
*/
void destroy_maps(MAP *small, MAP *big)
{
 destroy_map(small);
 destroy_map(big);
}

