#ifndef _VECTOR3D_H
#define _VECTOR3D_H

#include <math.h>

#define VZERO               1e-4
#define PI                  3.1415926535897932384626433832795
#define PIdiv2              1.5707963267948966192313216916395
#define PIdiv180            0.017453292519943295769236907684883
#define _180divPI           57.2957795130823208767981548141052

#ifndef NULL
#define NULL    0
#endif

template<class T> 
class Vector3d
{
public:
    Vector3d(): x( T( 0 ) ), y( T( 0 ) ), z( T( 0 ) ) {}
    Vector3d( const T x, const T y, const T z ): x( x ), y( y ), z( z ) {}
    Vector3d( const T *varray ): x( varray[ 0 ] ), y( varray[ 1 ] ), z( varray[ 2 ] ) {}
    Vector3d( const Vector3d &vector ): x( vector.x ), y( vector.y ), z( vector.z ) {}
//    ~Vector3d() {}

    union {
        T v[ 3 ];
        struct { T x, y, z; };
    };

    Vector3d operator+( const Vector3d &rhs ) const
    {
        return Vector3d( this->x + rhs.x,
                       this->y + rhs.y,
                       this->z + rhs.z );
    }

    Vector3d operator-( const Vector3d &rhs ) const
    {
        return Vector3d( this->x - rhs.x,
                       this->y - rhs.y,
                       this->z - rhs.z );
    }

    Vector3d operator*( const T factor ) const
    {
        return Vector3d( this->x * factor,
                       this->y * factor,
                       this->z * factor );
    }

    Vector3d operator/( const T factor ) const
    {
        return Vector3d( this->x / factor,
                       this->y / factor,
                       this->z / factor );
    }

    Vector3d operator-() const
    {
        return Vector3d( -this->x,
                       -this->y,
                       -this->z );
    }

    Vector3d operator+=( const Vector3d &rhs )
    {
        this->x = this->x + rhs.x;
        this->y = this->y + rhs.y;
        this->z = this->z + rhs.z;

        return *this;
    }

    Vector3d operator-=( const Vector3d &rhs )
    {
        this->x = this->x - rhs.x;
        this->y = this->y - rhs.y;
        this->z = this->z - rhs.z;

        return *this;
    }

    Vector3d operator*=( const T factor )
    {
        this->x = this->x * factor;
        this->y = this->y * factor;
        this->z = this->z * factor;

        return *this;
    }

    Vector3d operator/=( const T factor )
    {
        this->x = this->x / factor;
        this->y = this->y / factor;
        this->z = this->z / factor;

        return *this;
    }

    Vector3d operator=( const T *rhs )
    {
        this->x = rhs[ 0 ];
        this->y = rhs[ 1 ];
        this->z = rhs[ 2 ];

        return *this;
    }

    Vector3d operator=( const Vector3d &rhs )
    {
        this->x = rhs.x;
        this->y = rhs.y;
        this->z = rhs.z;

        return *this;
    }

    bool operator==( const Vector3d &rhs ) const
    {
        return ( this->x == rhs.x &&
                 this->y == rhs.y &&
                 this->z == rhs.z );
    }

    bool operator!=( const Vector3d &rhs ) const
    {
        return !( this->x == rhs.x &&
                  this->y == rhs.y &&
                  this->z == rhs.z );
    }

    T dotProduct( const Vector3d &rhs ) const
    {
        return ( this->x * rhs.x +
                 this->y * rhs.y +
                 this->z * rhs.z );
    }

    Vector3d crossProduct( const Vector3d &rhs ) const
    {
        return Vector3d( ( this->y * rhs.z ) - ( this->z * rhs.y ),
                       ( this->z * rhs.x ) - ( this->x * rhs.z ),
                       ( this->x * rhs.y ) - ( this->y * rhs.x ) );
    }

    bool isZero()
    {
        return !( x || y || z );
    }

    T getLength() const
    {
        return sqrt( this->x * this->x +
                     this->y * this->y +
                     this->z * this->z );
    }

    T getSquaredLength() const
    {
        return ( this->x * this->x +
                 this->y * this->y +
                 this->z * this->z );
    }

    void setLength( T length )
    {
        if ( fabs( length ) < VZERO )
        {
            this->zero();
        }
        else
        {
            *this *= ( length / getLength() );
        }
    }

    Vector3d normalize() const
    {
        Vector3d temp = *this;
        
        temp.setLength( T( 1 ) );
    	
        return temp;
    }

    void zero()
    {
        this->x = T( 0 );
        this->y = T( 0 );
        this->z = T( 0 );
    }

    T angleBetween( Vector3d v )
    {
        return ( this->dotProduct( v ) / ( getLength() * v.getLength() ) );
    }

    void rotateX( T angle )
    {
        T a = angle * PIdiv180;
        T c = cos( a );
        T s = sin( a );
        T ty = c * this->y + s * this->z;

        this->z = c * this->z - s * this->y;
        this->y = ty;
    }

    void rotateY( T angle )
    {
        T a = angle * PIdiv180;
        T c = cos( a );
        T s = sin( a );
        T tx = c * this->x + s * this->z;

        this->z = c * this->z - s * this->x;
        this->x = tx;
    }

    void rotateZ( T angle )
    {
        T a = angle * PIdiv180;
        T c = cos( a );
        T s = sin( a );
        T ty = c * this->y + s * this->x;

        this->x = c * this->x - s * this->y;
        this->y = ty;
    }

    void rotate( T angleX, T angleY, T angleZ )
    {
        this->rotateX( angleX );
        this->rotateY( angleY );
        this->rotateZ( angleZ );
    }
};

typedef Vector3d<double> Vector;

#endif

