#include "astar.h"

Astar::Astar( int **array, int width, int height )
{
    this->stack     = new Astar::Astack;
    this->open      = NULL;
    this->closed    = NULL;
    this->path      = NULL;
    this->isPath    = false;
    this->width     = 0;
    this->height    = 0;
    this->size      = 0;
    this->array     = NULL;

    this->initArray( array, width, height );
}

Astar::~Astar()
{
    this->freeNodes();

    if ( this->stack )
    {
        delete this->stack;
    }

    if ( this->array )
    {
        delete []this->array;
    }
}

void Astar::setWeightFunc( int (*calcWeight)(int col, int row ) )
{
    this->_calcWeight = calcWeight;
}

void Astar::setDistanceFunc( int (*calcDistance)( int sourceCol, int sourceRow, int destCol, int destRow ) )
{
    this->_calcDistance = calcDistance;
}

int Astar::calcWeight( int col, int row )
{
    if ( this->_calcWeight )
    {
        return this->_calcWeight( col, row );
    }

    return 1;
}

int Astar::calcDistance( int sourceCol, int sourceRow, int destCol, int destRow )
{
    if ( this->_calcDistance )
    {
        return this->_calcDistance( sourceCol, sourceRow, destCol, destRow );
    }

    return 0;
}

bool Astar::initArray( int **array, int width, int height )
{
    this->width     = width;
    this->height    = height;
    this->size      = this->height * this->width;

    if ( !( this->array = new int[ this->size ] ) )
    {
        return false;
    }

    for ( int row = 0; row < this->height; row++ )
    {
        for ( int col = 0; col < this->width; col++  )
        {
            this->array[ this->getNumber( col, row ) ] = array[ row ][ col ];
        }
    }

    return true;
}

void Astar::redoarray( int **array )
{
    for ( int row = 0; row < this->height; row++ )
    {
        for ( int col = 0; col < this->width; col++  )
        {
            this->array[ this->getNumber( col, row ) ] = array[ row ][ col ];
        }
    }
}

bool Astar::NewPath( int sourceCol, int sourceRow, int destCol, int destRow )
{
    this->isPath = false;

    if ( sourceRow != destRow &&
         sourceCol != destCol &&
         this->isFree( destCol, destRow ) && 
         this->isFree( sourceCol, sourceRow ) )
    {
        this->isPath = true;
        this->freeNodes();
        this->findPath( sourceCol, sourceRow, destCol, destRow );
    }

    return isPath;
}

bool Astar::haveReachedGoal()
{
    return !( this->isPath  && this->path->parent );
}

int Astar::getNumber( int col, int row )
{
    return ( col + row * this->width );
}

bool Astar::isFree( int col, int row )
{
    return ( this->array[ getNumber( col, row ) ] == 1 );
}

void Astar::freeNodes()
{
    Astar::ANode *node = NULL;

    if ( this->open )
    {
        node = this->open->next;

        while ( node )
        {
            Astar::ANode *oldNode = node;

            node = node->next;

            delete oldNode;
        }
    }

    if ( this->closed )
    {
        node = this->closed->next;

        while ( node )
        {
            Astar::ANode *oldNode = node;

            node = node->next;

            delete oldNode;
        }
    }
}

void Astar::pathNext()
{
    this->path = this->path->parent;
}

int Astar::getCol()
{
    return path->col;
}

int Astar::getRow()
{
    return this->path->row;
}


void Astar::findPath( int sourceCol, int sourceRow, int destCol, int destRow )
{
    Astar::ANode *Node = NULL;
    Astar::ANode *BestNode = NULL;
    int TileNumDest = this->getNumber( sourceCol, sourceRow );

    this->isPath = true;
    this->open   = new ANode;
    this->closed = new ANode;
    Node = new ANode;


    for ( int i = 0; i < childrenCount; i++ )
    {
        Node->child[ i ] = NULL;
    }

    Node->parent    = NULL;
    Node->next      = NULL;
    Node->col       = destCol;
    Node->row       = destRow;

    Node->number    = this->getNumber( destCol, destRow );
    Node->weight    = this->calcWeight( destCol, destRow );
    Node->distance  = this->calcDistance( sourceCol, sourceRow, destCol, destRow );

    Node->cost      = Node->weight + Node->distance;

    this->open->next = Node;
    
    while ( true )
    {
        BestNode = this->getBestNode();

        if ( !BestNode ||
             BestNode->number == TileNumDest )
        {
            break;
        }

        this->genChildren( BestNode, sourceCol, sourceRow );
    }

    this->path = BestNode;
}

Astar::ANode *Astar::getBestNode()
{
    ANode *temp = NULL;

    if ( !this->open->next )
    {
        this->isPath = false;
    }
    else
    {
        temp = this->open->next;
        this->open->next = temp->next;

        temp->next = this->closed->next;
        this->closed->next = temp;
    }

    return temp;
}
void Astar::genChildren( ANode *BestNode, int destCol, int destRow )
{
    int temp[ childrenCount ][ 2 ] =
    {
        {  0, -1 }, // u
        {  0,  1 }, // d
        { -1,  0 }, // l
        {  1,  0 }, // r
        { -1, -1 }, // ul
        { -1,  1 }, // dl
        {  1, -1 }, // ur
        {  1,  1 }  // dr
    };

    int col = 0;
    int row = 0;

    for ( int i = 0; i < childrenCount; i++ )
    {
        if ( this->isFree( col = ( BestNode->col + temp[ i ][ 0 ] ), 
                           row = ( BestNode->row + temp[ i ][ 1 ] ) ) )
        {
            this->genChild( BestNode, col, row, destCol, destRow );
        }
    }
}

void Astar::genChild( ANode *BestNode, int col, int row, int destCol, int destRow )
{
    ANode *old       = NULL;
    ANode *Successor = NULL;   
    int c            = 0;
    int tileNum      = this->getNumber( col, row );
    int weight       = BestNode->weight + this->calcWeight( col, row );
    int distance     = BestNode->distance + this->calcDistance( col, row, destCol, destRow );
    int cost         = weight + distance;

    if ( old = this->getOpen( tileNum ) )
    {
        for( c = 0; c < childrenCount; c++ )
        {
            if( !BestNode->child[ c ] )            
            {
                break;
            }
        }
        BestNode->child[ c ] = old;

        if ( cost < old->cost )
        {
            old->parent     = BestNode;
            old->weight     = weight;
            old->distance   = distance;
            old->cost       = cost;
        }
        else
        {
            if ( cost == old->cost && 
                 distance < old->distance )
            {
                old->parent     = BestNode;
                old->weight     = weight;
                old->distance   = distance;
                old->cost       = cost;
            }
        }
    }
    else 
    {
        if ( old = this->getClosed( tileNum ) )
        {
            for( c = 0; c < childrenCount; c++ )
            {
                if ( !BestNode->child[ c ] )
                {
                    break;
                }
            }
            
            BestNode->child[ c ] = old;

            if ( cost  < old->cost )
            {
                old->parent     = BestNode;
                old->weight     = weight;
                old->distance   = distance;
                old->cost       = cost;
                propDown( old );  
            }
            else
            {
                if ( cost == old->cost && 
                     distance < old->distance )
                {
                    old->parent     = BestNode;
                    old->weight     = weight;
                    old->distance   = distance;
                    old->cost       = cost;
                    propDown( old ); 
                }
            }
        }
        else
        {
            Successor = new ANode;
            if ( Successor )
            {
                Successor->parent   = BestNode;
                Successor->weight   = weight;
                Successor->distance = distance;
                Successor->cost     = cost;
                Successor->col      = col;
                Successor->row      = row;
                Successor->number   = tileNum;

                this->insertNode( Successor );    

                for( c = 0; c < childrenCount; c++ )
                {
                    if ( !BestNode->child[ c ] )
                    {
                        break;
                    }
                }

                BestNode->child[ c ] = Successor;
            }
        }
    }
}

Astar::ANode  *Astar::getOpen( int tileNum )
{
    ANode  *temp = this->open->next;

    while ( temp )
    {
        if ( temp->number == tileNum )
        {
            break;
        }
        else
        {
            temp = temp->next;
        }
    }

    return temp;
}

Astar::ANode *Astar::getClosed( int tileNum )
{
    ANode *temp = this->closed->next;

    while ( temp )
    {
        if ( temp->number == tileNum )
        {
            break;
        }
        else
        {
            temp = temp->next;
        }
    }

    return temp;
}

void Astar::insertNode( ANode *Successor )
{
    if ( !this->open->next )
    {
        this->open->next = Successor;
    }
    else
    {
        ANode *temp1 = NULL;
        ANode *temp2 = NULL;
        int cost = 0;

        cost = Successor->cost;
        temp1 = this->open;
        temp2 = this->open->next;

        while ( temp2 && 
                temp2->cost < cost )
        {
            temp1 = temp2;
            temp2 = temp2->next;
        }

        Successor->next = temp2;
        temp1->next = Successor;
    }
}

void Astar::propDown( ANode *Old )
{
    int c = 0;

    for ( c = 0; c < childrenCount; c++ )
    {
        ANode *child = Old->child[ c ];
        int weight = Old->weight;

        if ( !child )
        {
            break;
        }
        
        if ( weight + 1 < child->weight )
        {
            child->weight = weight + 1;
            child->cost = child->weight + child->distance;
            child->parent = Old;

            this->push( child );         
        }     
    }

    while ( this->stack->next )
    {
        ANode *Father = pop();

        for ( c = 0; c < childrenCount; c++ )
        {
            ANode *child = Father->child[ c ];

            if ( !child )
            {
                break;
            }

            if ( Father->weight + 1 < child->weight )
            {              
                child->weight = Father->weight + 1;
                child->cost = child->weight + child->distance;
                child->parent = Father;

                this->push( child );
            }
        }
    }
}

void Astar::push( ANode *node )
{
    Astack *temp = new Astack;

    if ( temp )
    {
        temp->node = node;
        temp->next = this->stack->next;

        this->stack->next = temp;
    }
}

Astar::ANode *Astar::pop()
{
    Astack *stack = this->stack->next;

    ANode *temp = NULL;
    
    if ( stack )
    {
        temp = stack->node;

        this->stack->next = stack->next;
   
        delete stack;
    }

    return( temp );
}


