/***************************************************************************/
/*!
  \file textwrap.c

  \brief Text-wrapping code

  \author Andrei Ellman

 **************************************************************************/




/****************************************************************************
 Includes
 */


/* Libs, etc. */
#include <allegro.h>
#include <allegro/internal/aintern.h>	/* Used to get the FONT_VTABLE */

#include <string.h>	// strncpy






/****************************************************************************
 Local Types
 */




/****************************************************************************
 Global Prototypes
 */


/****************************************************************************
 Local (Static) Prototypes
 */


/****************************************************************************
 Local Defines
 */

#define AEMULTILINEBUFSIZE 512 


/****************************************************************************
 Local Macros
 */


/****************************************************************************
 Global Variables (across program)
 */



/****************************************************************************/
/*!

 \brief Given a string, font and a line-length-limit (in pixels),
  returns a pointer to the character following the last character
  in the string that fully fits on the line.

  The first line of the passed in string is extracted. Lines are usually separated at the last
  whole word before the last whitespace (or start of line if only one whole word is on the line),
  or just before any newlines. If a word is longer than a line, the separation takes place
  after the last whole character to fit on the line.
  
  This is useful for disecting long strings of text in a proportional font into multiple lines.

  \note If we have a word that is longer than a line, we will continue it on the next line.

  \note The character at the returned string-position is either a newline, space, NULL, or the next letter of a word that is longer than a line.

  \note In the event that a line-break occurs at multiple consecutive spaces, the line is broken at the space that crosses the maximum width.

  \warning If a character in the string is longer than \p nMaxLength, then return value is szStart

  \remarks If \p szStart is empty, it returns \p szStart.

  \sa aeAdvanceToStartOfCurrentLineandGetNextLine


  \param  szStart		Points to the start of the line to disect
  \param  fpFont		The font that the string will be rendered in
  \param  nMaxLength	Maximum length in pixels that a line is allowed to be.
  \param  npLineLengthOut	(OUT) (optional) The length of this line in pixels, from the start of the string until the end of the last character. Passing in NULL means do not pass out the maximum length.

 */

char *
aeGetCharInStringThatIsAfterLastOnLine(char *szStart, AL_CONST AeFont *fpFont, unsigned int nMaxLength, unsigned int *npLineLengthOut)
{
	int nChar;

	unsigned int nLineLength = 0;
	unsigned int nLengthSinceStartOfLastWhitespace = 0;

	char *cpNext, *cpCurrent;

	char *cpSpaceMarker;	/* Last space we found (if equal to szStart, we have a word that is longer than one line) */

	
	cpNext = cpCurrent = szStart;	/* Initialise pointers */
	cpSpaceMarker = szStart;	/* Set here in case word longer than a line */



	while( (nChar = ugetx(&cpNext)) )
	{
		unsigned int nCharLength;
		
		if(uisspace(nChar))
		{
			nLineLength += nLengthSinceStartOfLastWhitespace;
			nLengthSinceStartOfLastWhitespace = 0;
			
			/* Check for newlines */
			if(nChar == '\n' || nChar == '\r')
			{
				break;	/* End this line */
			}

			cpSpaceMarker = cpCurrent;	/* Remember the last space we encountered. */
		}

		
		/* Add the current character length to the line-length */
		nCharLength = fpFont->vtable->char_length(fpFont, nChar);
		nLengthSinceStartOfLastWhitespace += nCharLength;	// TODO: If we ever use special-characters, their length will always be 0


		/* If the line length is over the limit */
		if(nLineLength+nLengthSinceStartOfLastWhitespace > nMaxLength)
		{
			if(cpSpaceMarker != szStart)
			{
				cpCurrent = cpSpaceMarker;
			}
			else
			{
				/* Else, we have a word that is longer than a line - in which case, we will break it at the end of the line and continue it on the next line */
				// If we wanted to have fancy hyphernation rules, we could
				// back-track a couple of characters until it is the right
				// place to insert a hyphen without wrapping past the edge.
				// Then, we would somehow have to tell caller to add a hyphen.
				nLineLength = nLengthSinceStartOfLastWhitespace-nCharLength;
			}

			break;	/* End this line */
		}


		cpCurrent = cpNext;
	}



	ASSERT(nLineLength<=nMaxLength);

	if(npLineLengthOut)
	{
		/* Only concerned with lne length if we asked to retrieve it */
		if(!nChar)
		{
			/* End of string */
			nLineLength += nLengthSinceStartOfLastWhitespace;
		}

		*npLineLengthOut = nLineLength;
	}

	return cpCurrent;	/* Pointer to the first char on the next line (the char that exceeded the line length, ot the newline at the end of the line) (this should be 0x00 if it's the last line ) */
}



/****************************************************************************/
/*!

 \brief Given a string, font and a line-length-limit (in pixels),
  returns a pointer to the character following the last character
  in the string that fully fits on the line, and also advances the start of
  the line by one character if the first character is a non-whitespace character.

  The first line of the passed in string is extracted. Extraction of lines
  takes place in a similar manner to aeGetCharInStringThatIsAfterLastOnLine(),
  but in the event that it finds a word that is longer than a line, it advances
  the position by an extra character, and would then produce a line longer than \p nMaxLength.

  Also, this advances the start of the line past any whitespace it finds at
  the first character of the line (and terminates the line if that first character is a newline).
  
  Also keeps a running count of the maximum line-length by passing in an optional pointer to the maximum line-length so far.

  This is useful for disecting long strings of text in a proportional font into multiple lines.

  This function is meant to be used to iterate through a string of text to break it into multiple lines.

  \attention If the first character is a space or newline, it will be munched.
  This is useful if the function is used iteratively, as it breaks the lines
  before the last whitespace, but if the first character in the string is a whitespace
  or newline, it will be munched.


  \note If we have a word that is longer than a line, we will continue it on the next line.

  \note The character at the returned string-position is either a newline, space, NULL, or the next letter of a word that is longer than a line.

  \note In the event that a line-break occurs at multiple consecutive spaces, the line is broken at the space that crosses the maximum width.

  \warning If a character in the string is longer than \p nMaxLength, then return value is the next character. This would make some lines longer than the limit, but to prevent this, make sure the line limit is at least as wide as the widest character in the font.

  \remarks If \p szStart is empty, it returns \p szStart.

  \sa aeGetCharInStringThatIsAfterLastOnLine

  
  \param  cppLineStart	(IN OUT) The character after the end of the previous line is entered. The start of the line to disect is returned.
  \param  fpFont		The font that the string will be rendered in
  \param  nMaxLength	Maximum length in pixels that a line is allowed to be.
  \param  npLongestLineLength	(IN OUT) (optional) Pass in the longest line-length we have so far, and update it if this line is longer. Passing in NULL means do not keep a running count.

 */
 
char *
aeAdvanceToStartOfCurrentLineandGetNextLine(char **cppLineStart, AL_CONST AeFont *fpFont, unsigned int nMaxLength, unsigned int *npLongestLineLength)
{
	char *cpLineStartNext;
	unsigned int nLineLength;

	if(uisspace(ugetc(*cppLineStart)))
	{
		int nChar;

		/* If first char of line is whitespace, then munch it */
		*cppLineStart += uwidth(*cppLineStart);

		nChar = ugetc(*cppLineStart);
		if(nChar == '\n' || nChar == '\r')
		{
			/* If the character after that is a newline, then end the line right here. */
			return *cppLineStart;
		}
	}


	cpLineStartNext	= aeGetCharInStringThatIsAfterLastOnLine(*cppLineStart, fpFont, nMaxLength, (npLongestLineLength?&nLineLength:NULL));
	if(npLongestLineLength)
	{
		*npLongestLineLength = MAX(*npLongestLineLength,nLineLength);
	}

	if(!*cpLineStartNext)
	{
		return cpLineStartNext;	/* We have reached the terminating \0 */
	}
	else
	{
		if(*cppLineStart==cpLineStartNext)
		{
			/* If the maximum length is so narrow that even a single character exceeds it, then just go ahead and print that one character regardless of ther maxlength. */
			/* To prevent this, either the caller should make sure that nMaxLength is at least the length of the longest char in the string or the bitmap should be clipped (part of the character will be lost) */
			cpLineStartNext += uwidth(cpLineStartNext);
			
			if(npLongestLineLength)
			{
				int nChar;
				unsigned int nCharLength;
				nChar = ugetc(*cppLineStart);
				nCharLength = fpFont->vtable->char_length(fpFont, nChar);
				*npLongestLineLength = MAX(*npLongestLineLength,nCharLength);
			}
		}
	}

	return cpLineStartNext;
}



/****************************************************************************/
/*!

 \brief Given a string, font and a line-length-limit (in pixels),
  returns the number of lines that this string will take up if split into lines.
  
  Also keeps a running count of the maximum line-length by passing in an optional pointer to the maximum line-length so far.

  This is useful for working out the area long strings of text in a proportional font
  will take up if split into into multiple lines.

  \attention If the first character is a space or newline, it will be munched.

  \note If we have a word that is longer than a line, we will continue it on the following line.

  \note In the event that a line-break occurs at multiple consecutive spaces, the line is broken at the space that crosses the maximum width.

  \warning If a character in the string is longer than \p nMaxLength, then this lines will be longer than the limit, but to prevent this, make sure the line limit is at least as wide as the widest character in the font.

  \attention Either set *npLongestLineLength to 0 before calling this function, or pass in a pre-existing longest-length.

  \sa aeTextoutMultiline()

  
  \param  szText	The string of text to measure the area of.
  \param  fpFont		The font that the string will be rendered in.
  \param  nMaxLength	Maximum length in pixels that a line is allowed to be.
  \param  npLongestLineLength	(IN OUT) (optional) Pass in the longest line-length we have so far, and update it if this line is longer. Passing in NULL means do not keep a running count.

 */
 
unsigned int
aeGetNumTextLinesWithLengthLimit(AL_CONST char *szText, AL_CONST AeFont *fpFont, unsigned int nMaxLength, unsigned int *npMaxLineLength)
{
	unsigned int nNumLines = 0;

	char *cpLineStart = (char *) szText;

	
	while(*cpLineStart)
	{
		char *cpLineStartNext;


		cpLineStartNext = aeAdvanceToStartOfCurrentLineandGetNextLine(&cpLineStart, fpFont, nMaxLength, npMaxLineLength);
		if(!*cpLineStart)
		{
			// ?: Is this ever reached?
			// A: Perhaps if string is empty
			break;	/* We have reached the terminating \0 */
		}


		nNumLines++;


		cpLineStart = cpLineStartNext;

	}

	
	return nNumLines;
}



/****************************************************************************/
/*!

 \brief Given a string, font and a line-length-limit (in pixels),
  renders this string on the given bitmap.

  The text-output is justified, except the first line which is left-justified.
  
  Also keeps a running count of the maximum line-length by passing in an optional pointer to the maximum line-length so far.

  \attention If the first character is a space or newline, it will be munched.

  \note If we have a word that is longer than a line, we will continue it on the following line.

  \note In the event that a line-break occurs at multiple consecutive spaces, the line is broken at the space that crosses the maximum width.

  \warning If a character in the string is longer than \p nMaxLength, then this lines will be longer than the limit, but to prevent this, make sure the line limit is at least as wide as the widest character in the font.

  \attention Either set *npLongestLineLength to 0 before calling this function, or pass in a pre-existing longest-length.

  \sa aeGetNumTextLinesWithLengthLimit()

  
  \param  bmp		The bitmap to render the text onto.
  \param  szText	The string of text to render the area of.
  \param  fpFont		The font that the string will be rendered in.
  \param  nXPos	  X position of text output
  \param  nYPos	  Y position of text output
  \param  nColor	Colour to render text in (-1 means render the text transparently)
  \param  nBGColour	Colour to render text-background in (-1 means render the text-background transparently)
  \param  nMaxLength	Maximum length in pixels that a line is allowed to be.
  \param  npLongestLineLength	(IN OUT) (optional) Pass in the longest line-length we have so far, and update it if this line is longer. Passing in NULL means do not keep a running count.

 */

void
aeTextoutMultiline(AeBmp *bmp, AL_CONST char *szText, AL_CONST AeFont *fpFont, int nXPos, int nYPos, int nColor, int nBGColour, unsigned int nMaxLength, unsigned int *npMaxLineLength)
{
	char szBuf[AEMULTILINEBUFSIZE];	/* Warning: Static buffer */

	char *cpLineStart = (char *) szText;


	while(*cpLineStart)
	{
		char *cpLineStartNext;


		cpLineStartNext = aeAdvanceToStartOfCurrentLineandGetNextLine(&cpLineStart, fpFont, nMaxLength, npMaxLineLength);
		if(!*cpLineStart)
		{
			// ?: Is this ever reached?
			// A: Perhaps if string is empty
			break;	/* We have reached the terminating \0 */
		}


		//ustrzncpy(szBuf, sizeof(szBuf), cpLineStart, int n);
		strncpy(szBuf, cpLineStart, MIN(cpLineStartNext-cpLineStart,AEMULTILINEBUFSIZE-1));
		szBuf[cpLineStartNext-cpLineStart]=0;
		
		if((*cpLineStartNext == '\n' || *cpLineStartNext == '\r') || !*cpLineStartNext)
		{
			/* The last line, or line ended by newline. Let's just left-justify it */
			textout_ex(bmp, fpFont, szBuf, nXPos, nYPos, nColor /* change the colour if debugging the justify thingy */, nBGColour);
		}
		else
		{
			textout_justify_ex(bmp, fpFont, szBuf, nXPos, nXPos+nMaxLength, nYPos, nMaxLength/2, nColor, nBGColour);
		}

		nYPos += text_height(fpFont);


		cpLineStart = cpLineStartNext;

	}

}
