/** \file videovtb.c
  * \brief video bitmaps (ie. texture rendering) vtable
  *  Some of these don't work correctly or will be very slow.
  *  The current vtable is in early stages, and will be replaced by a better
  *  (faster) version 'soon'. Don't count on regular Allegro functions
  *  working on the screen or on video bitmaps.
  */

#include <string.h>

#include <allegro.h>

#ifdef ALLEGRO_WINDOWS
#include <winalleg.h>
#endif

#include "alleggl.h"
#include "allglint.h"
#include "glvtable.h"
#include <allegro/internal/aintern.h>
#ifdef ALLEGRO_MACOSX
#include <OpenGL/glu.h>
#else
#include <GL/glu.h>
#endif


static GFX_VTABLE allegro_gl_video_vtable;

/* Counter for video bitmaps. screen = 1 */
static int video_bitmap_count = 2;



void allegro_gl_destroy_video_bitmap(BITMAP *bmp);



static int allegro_gl_make_video_bitmap_helper1(int w, int h, int x, int y,
                                                AGL_VIDEO_BITMAP **pvid) {

	int is_power_of_2 = (!(w & (w - 1)) && !(h & (h - 1)));

	(*pvid) = malloc(sizeof(AGL_VIDEO_BITMAP));
			
	if (!(*pvid))
		return -1;
				
	memset(*pvid, 0, sizeof(AGL_VIDEO_BITMAP));
		
	/* Create associated bitmap */
	(*pvid)->memory_copy = create_bitmap_ex(32, w, h);

	if (!(*pvid)->memory_copy)
		return -1;
				
	/* Fill in some values in the bitmap to make it act as a subbitmap
	 */
	if (allegro_gl_extensions_GL.ARB_texture_non_power_of_two
	 || is_power_of_2) {
		(*pvid)->target = GL_TEXTURE_2D;
	}
	else {
		(*pvid)->target = GL_TEXTURE_RECTANGLE_ARB;
	}
	(*pvid)->width  = w;
	(*pvid)->height = h;
	(*pvid)->x_ofs = x;
	(*pvid)->y_ofs = y;

	/* Make a texture out of it */
	glGenTextures(1, &((*pvid)->tex));
	if (!((*pvid)->tex))
		return -1;

	glBindTexture((*pvid)->target, ((*pvid)->tex));

	glTexImage2D((*pvid)->target, 0, GL_RGBA8, w, h,
	             0, GL_RGBA, GL_UNSIGNED_BYTE, NULL);					

	/* By default, use the Allegro filtering mode - ie: Nearest */
	glTexParameteri((*pvid)->target, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri((*pvid)->target, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

	/* Clamp to edge */
	{	GLenum clamp = GL_CLAMP_TO_EDGE;
		if (!allegro_gl_extensions_GL.SGIS_texture_edge_clamp) {
			clamp = GL_CLAMP;
		}
		glTexParameteri((*pvid)->target, GL_TEXTURE_WRAP_S, clamp);
		glTexParameteri((*pvid)->target, GL_TEXTURE_WRAP_T, clamp);
	}

	return 0;
}



static int allegro_gl_make_video_bitmap_helper0(int w, int h, int x, int y,
                                                AGL_VIDEO_BITMAP **pvid) {
		
	int is_power_of_2 = (!(w & (w - 1)) && !(h & (h - 1)));

	/* If we can use rectangle textures, or if the input is a power-of-2
	 * already and is below the max size, then just create one texture
	 * image.
	 */
	if (allegro_gl_extensions_GL.ARB_texture_rectangle
	 || allegro_gl_extensions_GL.NV_texture_rectangle
	 || allegro_gl_extensions_GL.ARB_texture_non_power_of_two
	 || (is_power_of_2
	  && w <= allegro_gl_info.max_texture_size
	  && h <= allegro_gl_info.max_texture_size)) {

		if (allegro_gl_make_video_bitmap_helper1(w, h, 0, 0, pvid)) {
			return -1;
		}
	}
	else if (is_power_of_2) {
		/* Texture is power-of-2-sized, but is too big to fit. Split it in 4
		 * and try again.
		 */
		int wc, hc;
		w = w / 2;
		h = h / 2;

		/* Even a 1x1 texture didn't work? Bail */
		if (!w && !h) {
			return -1;
		}

		wc = w ? w : 1;
		hc = h ? h : 1;

		/* Top-left corner */
		if (allegro_gl_make_video_bitmap_helper0(wc, hc, 0, 0, pvid)) {
			return -1;
		}
		pvid = &((*pvid)->next);

		/* Top-right corner */
		if (w) {
			if (allegro_gl_make_video_bitmap_helper0(w, hc, w, 0, pvid)) {
				return -1;
			}
			pvid = &((*pvid)->next);
		}
		/* Bottom-left corner */
		if (h) {
			if (allegro_gl_make_video_bitmap_helper0(wc, h, 0, h, pvid)) {
				return -1;
			}
			pvid = &((*pvid)->next);
		}
		/* Bottom-right corner */
		if (w && h) {
			if (allegro_gl_make_video_bitmap_helper0(wc, hc, w, h, pvid)) {
				return -1;
			}
			pvid = &((*pvid)->next);
		}
	}
	else {
		/* Okay, now we have a problem. The texture is non-power-of-2 sized
		 * and the GPU doesn't support those. Fail.
		 */
		return -1;
	}
	return 0;
}



/* Will make a (potentially piece-wise) texture out of the specified bitmap
 * Source -must- be a memory bitmap or memory subbitmap (created by Allegro
 * only).
 *
 * And must be a 32bpp image
 */
static BITMAP *allegro_gl_make_video_bitmap(BITMAP *bmp) {
	
	/* Grab a pointer to the bitmap data to patch */
	AGL_VIDEO_BITMAP **pvid = (AGL_VIDEO_BITMAP**)&bmp->extra;
	
	/* Get old binding */
	GLuint old_bind;
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &old_bind);
	
	/* Convert bitmap to texture */
	if (allegro_gl_make_video_bitmap_helper0(bmp->w, bmp->h, 0, 0, pvid)) {
		goto agl_error;
	}

	/* Restore old binding */
	glBindTexture(GL_TEXTURE_2D, old_bind);
	
	return bmp;
	
agl_error:
	allegro_gl_destroy_video_bitmap(bmp);
	return NULL;
}



/* BITMAP *allegro_gl_create_video_bitmap(int w, int h) */
/** create_video_bitmap() overload.
  * This function will create a video bitmap using texture memory.
  * Video bitmaps do not currently share space with the frame buffer (screen).
  * Video bitmaps may be of any size, but the color depth is fixed at 32bpp no
  * matter what the frame buffer's depth might be.
  * Video bitmaps are lost when switching screen modes.
  */
BITMAP *allegro_gl_create_video_bitmap(int w, int h) {

	BITMAP *bitmap;
	
	bitmap = malloc(sizeof(BITMAP) + sizeof(char *));
	
	if (!bitmap)
		return NULL;

	bitmap->dat = NULL;
	bitmap->w = bitmap->cr = w;
	bitmap->h = bitmap->cb = h;
	bitmap->clip = TRUE;
	bitmap->cl = bitmap->ct = 0;
	bitmap->write_bank = bitmap->read_bank = NULL;
	/* We should keep tracks of allocated bitmaps for the ref counter */
	bitmap->id = BMP_ID_VIDEO | video_bitmap_count;
	bitmap->extra = NULL;
	bitmap->x_ofs = 0;
	bitmap->y_ofs = 0;
	bitmap->seg = _default_ds();
	bitmap->line[0] = NULL;

	if (!allegro_gl_make_video_bitmap(bitmap)) {
		free(bitmap);
		return NULL;
	}
	video_bitmap_count++;
	
	/* XXX <rohannessian> We ought to leave the Allegro values intact
	 * Avoids bad interaction with correct Allegro programs.
	 */
	allegro_gl_video_vtable.color_depth = 32;
	allegro_gl_video_vtable.mask_color = makecol32(255, 0, 255);
	bitmap->vtable = &allegro_gl_video_vtable;

	return bitmap;
}



/* void allegro_gl_destroy_video_bitmap(BITMAP *bmp) */
/** destroy_video_bitmap() overload.
  * Will destroy the video bitmap. You mustn't use the BITMAP pointer after
  * calling this function.
  */
void allegro_gl_destroy_video_bitmap(BITMAP *bmp) {

	AGL_VIDEO_BITMAP *vid = bmp ? bmp->extra : NULL, *next;
	
	if (!bmp)
		return;
	
	while (vid) {
		if (vid->memory_copy)
			destroy_bitmap(vid->memory_copy);
				
		if (vid->tex)
			glDeleteTextures(1, &vid->tex);
								
		next = vid->next;
		free(vid);
		vid = next;
	}
	
	free(bmp);
	
	return;
}



/* static void allegro_gl_video_acquire(struct BITMAP *bmp) */
/** \ingroup glvtable
  * acquire_bitmap(bmp) overload.
  * This doesn't do anything, since OpenGL textures
  * doesn't need locking. You don't need to call this function
  * in your program.
  */
static void allegro_gl_video_acquire(struct BITMAP *bmp) {}



/* static void allegro_gl_video_release(struct BITMAP *bmp) */
/** \ingroup glvtable
  * release_bitmap(bmp) overload.
  * This doesn't do anything, since OpenGL textures
  * doesn't need locking. You don't need to call this function
  * in your program.
  */
static void allegro_gl_video_release(struct BITMAP *bmp) {}



static int allegro_gl_video_getpixel(struct BITMAP *bmp, int x, int y)
{
	int pix = -1;
	AGL_VIDEO_BITMAP *vid;
	AGL_LOG(2, "glvtable.c:allegro_gl_screen_getpixel\n");	
	
	if (is_sub_bitmap(bmp)) {
		x += bmp->x_ofs;
		y += bmp->y_ofs;
	}
	if (x < bmp->cl || x >= bmp->cr || y < bmp->ct || y >= bmp->cb) {
		return -1;
	}

	vid = bmp->extra;
	
	while (vid) {
		if (vid->x_ofs <= x && vid->y_ofs <= y
		 && vid->x_ofs + vid->memory_copy->w > x
		 && vid->y_ofs + vid->memory_copy->h > y) {
			
			pix = getpixel(vid->memory_copy, x - vid->x_ofs, y - vid->y_ofs);
			break;
		}
		vid = vid->next;
	}
	
	if (pix != -1) {
		return makeacol_depth(bitmap_color_depth(screen),
		                      getr_depth(32, pix), getg_depth(32, pix),
		                      getb_depth(32, pix), geta_depth(32, pix));
	}
	
	return -1;
}



static void allegro_gl_video_putpixel(struct BITMAP *bmp, int x, int y,
                                      int color) {
	GLbyte pixel[4];
	AGL_VIDEO_BITMAP *vid;

	if (is_sub_bitmap(bmp)) {
		x += bmp->x_ofs;
		y += bmp->y_ofs;
	}
	if (x < bmp->cl || x >= bmp->cr || y < bmp->ct || y >= bmp->cb) {
		return;
	}

	split_color(color, &pixel[0], &pixel[1], &pixel[2], &pixel[3],
	            bitmap_color_depth(screen));
	vid = bmp->extra;
	
	
	while (vid) {
		if (vid->x_ofs <= x && vid->y_ofs <= y
		 && vid->x_ofs + vid->memory_copy->w > x
		 && vid->y_ofs + vid->memory_copy->h > y) {
			
			putpixel(vid->memory_copy, x, y,
			        makeacol_depth(32, pixel[0], pixel[1], pixel[2], pixel[3]));
			
			glBindTexture(vid->target, vid->tex);
			glTexSubImage2D(vid->target, 0, x - vid->x_ofs,
				y - vid->y_ofs, 1, 1, GL_RGBA, GL_UNSIGNED_BYTE, &pixel);
				
			break;
		}
		vid = vid->next;
	}
	
	return;
}



static void allegro_gl_video_vline(BITMAP *bmp, int x, int y1, int y2,
                                   int color) {

	GLubyte pixel[4];
	AGL_VIDEO_BITMAP *vid;
	GLint saved_row_length;
	
	AGL_LOG(2, "glvtable.c:allegro_gl_video_vline\n");
	split_color(color, &pixel[0], &pixel[1], &pixel[2], &pixel[3],
	            bitmap_color_depth(screen));
	vid = bmp->extra;
	
	glGetIntegerv(GL_UNPACK_ROW_LENGTH, &saved_row_length);
	
	if (is_sub_bitmap(bmp)) {
		x  += bmp->x_ofs;
		y1 += bmp->y_ofs;
		y2 += bmp->y_ofs;
	}
	if (x < bmp->cl || x >= bmp->cr) {
		return;
	}
	
	if (y1 > y2) {
		int temp = y1;
		y1 = y2;
		y2 = temp;
	}

	if (y1 < bmp->ct) {
		y1 = bmp->ct;
	}
	if (y2 >= bmp->cb) {
		y2 = bmp->cb - 1;
	}
	
	while (vid) {
		BITMAP *vbmp = vid->memory_copy;
		
		int _y1, _y2, _x;
		if (vid->x_ofs > x || vid->y_ofs > y2
		 || vid->x_ofs + vbmp->w <= x
		 || vid->y_ofs + vbmp->h <= y1) {
			
			vid = vid->next;
			continue;
		}
		
		_y1 = MAX(y1, vid->y_ofs) - vid->y_ofs;
		_y2 = MIN(y2, vid->y_ofs + vbmp->h - 1)	- vid->y_ofs;
		_x = x - vid->x_ofs;

		vline(vbmp, _x, _y1, _y2,
		      makeacol_depth(32, pixel[0], pixel[1], pixel[2], pixel[3]));
		
		glPixelStorei(GL_UNPACK_ROW_LENGTH,
		              vbmp->h > 1
		             ? (vbmp->line[1] - vbmp->line[0]) / 4
		             : vbmp->w);

		glBindTexture(vid->target, vid->tex);
		glTexSubImage2D(vid->target, 0, _x, _y1, 1, _y2 - _y1 + 1, GL_RGBA,
			GL_UNSIGNED_BYTE, vbmp->line[_y1] + _x * 4);

		vid = vid->next;
	}
	
	glPixelStorei(GL_UNPACK_ROW_LENGTH,	saved_row_length);
	
	return;
}



static void allegro_gl_video_hline(BITMAP *bmp, int x1, int y, int x2,
                                   int color) {

	GLubyte pixel[4];
	AGL_VIDEO_BITMAP *vid;
	
	AGL_LOG(2, "glvtable.c:allegro_gl_video_hline\n");
	split_color(color, &pixel[0], &pixel[1], &pixel[2], &pixel[3],
	            bitmap_color_depth(screen));
	vid = bmp->extra;
	
	if (is_sub_bitmap(bmp)) {
		x1 += bmp->x_ofs;
		x2 += bmp->x_ofs;
		y  += bmp->y_ofs;
	}

	if (y < bmp->ct || y >= bmp->cb) {
		return;
	}
	
	if (x1 > x2) {
		int temp = x1;
		x1 = x2;
		x2 = temp;
	}

	if (x1 < bmp->cl) {
		x1 = bmp->cl;
	}
	if (x2 >= bmp->cr) {
		x2 = bmp->cr - 1;
	}
	
	while (vid) {
		BITMAP *vbmp = vid->memory_copy;
		
		int _x1, _x2, _y;
		if (vid->y_ofs > y || vid->x_ofs > x2
		 || vid->x_ofs + vbmp->w <= x1
		 || vid->y_ofs + vbmp->h <= y) {
			
			vid = vid->next;
			continue;
		}
		
		_x1 = MAX(x1, vid->x_ofs) - vid->x_ofs;
		_x2 = MIN(x2, vid->x_ofs + vbmp->w - 1)	- vid->x_ofs;
		_y = y - vid->y_ofs;

		hline(vbmp, _x1, _y, _x2,
			makeacol_depth(32, pixel[0], pixel[1], pixel[2], pixel[3]));
		
		glPixelStorei(GL_UNPACK_ROW_LENGTH,
		              vbmp->h > 1
		             ? (vbmp->line[1] - vbmp->line[0]) / 4
		             : vbmp->w);
		
		glBindTexture(vid->target, vid->tex);
		glTexSubImage2D(vid->target, 0, _x1, _y, _x2 - _x1 + 1, 1, GL_RGBA,
			GL_UNSIGNED_BYTE, vbmp->line[_y] + _x1 * 4);

		vid = vid->next;
	}
	
	return;
}



static void allegro_gl_video_line(struct BITMAP *bmp, int x1, int y1, int x2,
                                  int y2, int color) {
	
	/* Note: very very slow */						
	do_line(bmp, x1, y1, x2, y2, color, allegro_gl_video_putpixel);
	
	return;
}
	


static void allegro_gl_video_rectfill(struct BITMAP *bmp, int x1, int y1,
                                      int x2, int y2, int color) {

	GLubyte pixel[4];
	AGL_VIDEO_BITMAP *vid;
	GLint saved_row_length;
	
	AGL_LOG(2, "glvtable.c:allegro_gl_video_rectfill\n");
	split_color(color, &pixel[0], &pixel[1], &pixel[2], &pixel[3],
	            bitmap_color_depth(screen));
	vid = bmp->extra;
	
	glGetIntegerv(GL_UNPACK_ROW_LENGTH, &saved_row_length);
	
	if (is_sub_bitmap(bmp)) {
		x1 += bmp->x_ofs;
		x2 += bmp->x_ofs;
		y1 += bmp->y_ofs;
		y2 += bmp->y_ofs;
	}
	
	if (y1 > y2) {
		int temp = y1;
		y1 = y2;
		y2 = temp;
	}

	if (x1 > x2) {
		int temp = x1;
		x1 = x2;
		x2 = temp;
	}

	if (x1 < bmp->cl) {
		x1 = bmp->cl;
	}
	if (x2 >= bmp->cr) {
		x2 = bmp->cr - 1;
	}
	if (y1 < bmp->ct) {
		y1 = bmp->ct;
	}
	if (y1 >= bmp->cb) {
		y1 = bmp->cb;
	}
	
	while (vid) {
		BITMAP *vbmp = vid->memory_copy;
		
		int _y1, _y2, _x1, _x2;
		if (vid->x_ofs > x2 || vid->y_ofs > y2
		 || vid->x_ofs + vbmp->w <= x1
		 || vid->y_ofs + vbmp->h <= y1) {
			
			vid = vid->next;
			continue;
		}
		
		_y1 = MAX(y1, vid->y_ofs) - vid->y_ofs;
		_y2 = MIN(y2, vid->y_ofs + vbmp->h - 1)	- vid->y_ofs;
		_x1 = MAX(x1, vid->x_ofs) - vid->x_ofs;
		_x2 = MIN(x2, vid->x_ofs + vbmp->w - 1)	- vid->x_ofs;

		rectfill(vbmp, _x1, _y1, _x2, _y2,
			makeacol_depth(32, pixel[0], pixel[1], pixel[2], pixel[3]));
		
		glPixelStorei(GL_UNPACK_ROW_LENGTH,
		              vbmp->h > 1
		             ? (vbmp->line[1] - vbmp->line[0]) / 4
		             : vbmp->w);

		glBindTexture(GL_TEXTURE_2D, vid->tex);
		glTexSubImage2D(GL_TEXTURE_2D, 0,
		    _x1, _y1, _x2 - _x1 + 1, _y2 - _y1 + 1, GL_RGBA,
			GL_UNSIGNED_BYTE, vbmp->line[_y1] + _x1 * 4);

		vid = vid->next;
	}
	
	glPixelStorei(GL_UNPACK_ROW_LENGTH,	saved_row_length);
	
	return;
}


#if GET_ALLEGRO_VERSION() >= MAKE_VER(4, 1, 17)
static void allegro_gl_video_triangle(struct BITMAP *bmp, int x1, int y1,
                                      int x2, int y2, int x3, int y3, int color)
#else
static int allegro_gl_video_triangle(struct BITMAP *bmp, int x1, int y1,
                                     int x2, int y2, int x3, int y3, int color)
#endif
{	
	GLubyte pixel[4];
	AGL_VIDEO_BITMAP *vid;
	GLint saved_row_length;
	int min_y, max_y, min_x, max_x;
	
	AGL_LOG(2, "glvtable.c:allegro_gl_video_triangle\n");
	split_color(color, &pixel[0], &pixel[1], &pixel[2], &pixel[3],
	            bitmap_color_depth(screen));
	vid = bmp->extra;
	
	glGetIntegerv(GL_UNPACK_ROW_LENGTH, &saved_row_length);

	if (is_sub_bitmap(bmp)) {
		x1 += bmp->x_ofs;
		x2 += bmp->x_ofs;
		x3 += bmp->x_ofs;
		y1 += bmp->y_ofs;
		y2 += bmp->y_ofs;
		y3 += bmp->y_ofs;
	}

	min_y = MIN(y1, MIN(y2, y3));
	min_x = MIN(x1, MIN(x2, x3));
	max_y = MAX(y1, MAX(y2, y3));
	max_x = MAX(x1, MAX(x2, x3));
	
	
	while (vid) {
		BITMAP *vbmp = vid->memory_copy;
		
		int _y1, _y2, _x1, _x2, _x3, _y3;
		if (vid->x_ofs > max_x || vid->y_ofs > max_y
		 || vid->x_ofs + vbmp->w <= min_x
		 || vid->y_ofs + vbmp->h <= min_y) {
			
			vid = vid->next;
			continue;
		}
		
		_y1 = y1 - vid->y_ofs;
		_y2 = y2 - vid->y_ofs;
		_y3 = y3 - vid->y_ofs;
		_x1 = x1 - vid->x_ofs;
		_x2 = x2 - vid->x_ofs;
		_x3 = x3 - vid->x_ofs;

		set_clip(vbmp, bmp->cl - vid->x_ofs, bmp->cr - vid->x_ofs - 1,
		               bmp->ct - vid->y_ofs, bmp->cb - vid->y_ofs - 1);

		triangle(vbmp, _x1, _y1, _x2, _y2, _x3, _y3,
			makeacol_depth(32, pixel[0], pixel[1], pixel[2], pixel[3]));

		set_clip(vbmp, 0, 0, vbmp->w - 1, vbmp->h - 1);
		
		glPixelStorei(GL_UNPACK_ROW_LENGTH,
		              vbmp->h > 1
		             ? (vbmp->line[1] - vbmp->line[0]) / 4
		             : vbmp->w);
		
		/* Not quite the minimal rectangle occupied by the triangle, but
		 * pretty close
		 */
		_y1 = MAX(0, min_y - vid->y_ofs);
		_y2 = MIN(vbmp->h, max_y - vid->y_ofs);
		_x1 = MAX(0, min_x - vid->x_ofs);
		_x2 = MIN(vbmp->w, max_x - vid->x_ofs);
			
		glBindTexture(GL_TEXTURE_2D, vid->tex);
		glTexSubImage2D(GL_TEXTURE_2D, 0,
		    _x1, _y1, _x2 - _x1 + 1, _y2 - _y1 + 1, GL_RGBA,
			GL_UNSIGNED_BYTE, vbmp->line[_y1] + _x1 * 4);

		vid = vid->next;
	}
	
	glPixelStorei(GL_UNPACK_ROW_LENGTH,	saved_row_length);

#if GET_ALLEGRO_VERSION() < MAKE_VER(4, 1, 17)
	return 1;
#endif	
}



void allegro_gl_video_blit_from_memory(struct BITMAP *source,
                struct BITMAP *dest, int source_x, int source_y,
				int dest_x, int dest_y, int width, int height) {

	AGL_VIDEO_BITMAP *vid;
	BITMAP *dest_parent = dest;
	GLint saved_row_length;
	
	AGL_LOG(2, "glvtable.c:allegro_gl_video_blit_from_memory\n");

	if (is_sub_bitmap (dest)) {
	   dest_x += dest->x_ofs;
	   dest_y += dest->y_ofs;
	   while (dest_parent->id & BMP_ID_SUB)
	      dest_parent = (BITMAP *)dest_parent->extra;
	}

	if (dest_x < dest->cl) {
		dest_x = dest->cl;
	}
	if (dest_y < dest->ct) {
		dest_y = dest->ct;
	}
	if (dest_x + width >= dest->cr) {
		width = dest->cr - dest_x;
	}
	if (dest_y + height >= dest->cb) {
		height = dest->cb - dest_y;
	}
	if (width < 1 || height < 1) {
		return;
	}
	
	vid = dest_parent->extra;
	
	glGetIntegerv(GL_UNPACK_ROW_LENGTH, &saved_row_length);
	
	while (vid) {
		BITMAP *vbmp = vid->memory_copy;

		int _x, _y, _w, _h;
		if (vid->x_ofs >= dest_x + width || vid->y_ofs >= dest_y + height
		 || vid->x_ofs + vbmp->w <= dest_x
		 || vid->y_ofs + vbmp->h <= dest_y) {
			
			vid = vid->next;
			continue;
		}

		_x = MAX (vid->x_ofs, dest_x) - vid->x_ofs;
		_w = MIN (vid->x_ofs + vbmp->w, dest_x + width)
		   - vid->x_ofs - _x;
		_y = MAX (vid->y_ofs, dest_y) - vid->y_ofs;
		_h = MIN (vid->y_ofs + vbmp->h, dest_y + height)
		   - vid->y_ofs - _y;

		blit(source, vbmp, source_x + vid->x_ofs + _x - dest_x,
		     source_y + vid->y_ofs + _y - dest_y, _x, _y, _w, _h);

		glPixelStorei(GL_UNPACK_ROW_LENGTH,
		              vbmp->h > 1
		             ? (vbmp->line[1] - vbmp->line[0]) / 4
		             : vbmp->w);

		glBindTexture(vid->target, vid->tex);
		glTexSubImage2D(vid->target, 0, _x, _y, _w, _h,
			GL_RGBA, GL_UNSIGNED_BYTE, vbmp->line[_y] + _x * 4);
			
		vid = vid->next;
	}
	
	glPixelStorei(GL_UNPACK_ROW_LENGTH,	saved_row_length);

	return;	
}



void allegro_gl_video_blit_to_memory(struct BITMAP *source, struct BITMAP *dest,
                         int source_x, int source_y, int dest_x, int dest_y,
                         int width, int height) {

	AGL_VIDEO_BITMAP *vid;
	BITMAP *source_parent = source;
	
	AGL_LOG(2, "glvtable.c:allegro_gl_video_blit_to_memory\n");
	
	if (is_sub_bitmap(source)) {
	   source_x += source->x_ofs;
	   source_y += source->y_ofs;
	   while (source_parent->id & BMP_ID_SUB)
	      source_parent = (BITMAP *)source_parent->extra;
	}

	vid = source_parent->extra;
	
	while (vid) {
		BITMAP *vbmp = vid->memory_copy;

		blit(vbmp, dest, source_x - vbmp->x_ofs, source_y - vbmp->y_ofs,
		     dest_x + vbmp->x_ofs, dest_y + vbmp->y_ofs,
		     width - vbmp->x_ofs, height - vbmp->y_ofs);
	
		vid = vid->next;
	}

	return;	
}



static void allegro_gl_video_clear_to_color(BITMAP *bmp, int color) {

	AGL_LOG(2, "glvtable.c:allegro_gl_video_clear_to_color\n");
	
	allegro_gl_video_rectfill(bmp, 0, 0, bmp->w, bmp->h, color);
			
	return;
}



static void dummy_unwrite_bank(void)
{
}



static GFX_VTABLE allegro_gl_video_vtable = {
	0,
	0,
	dummy_unwrite_bank,			//void *unwrite_bank;  /* C function on some machines, asm on i386 */
	NULL,						//AL_METHOD(void, set_clip, (struct BITMAP *bmp));
	allegro_gl_video_acquire,
	allegro_gl_video_release,
	NULL,						//AL_METHOD(struct BITMAP *, create_sub_bitmap, (struct BITMAP *parent, int x, int y, int width, int height));
	allegro_gl_created_sub_bitmap,
	allegro_gl_video_getpixel,
	allegro_gl_video_putpixel,
	allegro_gl_video_vline,
	allegro_gl_video_hline,
	allegro_gl_video_hline,
	allegro_gl_video_line,
#if GET_ALLEGRO_VERSION() >= MAKE_VER(4, 1, 13)
	allegro_gl_video_line,
#endif	
	allegro_gl_video_rectfill,
	allegro_gl_video_triangle,
	NULL,/*allegro_gl_screen_draw_sprite,*/
	NULL,						//AL_METHOD(void, draw_256_sprite, (struct BITMAP *bmp, struct BITMAP *sprite, int x, int y));
	NULL,/*allegro_gl_screen_draw_sprite_v_flip,*/
	NULL,/*allegro_gl_screen_draw_sprite_h_flip,*/
	NULL,/*allegro_gl_screen_draw_sprite_vh_flip,*/
	NULL,						//AL_METHOD(void, draw_trans_sprite, (struct BITMAP *bmp, struct BITMAP *sprite, int x, int y));
	NULL,						//AL_METHOD(void, draw_trans_rgba_sprite, (struct BITMAP *bmp, struct BITMAP *sprite, int x, int y));
	NULL,						//AL_METHOD(void, draw_lit_sprite, (struct BITMAP *bmp, struct BITMAP *sprite, int x, int y, int color));
	NULL,						//AL_METHOD(void, draw_rle_sprite, (struct BITMAP *bmp, struct RLE_SPRITE *sprite, int x, int y));
	NULL,						//AL_METHOD(void, draw_trans_rle_sprite, (struct BITMAP *bmp, struct RLE_SPRITE *sprite, int x, int y));
	NULL,						//AL_METHOD(void, draw_trans_rgba_rle_sprite, (struct BITMAP *bmp, struct RLE_SPRITE *sprite, int x, int y));
	NULL,						//AL_METHOD(void, draw_lit_rle_sprite, (struct BITMAP *bmp, struct RLE_SPRITE *sprite, int x, int y, int color));
	NULL,						//AL_METHOD(void, draw_character, (struct BITMAP *bmp, struct BITMAP *sprite, int x, int y, int color));
	NULL,/*allegro_gl_screen_draw_glyph,*/
	allegro_gl_video_blit_from_memory,
	allegro_gl_video_blit_to_memory,
	NULL,						//AL_METHOD(void, blit_from_system, (struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height));
	NULL,						//AL_METHOD(void, blit_to_system, (struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height));
	allegro_gl_screen_blit_to_self, /* Video bitmaps use same method as screen */
	allegro_gl_screen_blit_to_self, /* ..._forward */
	allegro_gl_screen_blit_to_self, /* ..._backward */
	allegro_gl_memory_blit_between_formats,
	NULL,						//AL_METHOD(void, masked_blit, (struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int dest_x, int dest_y, int width, int height));
	allegro_gl_video_clear_to_color,
	NULL,						//AL_METHOD(void, pivot_scaled_sprite_flip, (struct BITMAP *bmp, struct BITMAP *sprite, fixed x, fixed y, fixed cx, fixed cy, fixed angle, fixed scale, int v_flip));
#if GET_ALLEGRO_VERSION() >= MAKE_VER(4, 1, 17)
	NULL,                       //AL_METHOD(void, do_stretch_blit, (struct BITMAP *source, struct BITMAP *dest, int source_x, int source_y, int source_width, int source_height, int dest_x, int dest_y, int dest_width, int dest_height, int masked));
	NULL,                       //AL_METHOD(void, draw_gouraud_sprite, (struct BITMAP *bmp, struct BITMAP *sprite, int x, int y, int c1, int c2, int c3, int c4));
	NULL,                       //AL_METHOD(void, draw_sprite_end, (void));
	NULL,                       //AL_METHOD(void, blit_end, (void));
	_soft_polygon,              //AL_METHOD(void, polygon, (struct BITMAP *bmp, int vertices, AL_CONST int *points, int color));
	_soft_rect,                 //AL_METHOD(void, rect, (struct BITMAP *bmp, int x1, int y1, int x2, int y2, int color));
	_soft_circle,               //AL_METHOD(void, circle, (struct BITMAP *bmp, int x, int y, int radius, int color));
	_soft_circlefill,           //AL_METHOD(void, circlefill, (struct BITMAP *bmp, int x, int y, int radius, int color));
	_soft_ellipse,              //AL_METHOD(void, ellipse, (struct BITMAP *bmp, int x, int y, int rx, int ry, int color));
	_soft_ellipsefill,          //AL_METHOD(void, ellipsefill, (struct BITMAP *bmp, int x, int y, int rx, int ry, int color));
	_soft_arc,                  //AL_METHOD(void, arc, (struct BITMAP *bmp, int x, int y, fixed ang1, fixed ang2, int r, int color));
	_soft_spline,               //AL_METHOD(void, spline, (struct BITMAP *bmp, AL_CONST int points[8], int color));
	_soft_floodfill,            //AL_METHOD(void, floodfill, (struct BITMAP *bmp, int x, int y, int color));
	_soft_polygon3d,            //AL_METHOD(void, polygon3d, (struct BITMAP *bmp, int type, struct BITMAP *texture, int vc, V3D *vtx[]));
	_soft_polygon3d_f,          //AL_METHOD(void, polygon3d_f, (struct BITMAP *bmp, int type, struct BITMAP *texture, int vc, V3D_f *vtx[]));
	_soft_triangle3d,           //AL_METHOD(void, triangle3d, (struct BITMAP *bmp, int type, struct BITMAP *texture, V3D *v1, V3D *v2, V3D *v3));
	_soft_triangle3d_f,         //AL_METHOD(void, triangle3d_f, (struct BITMAP *bmp, int type, struct BITMAP *texture, V3D_f *v1, V3D_f *v2, V3D_f *v3));
	_soft_quad3d,               //AL_METHOD(void, quad3d, (struct BITMAP *bmp, int type, struct BITMAP *texture, V3D *v1, V3D *v2, V3D *v3, V3D *v4));
	_soft_quad3d_f,             //AL_METHOD(void, quad3d_f, (struct BITMAP *bmp, int type, struct BITMAP *texture, V3D_f *v1, V3D_f *v2, V3D_f *v3, V3D_f *v4));
#else
	NULL,/*allegro_gl_screen_draw_sprite_end,*/
	NULL						//AL_METHOD(void, blit_end, (void));
#endif
};

